/**
 * Project Name:goods-center-biz File Name:RemoteGoodsDirectionalConfigBackendServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.remoteservice.impl Date:2016年6月28日上午11:14:29 Copyright (c) 2016, duiba.com.cn All
 * Rights Reserved.
 */

package cn.com.duiba.goods.center.biz.remoteservice.impl;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.RemoteGoodsDirectionalConfigBackendService;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsDirectionalConfigDto;
import cn.com.duiba.goods.center.biz.entity.GoodsDirectionalConfigEntity;
import cn.com.duiba.goods.center.biz.service.GoodsDirectionalConfigService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * ClassName:RemoteGoodsDirectionalConfigBackendServiceImpl <br/>
 * Date: 2016年6月28日 上午11:14:29 <br/>
 * 
 * @author wubo
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class RemoteGoodsDirectionalConfigBackendServiceImpl implements RemoteGoodsDirectionalConfigBackendService {

    @Autowired
    private GoodsDirectionalConfigService goodsDirectionalConfigService;
    @Autowired
    private RemoteStockBackendService     remoteStockBackendService;

    private static Logger                 log = LoggerFactory.getLogger(RemoteGoodsDirectionalConfigBackendServiceImpl.class);

    @Override
    public DubboResult<GoodsDirectionalConfigDto> find(Long id) {
        GoodsDirectionalConfigDto dto;
        GoodsDirectionalConfigEntity entity;
        if (id != null) {
            entity = this.goodsDirectionalConfigService.find(id);
        } else {
            return DubboResult.failResult("参数不能为null");
        }
        if (entity != null) {
            dto = entityToDto(entity);
        } else {
            return DubboResult.failResult("查询结果为null");
        }

        return DubboResult.successResult(dto);
    }

    @Override
    public DubboResult<List<GoodsDirectionalConfigDto>> findByGoods(GoodsTypeEnum gtype, Long goodsId) {
        try {
            if (gtype == null || goodsId == null) {
                return DubboResult.failResult("参数不能为null");
            }
            List<GoodsDirectionalConfigDto> dtos = new ArrayList<GoodsDirectionalConfigDto>();
            List<GoodsDirectionalConfigEntity> entities = this.goodsDirectionalConfigService.findByGoods(gtype, goodsId);
            if (entities != null) {
                for (GoodsDirectionalConfigEntity entity : entities) {
                    dtos.add(entityToDto(entity));
                }
            }
            return DubboResult.successResult(dtos);
        } catch (Exception e) {
            log.error("库存定向查询失败\n", e);
            return DubboResult.failResult(e.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> add(List<GoodsDirectionalConfigDto> list) {
        try {
            if (list == null || list.size() == 0) {
                return DubboResult.failResult("参数不能为空");
            }
            for (GoodsDirectionalConfigDto dto : list) {
                GoodsDirectionalConfigEntity entity = dtoToEntity(dto);
                
                Long stockId = getStockId(dto.getGoodsId(), dto.getStockChange());
                
                entity = dtoToEntity(dto);
                entity.setStockId(stockId);
                
                this.goodsDirectionalConfigService.add(entity);
            }

            return DubboResult.successResult(true);
        } catch (Exception e) {
            log.error("定向添加异常", e);
            return DubboResult.failResult(e.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> delete(Long id) {
        try {
            if (id == null) {
                return DubboResult.failResult("参数不能为空");
            }
            boolean rs = this.goodsDirectionalConfigService.remove(id);
            if (rs) {
                return DubboResult.successResult(true);
            } else {
                if (this.goodsDirectionalConfigService.find(id) != null) {// 返回结果为失败时，再去查询一下，如果查询没有，则认为删除成功
                    return DubboResult.successResult(true);
                } else {
                    return DubboResult.failResult("删除失败");
                }
            }
        } catch (Exception e) {
            log.error("delete,error,", e);
            return DubboResult.failResult(e.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> update(List<GoodsDirectionalConfigDto> list) {
        try {
            if (list == null || list.size() == 0) {
                return DubboResult.failResult("参数不能为空");
            }
            for (GoodsDirectionalConfigDto dto : list) {
                if (dto.getId() == null || dto.getId() == 0L) {// id为空的跳过
                    continue;
                }
                GoodsDirectionalConfigEntity entity = dtoToEntity(dto);
                Long stockId = this.goodsDirectionalConfigService.find(dto.getId()).getStockId();
                
                if (dto.getStockChange() != null) {
                    if(stockId==null){
                        Long newStockId = getStockId(dto.getGoodsId(),dto.getStockChange());
                        entity.setStockId(newStockId);
                        this.goodsDirectionalConfigService.update(entity.getId(), newStockId, entity.getDayLimit(), entity.getMinPrice());
                    }else{
                        stockChange(stockId, dto.getStockChange());
                        this.goodsDirectionalConfigService.update(entity.getId(), null, entity.getDayLimit(), entity.getMinPrice());
                    }
                }
            }
            return DubboResult.successResult(true);
        } catch (Exception e) {
            log.error("update, error", e);
            return DubboResult.failResult(e.getMessage());
        }
    }

    @Override
    public DubboResult<Boolean> isExist(GoodsTypeEnum gtype, Long goodsId, Long appId) {
        boolean rs;
        try{
            if (gtype == null || goodsId == null || appId == null) {
                return DubboResult.failResult("参数不能为空");
            }
    
            GoodsDirectionalConfigEntity entity = this.goodsDirectionalConfigService.findByGoodsAndApp(gtype, goodsId,
                                                                                                           appId);
    
            if (entity == null) {
                rs = false;
            } else {
                rs = true;
            }
    
            return DubboResult.successResult(rs);
        } catch (Exception e){
            log.error("isExist, error", e);
            return DubboResult.failResult(e.getMessage());
        }
    }

    @Override
    public DubboResult<GoodsDirectionalConfigDto> findByGoodsAndApp(GoodsTypeEnum gtype, Long goodsId, Long appId) {
        GoodsDirectionalConfigDto rs;
        try{
            if (gtype == null || goodsId == null || appId == null) {
                return DubboResult.failResult("参数不能为空");
            }
    
            GoodsDirectionalConfigEntity entity = this.goodsDirectionalConfigService.findByGoodsAndApp(gtype, goodsId,
                                                                                                           appId);
    
            rs = entityToDto(entity);
    
            return DubboResult.successResult(rs);
        } catch (Exception e){
            log.error("findByGoodsAndApp, error", e);
            return DubboResult.failResult(e.getMessage());
        }
    }
    
    // entity 转 dto
    private GoodsDirectionalConfigDto entityToDto(GoodsDirectionalConfigEntity entity) {
        GoodsDirectionalConfigDto dto;
        if (entity == null) {
            dto = null;
        } else {
            dto = new GoodsDirectionalConfigDto();
            long stock = findStock(entity.getStockId());
            long totalStock = findTotalStock(entity.getStockId());

            dto.setId(entity.getId());
            dto.setAppId(entity.getAppId());
            dto.setGoodsId(entity.getGoodsId());
            dto.setGtype(GoodsTypeEnum.getGoodsTypeEnum(entity.getGoodsType()));
            dto.setDayLimit(entity.getDayLimit());
            dto.setMinPrice(entity.getMinPrice());
            dto.setStock(stock);
            dto.setSales(totalStock - stock);
        }
        return dto;
    }

    // dto 转 entity
    private GoodsDirectionalConfigEntity dtoToEntity(GoodsDirectionalConfigDto dto) {
        GoodsDirectionalConfigEntity entity;
        if (dto == null) {
            entity = null;
        } else {
            entity = new GoodsDirectionalConfigEntity();
            entity.setId(dto.getId());
            entity.setGoodsId(dto.getGoodsId());
            entity.setGoodsType(dto.getGtype().getGtype());
            entity.setAppId(dto.getAppId());
            entity.setDayLimit(dto.getDayLimit());
            entity.setMinPrice(dto.getMinPrice());
        }
        return entity;
    }

    // 增减库存
    private boolean stockChange(long stockId, long stockChange) {
        boolean rs;
        if (stockChange > 0) {
            rs = this.remoteStockBackendService.increaseItemStock(stockId, stockChange).getResult();
        } else if (stockChange < 0) {
            rs = this.remoteStockBackendService.decreaseItemStock(stockId, -stockChange).getResult();
        } else {
            rs = true;
        }
        return rs;
    }
    
    private Long getStockId(long gid,long firstStock) throws Exception{
        Long stockId = null;
//        if(firstStock>0){//初始库存为0返回null
        try{
            DubboResult<Long> dr = this.remoteStockBackendService.newStock(gid, firstStock);// 库存中心获取库存id
            if(dr.isSuccess()){
                stockId = dr.getResult();
            }else {
                throw new Exception(dr.getMsg());
            }
        }catch(Exception e){
            log.error("remoteStockBackendService.newStock("+gid+","+firstStock+")",e);
            throw e;
        }
//        }

        return stockId;
    }
    
    private long findStock(Long stockId){//调用出错，stockId为空则返回0
        long stock = 0;
        if(stockId!=null){
            try{
                DubboResult<Long> dr = this.remoteStockBackendService.find(stockId);
                if(dr.isSuccess()){
                    stock = dr.getResult();
                }else{
                    log.error("remoteStockBackendService.find("+stockId+") 返回错误,"+dr.getMsg());
                }
                
            } catch(Exception e){
                log.error("remoteStockBackendService.find("+stockId+") error,",e);
            }
        }
        return stock;
    }
    
    private long findTotalStock(Long stockId){//调用出错，stockId为空则返回0
        long totalstock = 0;
        if(stockId!=null){
            try{
                DubboResult<Long> dr = this.remoteStockBackendService.findTotalStock(stockId);
                if(dr.isSuccess()){
                    totalstock = dr.getResult();
                }else{
                    log.error("remoteStockBackendService.findTotalStock("+stockId+") 返回错误,"+dr.getMsg());
                }
                
            } catch(Exception e){
                log.error("remoteStockBackendService.findTotalStock("+stockId+") error,",e);
            }
        }
        return totalstock;
    }



}
