/**
 * Project Name:goods-center-biz File Name:GoodsBatchServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午4:43:49 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.enums.CouponType;
import cn.com.duiba.goods.center.biz.dao.GoodsBatchDao;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponRetrieveLogEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponRetrieveLogService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.util.AppendUploadUtil;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * ClassName:GoodsBatchServiceImpl <br/>
 * Date: 2016年5月25日 下午4:43:49 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
public class GoodsBatchServiceImpl implements GoodsBatchService {

    private static Logger             log = LoggerFactory.getLogger(GoodsBatchServiceImpl.class);
    @Autowired
    private GoodsBatchDao             goodsBatchDao;
    @Autowired
    private RemoteStockService        remoteStockService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    @Autowired
    private GoodsCouponRetrieveLogService goodsCouponRetrieveLogService;

    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay,Long stockId) {

        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }
        return goodsBatchDao.insert(gtype, gid, CouponType.Normal, startDay, endDay, stockId);
    }

    @Override
    public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        return goodsBatchDao.insert(gtype, gid, CouponType.Link, startDay, endDay, stockId);
    }

    @Override
    public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        return goodsBatchDao.insert(gtype, gid, CouponType.Repeat, startDay, endDay, stockId);
    }

    @Override
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.findNormalBatchs");
            return goodsBatchDao.selectNotDeletedNotExpiredGtypeGid(gtype, gid);
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.findNotDeletedBatchs");
            return goodsBatchDao.selectNotDeletedByGtypeGid(gtype, gid);
        }finally{
            DBTimeProfile.release();
        }
        
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        GoodsBatchEntity e = goodsBatchDao.select(batchId);
        if (e.getImporting()) {
            return false;
        }
        int ret = goodsBatchDao.deleteBatch(gtype, gid, batchId);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }
    /*
    @Override
    public Boolean fillStockId(GoodsTypeEnum gtype, long gid, long goodsBatchId, long stockId) {
        int ret = goodsBatchDao.updateStockId(gtype, gid, goodsBatchId, stockId);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }*/

    @Override
    public GoodsBatchEntity find(Long goodsBatchId) {
        return goodsBatchDao.select(goodsBatchId);
    }

    @Autowired
    private GoodsCouponService goodsCouponService;

    @Override
    public GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid) {
        try {
            DBTimeProfile.enter(getClass().getSimpleName() + ".refreshBatchUsing");

            List<GoodsBatchEntity> batchs = findNormalBatchs(gtype, gid);

            for (GoodsBatchEntity e : batchs) {
                DubboResult<Long> stockRet=remoteStockService.find(e.getStockId());
                if(!stockRet.isSuccess()){
                    log.error("remoteStockService.find fail ,msg="+stockRet.getMsg());
                    continue;
                }
                if(stockRet.getResult()<=0){
                  //标记无库存
                    goodsBatchDao.updateStatusUsed(e.getId());
                }
            }

            // 选下一个批次
            batchs = findNormalBatchs(gtype, gid);
            List<GoodsBatchEntity> backups = new ArrayList<>();
            Date min = null;
            for (GoodsBatchEntity e : batchs) {
                if (e.getStatus() == GoodsBatchEntity.StatusNotUse) {
                    backups.add(e);
                    if (min == null) {
                        min = e.getStartDay();
                    } else {
                        if (min.getTime() > e.getStartDay().getTime()) {
                            min = e.getStartDay();
                        }
                    }
                }
            }

            for (GoodsBatchEntity e : backups) {
                if (e.getStartDay().equals(min)) {
                    // 选中
                    goodsBatchDao.updateStatusUsing(e.getId());
                    return find(e.getId());
                }
            }

            return null;
        } finally {
            DBTimeProfile.release();
        }
    }

    private Long getOrigionTotalBatchStock(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.getOrigionTotalBatchStock");
            GoodsBatchEntity batch = goodsBatchDao.select(goodsBatchId);
            if (batch.getStockId() == null) {
                // 如果没有填写stockId，认为0库存
                return 0L;
            }
            DubboResult<Long> ret = remoteStockBackendService.findTotalStock(batch.getStockId());
            if (ret.isSuccess()) {
                return ret.getResult();
            } else {
                throw new RuntimeGoodsException(ErrorCode.E0203003);
            }

        } finally{
            DBTimeProfile.release();
        }
        
    }

    @Override
    public Long getSumBatchStock(GoodsTypeEnum gtype, long gid) {
        List<GoodsBatchEntity> list = findNormalBatchs(gtype, gid);
        long total = 0L;
        for (GoodsBatchEntity e : list) {
            //已删除，或者已使用的，跳过
            if (e.getStatus() != GoodsBatchEntity.StatusDeleted &&
                    e.getStatus()!=GoodsBatchEntity.StatusUsed) {
                try{
                    DBTimeProfile.enter("GoodsBatchServiceImpl.getSumBatchStock");
                    DubboResult<Long> stockRet=remoteStockService.find(e.getStockId());
                    if(!stockRet.isSuccess()){
                        log.error("remoteStockService.find error,msg="+stockRet.getMsg());
                        total+=0;
                    }else{
                        total+= stockRet.getResult();
                    }
                }finally{
                    DBTimeProfile.release();
                }
                
            }
        }
        return total;
    }

    @Override
    public Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.getTotalAllBatchStock");
            List<GoodsBatchEntity> list = findNotDeletedBatchs(gtype, gid);
            long total = 0L;
            for (GoodsBatchEntity e : list) {
                if (e.getStatus() != GoodsBatchEntity.StatusDeleted) {
                    total += getOrigionTotalBatchStock(gtype, gid, e.getId());
                }
            }

            return total;
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public void markBatchStatusUsed(long batchId) {
        goodsBatchDao.updateStatusUsed(batchId);
    }

    @Override
    public void markBatchStatusUsing(long batchId) {
        goodsBatchDao.updateStatusUsing(batchId);
    }

    @Override
    public void markBatchStatusNotUsed(long batchId) {
        goodsBatchDao.updateStatusNotUsed(batchId);
    }

    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay) {
        // 只允许精确到日，不允许精确到秒
        startDay = new DateTime(startDay).withTimeAtStartOfDay().toDate();
        endDay = new DateTime(endDay).withTimeAtStartOfDay().toDate();

        if (goodsBatchDao.updateValidDate(batchId, startDay, endDay) == 1) {
            return true;
        }
        return false;
    }

    @Override
    public boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType) {
        if (gtype.equals(GoodsTypeEnum.ADVERT)) {
            int row = goodsBatchDao.updateBatchType(gid, dstType);
            if (row >= 1) {
                return true;
            }
            return false;
        }
        return false;
    }

    @Override
    public Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        
        int row = goodsBatchDao.physicalDeleteBatch(gtype, gid, batchId);
        if (row >= 1) {
            return true;
        }
        return false;
    }

	@Override
	public List<GoodsBatchEntity> findGoodsBatchs(GoodsTypeEnum gtype,
			List<Long> gids) {
		return goodsBatchDao.selectGoodsBatchs(gtype,gids);
	}
    private static final long perRetrieveGoodsCoupon = 999;

    private static ExecutorService executorService = Executors.newSingleThreadExecutor();

    private Date date ;
    @Override
    public Long retrieveGoodsBatch(final GoodsTypeEnum gtype, final Long gid, final Long batchId, final Long count) throws Exception {

        //获取分布式锁
        if(!this.getBatchLock(batchId,gid,gtype)){
            throw new Exception("批次已被锁");
        }

        //检查批次状态
        final GoodsBatchEntity batch = this.find(batchId);
        Long rst = null;
        if(batch==null){
            throw new Exception("批次不存在");
        }
        final int oriStatus = batch.getStatus();

        if(GoodsBatchEntity.StatusNotUse!=oriStatus && GoodsBatchEntity.StatusUsing!=oriStatus){
            throw new Exception("[已使用,未使用]的批次才能回收");
        }
        date = new Date();

        try {
            // 冻结批次,更新成已使用
            this.markBatchStatusUsed(batchId);
            rst = this.createRetrieveLog(gtype, gid, batchId, 0, GoodsCouponRetrieveLogEntity.retrieveStatus_ing, "");
            final Long count1 = rst;
            executorService.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        doRetrieveGoodsCoupon(batch,oriStatus,gtype, gid, batchId, count, count1);
                    } catch (Exception e) {
                        log.error("",e);
                    }finally {
                        updateToOriStatus(batchId,oriStatus);
//                        释放分布式锁
                        releaseBatchLock(batchId,gid,gtype);

                    }
                }
            });
        }catch (Exception e){
            this.updateToOriStatus(batchId,oriStatus);
            log.error("回收批次出错:",e);
            throw e;
        }
        return rst;
    }

    @Override
    public boolean getBatchLock(Long goodsBatchId, Long gid, GoodsTypeEnum gtype) {
        //在GoodsBatchRedisService.java中获取锁
        return false;
    }

    @Override
    public void releaseBatchLock(Long goodsBatchId, Long gid, GoodsTypeEnum gtype) {

    }

    private void updateToOriStatus(Long batchId,int oriStatus){
        if(oriStatus==GoodsBatchEntity.StatusNotUse){
            this.markBatchStatusNotUsed(batchId);
        }else if(oriStatus==GoodsBatchEntity.StatusUsing){
            this.markBatchStatusUsing(batchId);
        }
    }
    private Long position = 0L;

    private void doRetrieveGoodsCoupon(GoodsBatchEntity batch ,int oriStatus,GoodsTypeEnum gtype, Long gid, Long batchId, Long count,Long logid) throws Exception {
        int successRetrieveCount = 0;
        position = 0L;
        String filePath = this.getFileName(gid,batchId);
        try {

            DubboResult<Long> dubboResult = remoteStockService.find(batch.getStockId());
            if(dubboResult.isSuccess()){
                Long stockNum = dubboResult.getResult();
                if(stockNum!=null){
                    if(stockNum.longValue()==0){
                        log.error("库存为0");
                        goodsCouponRetrieveLogService.updateStatusFail(logid);
                        throw new Exception("库存为0");
                    }
                    //如果回收数量大于库存数量,则回收数量等于库存数量
                    if(stockNum.longValue()<count.longValue()){
                        count = stockNum;
                    }

                }else{
                    log.error("库存不存在");
                    goodsCouponRetrieveLogService.updateStatusFail(logid);
                    throw new Exception("库存不存在");
                }
            }

            long forCount = caculateForCount(count);
            String key = filePath+".txt";
            appendUploadUtil.setKey(key);

            boolean isLastPage = false;
            List<GoodsCouponEntity> list = null;
            if(forCount==1){
                list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                        0,(int)count.longValue());
                isLastPage = true;
                successRetrieveCount += this.deleteGoodsCoupons(gid,list,logid,isLastPage);
            }else{

                try {
                    for (int i = 0 ;i<forCount;i++){
                        //如果为最后一页
                        if(i==forCount-1){
                            list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                                    0,(int)(count.longValue()-successRetrieveCount));
                            isLastPage = true;
                        }else{
                            list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                                    0,(int)perRetrieveGoodsCoupon);
                        }
                        successRetrieveCount += this.deleteGoodsCoupons(gid,list,logid,isLastPage);
                    }
                }catch (Exception e){
                    log.error("分页回收批次出错", e);
                }
            }

            //上传文件,更新库存,更新回收记录状态
            completeRetrieve(gid,batchId,count,batch,successRetrieveCount,logid,key);
            log.info("成功回收券码总数"+successRetrieveCount+"条");
        }catch (Exception e){
            log.error("回收券码失败。",e);
            throw  e;
        }
    }

    /**
     * @param gtype
     * @param gid
     * @param batchId
     * @param statusUsed
     * @param start
     * @param perRetrieveGoodsCoupon
     * 写入文件,并删除数据
     * @return
     */
    private int deleteGoodsCoupons(Long gid, List<GoodsCouponEntity> list ,Long logid,boolean isLastPage) {
        int rst = 0;

        //把优惠券写入文件 //删除优惠券
        if(!CollectionUtils.isEmpty(list)){
            log.info("分页回收券码数量:"+list.size());
            try {
                List<Long> deleteIds = appendContenToOss(list,isLastPage);
                rst = goodsCouponService.deleteGoodsCouponByIds(gid, deleteIds);
            }catch (Exception e){
                log.error("appendContenToOss, deleteGoodsCouponByIds", e);
                rst = 0;
            }
        }else {
            log.info("没有可回收的券码");
        }
        return rst;
    }

    private void completeRetrieve(Long gid, Long batchId,Long count,
                                  GoodsBatchEntity batch, int successRetrieveCount,
                                  Long logId,String filePath) {

        //更新库次及缓存数据
        remoteStockBackendService.decreaseItemStock(batch.getStockId(), successRetrieveCount);
        updateLogStatus(logId,successRetrieveCount,count,filePath);

    }

    private final static String fileUrlPrefix = "https://yun.duiba.com.cn/";

    private void updateLogStatus(Long logId,int successRetrieveCount,Long count,String fileUrl){
        String url = fileUrlPrefix+fileUrl;
        log.info("update the fileUrl to the tb_goods_coupon_retrieve_log "+url);
        if(successRetrieveCount !=0 && count.longValue()==successRetrieveCount){
            goodsCouponRetrieveLogService.updateStatusSuccess(logId,url,successRetrieveCount);
        }else if(successRetrieveCount !=0 && count.longValue()>successRetrieveCount){
            goodsCouponRetrieveLogService.updateStatusPart(logId,url,successRetrieveCount);
        }
    }








    private long caculateForCount(Long count) {
        long forCount = 0;
        if((count%perRetrieveGoodsCoupon)==0){
            forCount = count.longValue()/perRetrieveGoodsCoupon;
        }else{
            forCount = count.longValue()/perRetrieveGoodsCoupon+1;
        }
        return forCount;
    }

    private BufferedWriter getBufferedWriter(File file) throws IOException {
        BufferedWriter writer;
        writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file, true)));
        return writer;
    }



    private String getFileName(Long gid, Long batchId) {
        return "tuia/manager/coupon-retrieve/"+gid+"_"+batchId+"_"+date.getTime();
    }

    private Long createRetrieveLog(GoodsTypeEnum gtype, Long gid, Long batchId,
                                   int successRetrieveCount, Integer retrieveStatus,String filePath) {
        GoodsCouponRetrieveLogEntity entity = new GoodsCouponRetrieveLogEntity();
        entity.setBatchId(batchId);
        entity.setGid(gid);
        entity.setStatus(retrieveStatus);
        entity.setGoodsType(gtype.getGtype());
        entity.setRecoveryAmount(successRetrieveCount);
        entity.setFileUrl(filePath);
        return goodsCouponRetrieveLogService.create(entity);
    }

    @Autowired
    private AppendUploadUtil appendUploadUtil;


//    private  File writeToTempFile(String content,String key,String path) throws IOException {
//        File file = File.createTempFile(path, ".txt");
//        file.deleteOnExit();
//        Writer writer = new OutputStreamWriter(new FileOutputStream(file));
//        writer.write(content);
//        writer.close();
//        return file;
//    }
    private List<Long> appendContenToOss(List<GoodsCouponEntity> list,boolean isLastPage) throws Exception {
        List<Long> rst = new ArrayList<Long>(1000);
        StringBuilder content = new StringBuilder();
        int i = 0;
        for(GoodsCouponEntity entity:list){
            i++;
            if(!StringUtils.isEmpty(entity.getCode())){
                content.append(entity.getCode());
            }
            if(!StringUtils.isEmpty(entity.getPassword())){
                content.append("\t");
                content.append(entity.getPassword());
            }
            if(!(isLastPage && i==list.size())){
                content.append("\n");
            }
            rst.add(entity.getId());


        }
        position = appendUploadUtil.upload(content.toString(),position);
        return rst;
    }
}
