package cn.com.duiba.goods.center.biz.service.item.impl;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ElasticGiftsItemShowDto;
import cn.com.duiba.goods.center.biz.dao.GoodsLimitRecordDao;
import cn.com.duiba.goods.center.biz.dao.ItemDao;
import cn.com.duiba.goods.center.biz.entity.GoodsLimitRecordEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.item.ItemGoodsService;
import cn.com.duiba.service.domain.dataobject.AppDO;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.service.remoteservice.RemoteAppService;
import cn.com.duiba.service.remoteservice.RemotePreStockService;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Created by gyf .
 * 16/10/17 .
 */
@Service("itemGoodsService")
public class ItemGoodsServiceImpl implements ItemGoodsService {

    @Autowired
    private ItemDao itemDao;

    @Autowired
    private GoodsLimitRecordDao goodsLimitRecordDao;

    @Autowired
    private RemotePreStockService remotePreStockService;

    @Autowired
    private RemoteAppService remoteAppService;

    @Override
    public List<ElasticGiftsItemShowDto> getItemsByElItemIds4Mobile(List<Long> itemIds, Long appId, Long consumerId) throws Exception {
        List<ItemEntity> items = getItemWithoutConsumerWithCache(itemIds, appId);
        itemLimitFilter(items, consumerId);

        List<ElasticGiftsItemShowDto> result = new ArrayList<>();
        for (ItemEntity it : items) {
            result.add(new ElasticGiftsItemShowDto(it.getId(), it.getWhiteImage()));
        }
        return result;
    }

    @Override
    public void checkElasticTakeOrder(Long itemId, Long appId, Long consumerId) throws Exception {
        Preconditions.checkNotNull(itemId, "兑换项参数不全");
        Preconditions.checkNotNull(appId, "app参数不全");
        Preconditions.checkNotNull(consumerId, "用户参数不全");
        ItemEntity item = itemDao.find(itemId);
        if (item == null || item.getDeleted()) {
            throw new Exception("兑换项不存在");
        }
        if (!item.getEnable()) {
            throw new Exception("兑换项未启用");
        }
        ItemDO itemDO = new ItemDO();
        BeanUtils.copyProperties(item, itemDO);
        AppDO app = remoteAppService.find(appId);
        if (app == null) {
            throw new Exception("应用不存在");
        }
        Long stock = remotePreStockService.getPreStock(itemDO, app);
        if (stock == null) {
            stock = Long.valueOf(item.getRemaining());
        }
        if (stock <= 0) {
            throw new Exception("库存不足");
        }
        GoodsLimitRecordEntity limitRecord = goodsLimitRecordDao.selectOne(GoodsTypeEnum.DUIBA.getGtype(), itemId, consumerId);
        if (limitRecord == null) {  // 表示没兑换过
            return;
        }
        if (limitRecord.getCount() == 0) {  // 表示没兑换过
            return;
        }
        throw new Exception("用户兑换超过限制");
    }

    /**
     * 过滤兑换项兑换项限制
     * @param items
     * @param consumerId
     */
    private void itemLimitFilter(List<ItemEntity> items, Long consumerId) {
        if (CollectionUtils.isEmpty(items)) {
            return;
        }
        List<Long> itemIds = new ArrayList<>(items.size());
        for (ItemEntity it : items) {
            itemIds.add(it.getId());
        }
        List<GoodsLimitRecordEntity> limitRecords = goodsLimitRecordDao.selectCounts(GoodsTypeEnum.DUIBA.getGtype(), itemIds, consumerId);
        Map<Long, GoodsLimitRecordEntity> limitMap = new HashMap<>();
        for (GoodsLimitRecordEntity it : limitRecords) {
            limitMap.put(it.getGid(), it);
        }
        Iterator<ItemEntity> iterator = items.iterator();
        while (iterator.hasNext()) {
            ItemEntity entity = iterator.next();
            GoodsLimitRecordEntity limit = limitMap.get(entity.getId());
            if (limit == null) {    // 没兑换过
                continue;
            }
            if (limit.getCount() == 0) {    // 没兑换过
                continue;
            }
            iterator.remove();
        }
    }

    Cache<Long, List<ItemEntity>> itemsCache = CacheBuilder.newBuilder().expireAfterWrite(1, TimeUnit.MINUTES).build();

    /**
     * 缓存获取 items
     * @param ids
     * @param appId
     * @return
     * @throws ExecutionException
     */
    private List<ItemEntity> getItemWithoutConsumerWithCache(final List<Long> ids, Long appId) throws Exception {
        return itemsCache.get(appId, new Callable<List<ItemEntity>>() {

            @Override
            public List<ItemEntity> call() throws Exception {
                return getItemWithoutConsumer(ids);
            }

        });
    }

    /**
     * 获取 items 不通过 consumerid 筛选
     * @param ids
     * @param appId
     * @return
     */
    private List<ItemEntity> getItemWithoutConsumer(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }
        List<ItemEntity> items = itemDao.findAllByIds(ids);
        if (CollectionUtils.isEmpty(items)) {
            return Collections.emptyList();
        }
        List<ItemEntity> result = new ArrayList<>();
        for (ItemEntity it : items) {
            if (!it.getEnable()) {  // 未启用
                continue;
            }
            if (!it.isOpTypeItem(ItemDO.OpTypeActivity)) {   // 非活动专用
                continue;
            }
            if (!ItemDO.TypeCoupon.equals(it.getType())) {  // 非优惠券
                continue;
            }
            if (ItemDO.SubTypeLink != it.getSubType()) {    // 非链接券
                continue;
            }
            result.add(it);
        }
        return result;
    }
}
