package cn.com.duiba.goods.center.biz.service.item.impl;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import net.rubyeye.xmemcached.GetsResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.EverydayLimitDAO;
import cn.com.duiba.goods.center.biz.dao.ItemLimitDao;
import cn.com.duiba.goods.center.biz.entity.EverydayLimitEntity;
import cn.com.duiba.goods.center.biz.service.item.ItemLimitService;
import cn.com.duiba.goods.center.biz.util.DateUtil;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.service.item.domain.dataobject.ItemLimitDO;
import cn.com.duiba.wolf.cache.CacheClient;

@Service("itemLimitService")
public class ItemLimitServiceImpl implements ItemLimitService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ItemLimitServiceImpl.class);

    private static final String ITEM_NAMESPACE    = "itemcache";
    private static final String APPITEM_NAMESPACE = "appitemcache";

    @Autowired
    private EverydayLimitDAO everydayLimitDAO;

    @Autowired
    private ItemLimitDao itemLimitDao;

    @Autowired
    private CacheClient memcachedClient;

    @Override
    public Boolean addQuantityLimitCounter(ItemKeyDto itemKey) {
        if (!isQuantityLimitItem(itemKey)) {
            return null;
        }
        if (!everydayLimitCheck(itemKey)) {
            return false;
        }
        try {
            final String key = getKey(itemKey);
            Integer limit = getLimit(itemKey);
            Integer state = setByCas(key, limit, 1);
            boolean result = state == 0;
            if (!result && state == 2) {
                Integer countNum = countOrder(itemKey);
                memcachedClient.add(key, DateUtil.getToTomorrowSeconds(), countNum);
                state = setByCas(key, limit, 1);
                result = state == 0;
            }
            return result;
        } catch (Exception e) {
            LOGGER.error("限量库存商品加1异常：", e);
        }
        return false;
    }

    @Override
    public Boolean subQuantityLimitCounter(ItemKeyDto itemKey) throws GoodsException {
        if (!isQuantityLimitItem(itemKey)) {
            return null;
        }
        rollbackEverydayLimitCheck(itemKey);
        final String key = getKey(itemKey);
        Integer state = setByCas(key, 0, -1);
        boolean result = state == 0;
        if (!result && state == 2) {
            result = deleteCounterByItemkey(itemKey);
        }
        return result;
    }

    /**
     * 删除
     * @param itemKey
     * @return
     * @throws Exception
     */
    private Boolean deleteCounterByItemkey(ItemKeyDto itemKey) throws GoodsException {
        String key = getKey(itemKey);
        boolean result = false;
        try {
            return memcachedClient.remove(key);
        } catch (Exception e) {
            LOGGER.error("删除限量库存标记异常：", e);
        }
        return result;
    }

    /**
     * 获取商品订单数量
     *
     * @param itemKey
     * @return
     */
    private Integer countOrder(ItemKeyDto itemKey) throws GoodsException {
        return countItemLimit(itemKey);
    }

    private Integer countItemLimit(ItemKeyDto itemKey) throws GoodsException {
        String key = null;
        if (itemKey.isItemMode()) {
            key = ItemLimitDO.getItemKey(itemKey.getItem().getId(), null);
        } else if (itemKey.isDuibaAppItemMode()) {
            key = ItemLimitDO.getItemKey(itemKey.getItem().getId(), itemKey.getAppItem().getId());
        } else if (itemKey.isSelfAppItemMode()) {
            key = ItemLimitDO.getItemKey(null, itemKey.getAppItem().getId());
        }
        return itemLimitDao.countByItemKeyAndGmtCreate(key, DateUtil.getDayDate(new Date()));
    }

    /**
     * 每日限量检测
     */
    private boolean everydayLimitCheck(ItemKeyDto itemKey) {
        Long appItemId = itemKey.getAppItem() == null ? 0 : itemKey.getAppItem().getId();
        Long itemId = itemKey.getItem() == null ? 0 : itemKey.getItem().getId();
        Date day = new Date();
        EverydayLimitEntity everyday = everydayLimitDAO.find(appItemId, itemId, day);
        if (everyday == null) {
            everyday = new EverydayLimitEntity();
            everyday.setAppItemId(appItemId);
            everyday.setItemId(itemId);
            everyday.setSales(0);
            everyday.setDay(day);
            try {
                everydayLimitDAO.insert(everyday);
            } catch (DuplicateKeyException e) {
                LOGGER.error("everydayLimitCheck unique:appItemId:" + appItemId + " itemId:" + itemId + " day:"
                             + new SimpleDateFormat("yyyy-MM-dd").format(day), e);
            }
        }
        int limit = getLimit(itemKey);
        int ret = everydayLimitDAO.updateSales(everyday.getId(), limit);
        return ret >= 1;
    }

    /**
     * 判断商品是否为每日限量商品
     *
     * @param itemKey
     * @return
     */
    private Boolean isQuantityLimitItem(ItemKeyDto itemKey) {
        if (itemKey.isItemMode()) {
            return itemKey.getItem().isOpTypeItem(ItemDO.OpTypeQuantityLimit);
        } else if (itemKey.isDuibaAppItemMode()) {
            if (itemKey.getAppItem().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
                return true;
            } else if (itemKey.getItem().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
                return true;
            }
        } else if (itemKey.isSelfAppItemMode()) {
            return itemKey.getAppItem().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit);
        }
        return false;
    }

    /**
     * 获取限量商品限制数
     *
     * @param itemKey
     * @return
     */
    private Integer getLimit(ItemKeyDto itemKey) {
        Integer limit = null;
        if (itemKey.isItemMode()) {
            limit = itemKey.getItem().getLimitEverydayQuantity();
        } else if (itemKey.isDuibaAppItemMode()) {
            if (itemKey.getItem().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
                limit = itemKey.getItem().getLimitEverydayQuantity();
            } else if (itemKey.getAppItem().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
                limit = itemKey.getAppItem().getLimitEverydayQuantity();
            }
        } else if (itemKey.isSelfAppItemMode()) {
            limit = itemKey.getAppItem().getLimitEverydayQuantity();
        }
        return limit;
    }

    /**
     * 获取商品key避免item和appItem ID重复
     *
     * @param itemKey
     * @return
     */
    private String getKey(ItemKeyDto itemKey) throws GoodsException {
        String key = null;
        if (itemKey.isItemMode()) {
            key = ITEM_NAMESPACE + itemKey.getItem().getId();
        } else if (itemKey.isDuibaAppItemMode()) {
            if (itemKey.getItem().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
                key = ITEM_NAMESPACE + itemKey.getItem().getId();
            } else if (itemKey.getAppItem().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
                key = APPITEM_NAMESPACE + itemKey.getAppItem().getId();
            }
        } else if (itemKey.isSelfAppItemMode()) {
            key = APPITEM_NAMESPACE + itemKey.getAppItem().getId();
        }
        return key;
    }

    private void rollbackEverydayLimitCheck(ItemKeyDto itemKey) {
        Long appItemId = itemKey.getAppItem() == null ? 0 : itemKey.getAppItem().getId();
        Long itemId = itemKey.getItem() == null ? 0 : itemKey.getItem().getId();
        Date day = new Date();
        EverydayLimitEntity everyday = everydayLimitDAO.find(appItemId, itemId, day);
        if (everyday == null) {
            return;
        }
        everydayLimitDAO.rollbackSales(everyday.getId());
    }

    /**
     * 执行 cas 操作
     *
     * @param key
     * @param limit
     * @param type
     * @return
     */
    private Integer setByCas(final String key, final int limit, final int type) {
        Integer result = null;
        boolean loop = true;
        try {
            int loopNum = 0;
            while (loop) {
                if (loopNum >= 10) {
                    loop = false;
                    result = 1;//系统繁忙
                    break;
                }
                GetsResponse<Integer> last = memcachedClient.gets(key);
                if (last == null) {
                    loop = false;
                    result = 2;//缓存对象为空
                    break;
                } else {
                    int nextValue = (Integer) last.getValue();
                    if (type > 0) {
                        nextValue = nextValue + 1;
                        if (nextValue > limit) {
                            loop = false;
                            result = 3;//超出限制
                            break;
                        }
                    } else if (type < 0) {
                        nextValue = nextValue - 1;
                        if (nextValue < limit) {
                            loop = false;
                            result = 3;//超出限制
                            break;
                        }
                    } else {
                        loop = false;
                        result = 4;//参数错误
                        break;
                    }
                    boolean success = memcachedClient.cas(key, DateUtil.getToTomorrowSeconds(),TimeUnit.SECONDS,
                                                                   nextValue, last.getCas());
                    if (success) {
                        loop = false;
                        result = 0; //更新成功
                    }else{
                        loop = true;
                    }
                }
                loopNum++;
            }
        } catch (Exception e) {
            result = 5;
            LOGGER.error("更新缓存异常:", e);
        }
        return result;
    }

}
