/**
 * Project Name:goods-center-biz
 * File Name:RemotePCGGoodsServiceImpl.java
 * Package Name:cn.com.duiba.goods.center.biz.remoteservice.pcg
 * Date:2016年6月27日下午8:40:20
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.remoteservice.pcg;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.constant.PlatformCouponGoodsConstant;
import cn.com.duiba.goods.center.api.remoteservice.front.RemotePCGGoodsService;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.bo.pcg.PCGUtilBO;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.entity.PlatformCouponGoodsEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsDirectionalConfigService;
import cn.com.duiba.goods.center.biz.service.GoodsLimitRecordService;
import cn.com.duiba.goods.center.biz.service.GoodsTagsService;
import cn.com.duiba.goods.center.biz.service.PCGFilterConfigService;
import cn.com.duiba.goods.center.biz.service.PlatformCouponGoodsService;
import cn.com.duiba.goods.center.biz.service.impl.GoodsDirectionalConfigServiceImpl.DayLimitConfig;
import cn.com.duiba.goods.center.biz.service.stock.PcgDirectionalSalesVolumeService;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.service.remoteservice.RemoteConsumerService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;


/**
 * ClassName:RemotePCGGoodsServiceImpl <br/>
 * Date:     2016年6月27日 下午8:40:20 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("remotePCGGoodsService")
public class RemotePCGGoodsServiceImpl implements RemotePCGGoodsService {
    
    @Autowired
    private GoodsBatchService goodsBatchService;
    @Autowired
    private PlatformCouponGoodsService platformCouponGoodsService;
    @Autowired
    private PCGFilterConfigService pcgFilterConfigService;
    @Autowired
    private GoodsLimitRecordService goodsLimitRecordService;
    @Autowired
    private GoodsCouponBO goodsCouponBO;
    @Autowired
    private PCGUtilBO pCGUtilBO;
    @Autowired
    private GoodsDirectionalConfigService goodsDirectionalConfigService;
    @Autowired
    private GoodsTagsService goodsTagsService;
    @Resource
    private RemoteConsumerService remoteConsumerService;
    @Autowired
    private PcgDirectionalSalesVolumeService pcgDirectionalSalesVolumeService;
    
    @Override
    public DubboResult<Boolean> checkStockEnough(long pcgId, long appId, long consumerId) {
        try{
            DBTimeProfile.enter("stock enough check");
            boolean enough=pCGUtilBO.checkStockEnough(pcgId, appId, consumerId);
            return DubboResult.successResult(enough);
        }finally{
            DBTimeProfile.release();
        }
       
    }

    @Override
    public DubboResult<Long> calcultePCGCredits(long pcgId, long appId, long consumerId) {
        try{
            DBTimeProfile.enter("credits calculte");
            Long credits=pCGUtilBO.calcultePCGCredits(pcgId, appId, consumerId);
            return DubboResult.successResult(credits);
        }finally{
            DBTimeProfile.release();
        }
       
    }
    
    @Override
    public DubboResult<Long> calcultePCGCredits(long pcgId, long appId, long consumerId, long appCreditsRate) {
        try{
            DBTimeProfile.enter("credits calculte");
            Long credits=pCGUtilBO.calcultePCGCredits(pcgId, appId, consumerId, appCreditsRate);
            return DubboResult.successResult(credits);
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public DubboResult<Boolean> canTakePlatformCouponGoods(long pcgId, long appId, long consumerId) {
        PlatformCouponGoodsEntity pcg=platformCouponGoodsService.find(pcgId);
        if(pcg==null){
            //商品不存在
            return DubboResult.failResult("商品已下架(1)");
        }
        if(pcg.getDeleted()){
            //商品已删除
            return DubboResult.failResult("商品已下架(2)");
        }
        //校验商品状态
        if(pcg.getPcgStatus()==PlatformCouponGoodsConstant.PcgStatusOff){
            //商品已下架
            return DubboResult.failResult("商品已下架(3)");
        }
        //校验是否被屏蔽
        if(pCGUtilBO.isInBlack(pcg,appId)){
            //商品被屏蔽
            return DubboResult.failResult("商品已下架(4)");
        }
        // 校验券库黑名单
        if (goodsTagsService.containsAppTag(pcgId, appId)) {
            // 商品标签和 app 标签匹配, 被拦截
            return DubboResult.failResult("下单失败");
        }
        // 用户兑换限制校验
        if(pcg.getLimitPerConsumer()!=null){
            
            try{
                DBTimeProfile.enter("user exchange limit check");
                int limit=pcg.getLimitPerConsumer();
                int count=goodsLimitRecordService.findCount(GoodsTypeEnum.PLATFORM.getGtype(), pcgId, consumerId);
                if(count>=limit){
                    //超出用户兑换限制
                    return DubboResult.failResult("用户已兑换");
                }
            }finally{
                DBTimeProfile.release();
            }
        }
        
        //查询是否有库存(总库存，预分配库存，每日限量库存)
        if(!checkStockEnough(pcgId, appId, consumerId).getResult()){
            return DubboResult.failResult("库存不足");
        }
        
        //如果定向App每日限量库存，先加销量，再兑换
        DayLimitConfig config=goodsDirectionalConfigService.getDayLimit(GoodsTypeEnum.PLATFORM, pcgId, appId);
        if(config!=null){
            if(config.getDayLimit()<=0){
                return DubboResult.failResult("库存不足");
            }
            //判断如果销量超过限额，认为库存不足
            Long sales=pcgDirectionalSalesVolumeService.findSaleCountToday(pcgId, config.getConfig().getId());
            if(sales>=config.getDayLimit().intValue()){
                return DubboResult.failResult("库存不足");
            }
        }
        
        return DubboResult.successResult(true);
    }

    @Override
    public DubboResult<Long> takePlatformCouponGoods(long pcgId, long appId, long consumerId, String orderNum) {
        PlatformCouponGoodsEntity pcg=platformCouponGoodsService.find(pcgId);
        //检验是否可下单
        DubboResult<Boolean> canTake=canTakePlatformCouponGoods(pcgId, appId, consumerId);
        if(!canTake.isSuccess() || !canTake.getResult()){
            return DubboResult.failResult(canTake.getMsg());
        }
        
        //如果定向App每日限量库存，先加销量，再兑换
        DayLimitConfig config=goodsDirectionalConfigService.getDayLimit(GoodsTypeEnum.PLATFORM, pcgId, appId);
        if(config!=null){
            if(config.getDayLimit()<=0){
                return DubboResult.failResult("库存不足");
            }
            //判断如果销量超过限额，认为库存不足
            boolean succ=pcgDirectionalSalesVolumeService.addSaleToday(pcgId, config.getConfig().getId(), orderNum, config.getDayLimit());
            if(!succ){
                return DubboResult.failResult("库存不足");
            }
        }

        GoodsCouponEntity coupon=goodsCouponBO.takeCoupon(GoodsTypeEnum.PLATFORM, pcgId, consumerId, orderNum);
        if(coupon==null){
            if(config!=null && config.getDayLimit()>0){
                pcgDirectionalSalesVolumeService.rollbackSale(pcgId, config.getConfig().getId(), orderNum);
            }
            return DubboResult.failResult("兑换失败");
        }
        
        //兑换限制记录
        if(pcg.getLimitPerConsumer()!=null){
            goodsLimitRecordService.addRecord(GoodsTypeEnum.PLATFORM.getGtype(), pcgId, consumerId);
        }
        
        return DubboResult.successResult(coupon.getGoodsCouponId());
    }
    
    @Override
    public void rollbackPCGCoupon(long pcgId,long consumerId,long couponId,String orderNum) {
        PlatformCouponGoodsEntity pcg=platformCouponGoodsService.find(pcgId);
        
        Boolean succ=goodsCouponBO.rollbackCoupon(couponId, orderNum);
        
        if(succ && pcg.getLimitPerConsumer()!=null){
            //如果有兑换限制，也回滚兑换记录次数
            goodsLimitRecordService.rollbackRecord(GoodsTypeEnum.PLATFORM.getGtype(), pcgId, consumerId);
        }
        
        //如果是每日限量的，需要对销量进行回滚
        ConsumerDO consumer=remoteConsumerService.find(consumerId);
        DayLimitConfig config=goodsDirectionalConfigService.getDayLimit(GoodsTypeEnum.PLATFORM, pcgId, consumer.getAppId());
        if(config!=null){
            pcgDirectionalSalesVolumeService.rollbackSale(pcgId, config.getConfig().getId(), orderNum);
        }
        
    }

	@Override
	public DubboResult<Void> rollbackPCGCoupon(long pcgId, long consumerId, long couponId, String orderNum, Long appId) {
		PlatformCouponGoodsEntity pcg = platformCouponGoodsService.find(pcgId);
		Boolean succ = goodsCouponBO.rollbackCoupon(couponId, orderNum);
		if (succ && pcg.getLimitPerConsumer() != null) {
			// 如果有兑换限制，也回滚兑换记录次数
			goodsLimitRecordService.rollbackRecord(GoodsTypeEnum.PLATFORM.getGtype(), pcgId, consumerId);
		}
		// 如果是每日限量的，需要对销量进行回滚
		DayLimitConfig config = goodsDirectionalConfigService.getDayLimit(GoodsTypeEnum.PLATFORM, pcgId, appId);
		if (config != null) {
			pcgDirectionalSalesVolumeService.rollbackSale(pcgId, config.getConfig().getId(), orderNum);
		}
		return DubboResult.successResult(null);
	}

}

