/**
 * Project Name:goods-center-biz File Name:GoodsCouponService4Speed.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午1:07:36 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RedisKeyTool;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName:GoodsCouponService4Speed <br/>
 * Date: 2016年5月25日 下午1:07:36 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
@Service("goodsCouponService4Speed")
public class GoodsCouponService4Speed extends GoodsCouponServiceImpl implements GoodsCouponService, InitializingBean {
	
    private static final int              REDISBATCHSIZE = 10000;
    /** 存放队列，内部将GoodsCouponEntity使用json转换为String */
    @Autowired
    private RedisTemplate<String, String> redisStringTemplate;
    /** redis券队列缓存管理类 */
    private CouponQueueRedisCache         couponQueueRedisCache;
    /** 批次是否售罄的redis缓存管理类 */
    private BatchSaleoutRedisCache        batchSaleoutRedisCache;
    

    @Override
    public void afterPropertiesSet() throws Exception {
        batchSaleoutRedisCache=new BatchSaleoutRedisCache(redisStringTemplate);
        couponQueueRedisCache=new CouponQueueRedisCache(redisStringTemplate,batchSaleoutRedisCache);
        
    }
    
    @Override
    public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
    	boolean ret = super.deleteBatchUnusedCoupons(batch);
    	couponQueueRedisCache.clearCache(batch.getId());
    	return ret;
    }
    
    @Override
    public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
    	boolean ret = super.deleteUnusedCoupon(gtype, gid, couponId, batchId);
    	couponQueueRedisCache.clearCache(batchId);
    	return ret;
    }
    

    @Override
    public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                              String bizNum) {
        try {
            DBTimeProfile.enter("GoodsCouponService4Speed.takeNormalCoupon");

            ConsumerDO c = remoteConsumerService.find(consumerId);

            if (!consumeStocks(gtype, batch, consumerId, bizNum, c)) {
                // 如果减库存失败，返还null
                return null;
            }

            if (!batchSaleoutRedisCache.isSaleOut(batch.getId())) {
                return tryTakeCoupon(gtype, batch, consumerId, 1);
            } else {
                // 该批次已售罄
                throw new RuntimeGoodsException(ErrorCode.E0202005);
            }
        } finally {
            DBTimeProfile.release();
        }
    }
    
	@Override
	public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		Boolean consumeSuccess = false;
		try {
			DBTimeProfile.enter("GoodsCouponService4Speed.takeNormalCoupon");
			consumeSuccess = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSuccess) {
				return null;
			}
			if (!batchSaleoutRedisCache.isSaleOut(batch.getId())) {
				return tryTakeCoupon(gtype, batch, consumerId, 1, bizNum);
			} else {
				throw new RuntimeGoodsException(ErrorCode.E0202005);
			}
		} catch (Exception e) {
			// 如果库存中心扣库存成功返还库存
			if (consumeSuccess) {
				rollbackStocks(gtype, bizNum);
			}
			throw e;
		} finally {
			DBTimeProfile.release();
		}
	}
	
    /**
     * 从redis队列中取数据 执行lock动作 如果lock动作失败，递归最多尝试3次
     *
     * @author xuhengfei
     * @param gtype
     * @param batch
     * @param consumerId
     * @param tryCount
     * @return
     * @since JDK 1.6
     */
    private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, int tryCount) {
        if (tryCount > 3) {
            throw new RuntimeGoodsException(ErrorCode.E0202003);
        }
        try {
        	int counter = tryCount + 1;
            DBTimeProfile.enter("GoodsCouponService4Speed.tryTakeCoupon");

            long goodsBatchId = batch.getId();
            long gid = batch.getGid();

            long size = couponQueueRedisCache.getQueueSize(goodsBatchId);
            if (size > 0) {
                GoodsCouponEntity entity = couponQueueRedisCache.pop(goodsBatchId);
                boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId);
                if (success) {
                    return entity;
                } else {
                    return tryTakeCoupon(gtype, batch, consumerId, counter);
                }
            } else {
                couponQueueRedisCache.tryLoadCouponByBatchId(gtype, gid, goodsBatchId, REDISBATCHSIZE);
                return tryTakeCoupon(gtype, batch, consumerId, counter);
            }
        } finally {
            DBTimeProfile.release();
        }
    }

    /**
     * BatchSaleoutRedisCache
     */
    public static class BatchSaleoutRedisCache {

        private RedisTemplate<String, String> redisTemplate;

        /**
         * BatchSaleoutRedisCache
         * @param redis
         */
        public BatchSaleoutRedisCache(RedisTemplate<String, String> redis) {
            this.redisTemplate = redis;
        }

        /**
         * 是否售罄，如果缓存中没有数据，默认返回还没有售罄
         *
         * @author xuhengfei
         * @param batchId
         * @return
         * @since JDK 1.6
         */
        public boolean isSaleOut(long batchId) {
            String saleoutStr = redisTemplate.opsForValue().get(getSaleOutKey(batchId));
            Boolean saleout = null;
            if (saleoutStr != null) {
                saleout = Boolean.valueOf(saleoutStr);
            }
            if (saleout == null) {
                return false;
            }
            return Boolean.TRUE.equals(saleout);
        }

        /**
         * 设置一个批次是否售罄的缓存值，1分钟后失效删除
         *
         * @author xuhengfei
         * @param batchId
         * @param saleout
         * @since JDK 1.6
         */
        public void setSaleOut(long batchId, boolean saleout) {
            redisTemplate.opsForValue().set(getSaleOutKey(batchId), String.valueOf(saleout));
            redisTemplate.expire(getSaleOutKey(batchId), 1, TimeUnit.MINUTES);
        }

        private String getSaleOutKey(long batchId) {
            return "goods-" + getClass().getSimpleName() + "-saleout-" + batchId;
        }
    }

    /**
     * ClassName: CouponQueueRedisCache <br/>
     * Function: 待发放的优惠劵队列Redis管理类 <br/>
     * date: 2016年6月5日 下午2:28:14 <br/>
     *
     * @author xuhengfei
     * @version GoodsCouponService4Speed
     * @since JDK 1.6
     */
    public class CouponQueueRedisCache {

        private RedisTemplate<String, String> redisTemplate;
        private BatchSaleoutRedisCache        batchSaleoutRedisCache;

        /**
         * CouponQueueRedisCache
         * @param redis
         * @param batchSaleoutRedisCache
         */
        public CouponQueueRedisCache(RedisTemplate<String, String> redis, BatchSaleoutRedisCache batchSaleoutRedisCache) {
            this.redisTemplate = redis;
            this.batchSaleoutRedisCache = batchSaleoutRedisCache;
        }

        /**
         * 查询redis中该批次的队列长度
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        public long getQueueSize(long goodsBatchId) {
            return redisTemplate.opsForSet().size(getBatchCouponsKey(goodsBatchId));
        }

        /**
         * 从redis的队列中取出一条
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
		public GoodsCouponEntity pop(long goodsBatchId) {
			return JSONObject.parseObject(redisTemplate.opsForSet().pop(getBatchCouponsKey(goodsBatchId)), GoodsCouponEntity.class);
		}

        /**
         * 清理缓存
         * @param goodsBatchId
         */
        public void clearCache(Long goodsBatchId){
        	String key=getBatchCouponsKey(goodsBatchId);
        	redisTemplate.delete(key);
        }
        
        /**
         * 尝试加载数据库中的券码列表,并放入redis
         * @param gtype
         * @param gid
         * @param goodsBatchId
         * @param limit
         */
        public void tryLoadCouponByBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId, int limit) {
            // 分布式锁，防止并发
            boolean flag = redisTemplate.opsForValue().setIfAbsent(getBatchLockKey(goodsBatchId), "1");
            if (!flag) {
                throw new RuntimeGoodsException(ErrorCode.E0202004);
            } else {
                try {
                    DBTimeProfile.enter("redis tryLoadCouponByBatchId");
                    // 设置30秒超时失效
                    redisTemplate.expire(getBatchLockKey(goodsBatchId), 30, TimeUnit.SECONDS);
                    List<GoodsCouponEntity> list = GoodsCouponService4Speed.this.loadCouponByBatchId(gtype, gid,
                                                                                                     goodsBatchId,
                                                                                                     limit);
                    if (list.isEmpty()) {
                        // 如果查不出数据，标记已经售罄
                        batchSaleoutRedisCache.setSaleOut(goodsBatchId, true);
                    } else {
                        // 如果查到数据，将数据放入redis，并且标记未售罄
                        String[] array = new String[list.size()];
                        for (int i = 0; i < list.size(); i++) {
                            array[i] = JSONObject.toJSONString(list.get(i));
                        }
                        redisTemplate.opsForSet().add(getBatchCouponsKey(goodsBatchId), array);
                        // 队列数据1小时后失效
                        redisTemplate.expire(getBatchCouponsKey(goodsBatchId), 1, TimeUnit.HOURS);

                        batchSaleoutRedisCache.setSaleOut(goodsBatchId, false);
                    }

                } finally {
                    DBTimeProfile.release();
                    // 解锁
                    redisTemplate.delete(getBatchLockKey(goodsBatchId));
                }
            }
        }

        /**
         * redis 存放队列的key
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        private String getBatchCouponsKey(Long goodsBatchId) {
            return RedisKeyTool.getRedisKey(this.getClass(), "batch_coupons", goodsBatchId + "");
        }

        /**
         * 批次的分布式锁，加锁后从数据库捞取数据
         * 
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        private String getBatchLockKey(Long goodsBatchId) {
            return RedisKeyTool.getRedisKey(this.getClass(), "batch_lock", goodsBatchId + "");
        }

    }

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		Boolean ret = super.deleteBatchUnusedCoupons(gtype, gid, batchId, couponIds);
		couponQueueRedisCache.clearCache(batchId);
		return ret;
	}
	
    /**
     * 从redis队列中取数据 执行lock动作 如果lock动作失败，递归最多尝试3次
     * @param gtype
     * @param batch
     * @param consumerId
     * @param tryCount
     * @param bizNum
     * @return
     */
    private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, int tryCount, String bizNum) {
        if (tryCount > 3) {
            throw new RuntimeGoodsException(ErrorCode.E0202003);
        }
        try {
        	int counter = tryCount + 1;
            DBTimeProfile.enter("GoodsCouponService4Speed.tryTakeCoupon");

            long goodsBatchId = batch.getId();
            long gid = batch.getGid();

            long size = couponQueueRedisCache.getQueueSize(goodsBatchId);
            if (size > 0) {
                GoodsCouponEntity entity = couponQueueRedisCache.pop(goodsBatchId);
                boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId, bizNum);
                if (success) {
                    return entity;
                } else {
                    return tryTakeCoupon(gtype, batch, consumerId, counter, bizNum);
                }
            } else {
                couponQueueRedisCache.tryLoadCouponByBatchId(gtype, gid, goodsBatchId, REDISBATCHSIZE);
                return tryTakeCoupon(gtype, batch, consumerId, counter, bizNum);
            }
        } finally {
            DBTimeProfile.release();
        }
    }

}
