package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.CouponBatchDao;
import cn.com.duiba.goods.center.biz.dao.item.CouponDao;
import cn.com.duiba.goods.center.biz.entity.AppItemEntity;
import cn.com.duiba.goods.center.biz.entity.CouponBatchEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.CouponFlowInnerService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.util.DataSource;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;

/**
 * 优惠券内部流程 service
 */
@Service
public class CouponFlowInnerServiceImpl implements CouponFlowInnerService {

	@Autowired
	private CouponBatchDao couponBatchDao;
	@Autowired
	private CouponDao couponDao;
	@Autowired
	private ItemService itemService;
	@Autowired
	private AppItemService appItemService;

	@Override
	public CouponBatchEntity checkCouponBatch(ItemKeyDto itemKey) {
		CouponBatchEntity couponBatch = this.getItemKeyCurrentCouponBatch(itemKey);
		if (couponBatch == null || couponBatch.getRemaining() <= 0) {
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
		return couponBatch;
	}

	@Override
	public CouponBatchEntity getItemKeyCurrentCouponBatch(ItemKeyDto itemKey) {
		Long batchId = null;
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			batchId = itemKey.getItem().getBatchId();
		} else if (itemKey.isSelfAppItemMode()) {
			batchId = itemKey.getAppItem().getBatchId();
		}
		CouponBatchEntity couponBatch = null;
		if (batchId != null) {
			couponBatch = couponBatchDao.find(batchId);
		}
		return couponBatch;
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void switchBatch(ItemKeyDto itemKeyDto) {
		if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
			this.switchItemBatch(itemKeyDto.getItem().getId());
		} else if (itemKeyDto.isSelfAppItemMode()) {
			this.switchAppItemBatch(itemKeyDto.getAppItem().getId());
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void switchItemBatch(long itemId) {
		ItemEntity item = itemService.find(itemId);
		if (item == null) {
			return;
		}
		if (!ItemDto.TypeCoupon.equals(item.getType())) {
			return;
		}
		CouponBatchEntity nextBatch;
		if (item.getBatchId() != null) {
			CouponBatchEntity currentBatch = couponBatchDao.find(item.getBatchId());
			// 当前批次可用
			if (currentBatch.getRemaining() > 0 && currentBatch.getValidEndDate().after(new Date())) {
				return;
			} else {
				// 刷新批次信息
				this.refreshBatch(currentBatch);
				// 取消商品批次绑定
				couponBatchDao.setItemBatchEmpty(itemId);
				// 查询下一个可用批次（新建，使用中，库存>0）
				nextBatch = couponBatchDao.findItemNextBatch(itemId);
			}
		} else {
			// 查询下一个可用批次（新建，使用中，库存>0）
			nextBatch = couponBatchDao.findItemNextBatch(itemId);
		}
		if (nextBatch != null) {
			Integer remaining = couponBatchDao.countBatchRemaining(nextBatch.getId());
			// 修改下一批次信息
			couponBatchDao.updateActiveAndRemaining(nextBatch.getId(), CouponBatchEntity.ACTIVEUSING, remaining);
			// 绑定批次到商品
			couponBatchDao.setItemBatchId(itemId, nextBatch.getId());
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void switchAppItemBatch(long appItemId) {
		AppItemEntity appItem = appItemService.find(appItemId);
		if (appItem == null) {
			return;
		}
		if (!ItemDto.TypeCoupon.equals(appItem.getType())) {
			return;
		}
		CouponBatchEntity nextBatch;
		if (appItem.getBatchId() != null) {
			CouponBatchEntity currentBatch = couponBatchDao.find(appItem.getBatchId());
			// 当前批次可用
			if (currentBatch.getRemaining() > 0 && currentBatch.getValidEndDate().after(new Date())) {
				return;
			} else {
				// 刷新批次信息
				this.refreshBatch(currentBatch);
				// 取消商品批次绑定
				couponBatchDao.setAppItemBatchEmpty(appItemId);
				// 查询下一个可用批次
				nextBatch = couponBatchDao.findAppItemNextBatch(appItemId);
			}
		} else {
			nextBatch = couponBatchDao.findAppItemNextBatch(appItemId);
		}
		if (nextBatch != null) {
			Integer remaining = couponBatchDao.countBatchRemaining(nextBatch.getId());
			// 修改下一批次信息
			couponBatchDao.updateActiveAndRemaining(nextBatch.getId(), CouponBatchEntity.ACTIVEUSING, remaining);
			// 绑定批次到商品
			couponBatchDao.setAppItemBatchId(appItemId, nextBatch.getId());
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void refreshBatch(CouponBatchEntity batch) {
		Integer active = batch.getActive();
		// 是否过期
		if (batch.getValidEndDate().before(new Date())) {
			active = CouponBatchEntity.ACTIVEOVERDUE;
		}
		Integer remaining = couponBatchDao.countBatchRemaining(batch.getId());
		// 如果返还的批次为已使用，修改为新建，确保下次刷新切换批次时后返的批次可以被选中
		if (CouponBatchEntity.ACTIVEUSED == batch.getActive().intValue() && remaining > 0) {
			active = CouponBatchEntity.ACTIVENEW;
		}
		// 没有库存 又没有券，就应该删除
		if (remaining <= 0) {
			Integer ret = couponDao.ishavingCoupon(batch.getId());
			if (ret == null) {
				// 没有库存又没有券批次状态置为删除
				active = CouponBatchEntity.ACTIVEDELETED;
			} else {
				// 没有库存有券批次状态置为已使用
				active = CouponBatchEntity.ACTIVEUSED;
			}
		}
		// 修改当前批次信息
		couponBatchDao.updateActiveAndRemaining(batch.getId(), active, remaining);
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void refreshItemKeyCoupon(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			this.refreshItemCoupon(itemKey.getItem().getId());
		} else if (itemKey.isSelfAppItemMode()) {
			this.refreshAppItemCoupon(itemKey.getAppItem().getId());
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void refreshItemCoupon(long itemId) {
		ItemEntity item = itemService.find(itemId);
		if (item == null) {
			return;
		}
		if (!ItemDto.TypeCoupon.equals(item.getType())) {
			return;
		}
		CouponBatchEntity batch = couponBatchDao.findItemMaxValidEndDateBatch(itemId);
		Date validEndDate = item.getValidEndDate();
		if (batch != null) {
			validEndDate = batch.getValidEndDate();
			// 如果最大过期时间大于现在把appItem中sub_status过期置为null
			if (batch.getValidEndDate().after(new Date())) {
				appItemService.updateExpiedAppItem(itemId);
			}
		}
		Integer totalRemaining = couponBatchDao.countAllRemaining(null, itemId);
		itemService.updateRemainingAndvalidEndDate(itemId, Long.valueOf(totalRemaining), validEndDate);
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void refreshAppItemCoupon(long appItemId) {
		AppItemEntity appItem = appItemService.find(appItemId);
		if (appItem == null) {
			return;
		}
		if (!ItemDto.TypeCoupon.equals(appItem.getType())) {
			return;
		}
		CouponBatchEntity batch = couponBatchDao.findAppItemMaxValidEndDateBatch(appItemId);
		Date validEndDate = appItem.getValidEndDate();
		if (batch != null) {
			validEndDate = batch.getValidEndDate();
			// 如果最大过期时间大于现在把appItem中sub_status过期置为null
			if (batch.getValidEndDate().after(new Date())) {
				appItemService.updateExpiedAppItemById(appItemId);
			}
		}
		Integer totalRemaining = couponBatchDao.countAllRemaining(appItemId, null);
		appItemService.updateRemainingAndvalidEndDate(appItemId, Long.valueOf(totalRemaining), validEndDate);
	}

}
