package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemStockConsumeDto;
import cn.com.duiba.goods.center.biz.entity.AppItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.entity.ItemStockSpecifyConsumeEntity;
import cn.com.duiba.goods.center.biz.entity.PreStockPointEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyService;
import cn.com.duiba.goods.center.biz.service.item.ItemKeyStockService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 
 * ItemKeyServiceImpl
 *
 */
@Service
public class ItemKeyServiceImpl implements ItemKeyService {

	private static final Logger log = LoggerFactory.getLogger(ItemKeyServiceImpl.class);

	@Autowired
	private AppItemService appItemService;
	@Autowired
	private ItemService itemService;
	@Autowired
	private PreStockService preStockService;
	@Autowired
	private ItemKeyStockService itemKeyStockService;
	@Autowired
	private EverydayLimitService everydayLimitService;

	private static final String DECR_STOCK = "decr_stock";
	private static final String DECR_PRE_STOCK = "decr_pre_stock";
	private static final String DECR_APP_SPECIFY_STOCK = "decr_app_specify_stock";
	private static final String DECR_EVERDAY_STOCK = "decr_everday_stock";

	@Override
	public ItemKeyDto findItemKey(Long appItemId, Long itemId, Long appId) {
		return findItemKey(appItemId, itemId, appId, false);
	}
	
	@Override
	public ItemKeyDto findItemKeyIncludeDeleted(Long appItemId, Long itemId, Long appId) {
		return findItemKey(appItemId, itemId, appId, true);
	}
	
	private ItemKeyDto findItemKey(Long appItemId, Long itemId, Long appId, boolean includeDeleted){
		AppItemEntity appItemEntity = null;
		ItemEntity itemEntity = null;
		if (appItemId != null) {
			appItemEntity = appItemService.find(appItemId);
			if (appItemEntity != null && appItemEntity.getItemId() != null) {
				itemEntity = itemService.find(appItemEntity.getItemId());
			}
		} else if (itemId != null) {
			itemEntity = itemService.find(itemId);
			appItemEntity = appItemService.findByAppIdAndItemId(appId, itemId);
		}
		if (!includeDeleted && appItemEntity != null && appItemEntity.getDeleted()) {
			appItemEntity = null;
		}
		AppItemDto appItemDto = null;
		if (appItemEntity != null) {
			appItemDto = BeanUtils.copy(appItemEntity, AppItemDto.class);
		}
		ItemDto itemDto = null;
		if (itemEntity != null) {
			itemDto = BeanUtils.copy(itemEntity, ItemDto.class);
		}
		return new ItemKeyDto(appItemDto, itemDto, appId);
	}

	@Override
	public Long findStock(ItemKeyDto itemKeyDto) {
		Long stock = 0L;
		if (itemKeyDto.isSelfAppItemMode()) {
			AppItemEntity entity = appItemService.find(itemKeyDto.getAppItem().getId());
			// 商品库存
			if (entity.getRemaining() != null) {
				stock = Long.valueOf(entity.getRemaining());
			}
			// 每日限量库存
			Long everydayStock = everydayLimitService.findEverydayStock(itemKeyDto);
			if (everydayStock != null && everydayStock <= stock) {
				stock = everydayStock;
			}
		} else if (itemKeyDto.isItemMode() || itemKeyDto.isDuibaAppItemMode()) {
			ItemEntity entity = itemService.find(itemKeyDto.getItem().getId());
			// 商品库存
			if (entity.getRemaining() != null) {
				stock = Long.valueOf(entity.getRemaining());
			}
			// 直充类不需要库存商品
			if (itemService.isRechargeGoods(entity)) {
				return 1L;
			}
			// 取定向库存
			if (stock != null && stock > 0 && entity.isOpTypeItem(ItemDO.OpTypeSpecify)) {
				Long specifyStock = itemService.findSpecifyRemaining(entity.getId(), itemKeyDto.getAppId());
				if (specifyStock != null && specifyStock <= stock) {
					stock = specifyStock;
				}
			}
			// 预分配库存
			if (stock != null && stock > 0 && entity.isOpTypeItem(ItemDO.OpTypePreStockSwith)) {
				Long preStock = preStockService.getPreStock(entity, itemKeyDto.getAppId());
				if (preStock != null && preStock <= stock) {
					stock = preStock;
				}
			}
			// 每日限量库存
			Long everydayStock = everydayLimitService.findEverydayStock(itemKeyDto);
			if (everydayStock != null && everydayStock <= stock) {
				stock = everydayStock;
			}
		}
		return stock;
	}

	@Override
	public Boolean consumeStock(ItemKeyDto itemKeyDto, String bizId, String bizSource) {
		Map<String, Boolean> context = new HashMap<>();
		try {
			// 扣每日限量库存
			if (everydayLimitService.isEverydayLimit(itemKeyDto)) {
				boolean decrEverydayStock = itemKeyStockService.decrEverydayStock(itemKeyDto);
				context.put(DECR_EVERDAY_STOCK, decrEverydayStock);
			}
			// 扣商品库存
			if (itemKeyDto.getItemDtoType().equals(ItemDO.TypeObject) || itemKeyDto.getItemDtoType().equals(ItemDO.TypeVirtual)) {
				boolean decrStock = itemKeyStockService.decrStock(itemKeyDto, bizId, bizSource);
				context.put(DECR_STOCK, decrStock);
			}
			if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
				ItemEntity itemEntity = BeanUtils.copy(itemKeyDto.getItem(), ItemEntity.class);
				// 扣预分配库存
				PreStockPointEntity pointStock = preStockService.getPointStock(itemEntity, itemKeyDto.getAppId());
				if (pointStock != null) {
					boolean decrPreStock = itemKeyStockService.decrPreStock(itemKeyDto.getAppId(), itemEntity.getId(), pointStock, bizId, bizSource);
					context.put(DECR_PRE_STOCK, decrPreStock);
				}
				// 扣定向库存
				if (ItemDO.TypeObject.equals(itemKeyDto.getItemDtoType()) || ItemDO.TypeCoupon.equals(itemEntity.getType())) {
					String redirectBizType = redirectType(bizSource);
					boolean decrAppSpecifyStock = itemKeyStockService.decrAppSpecifyStock(itemKeyDto.getAppId(), itemEntity, bizId, redirectBizType);
					context.put(DECR_APP_SPECIFY_STOCK, decrAppSpecifyStock);
				}
			}
			return true;
		} catch (Exception e) {
			Long itemId = itemKeyDto.getItem() == null ? null : itemKeyDto.getItem().getId();
			Long appItemId = itemKeyDto.getAppItem() == null ? null : itemKeyDto.getAppItem().getId();
			if (e instanceof GoodsException) {
				log.warn("consumeStock:appId=" + itemKeyDto.getAppId() + ":itemId=" + itemId + ":appItemId=" + appItemId + ":bizId=" + bizId + ":bizSource=" + bizSource + ":" + e.getMessage());
			} else {
				log.error("consumeStock:appId=" + itemKeyDto.getAppId() + ":itemId=" + itemId + ":appItemId=" + appItemId + ":bizId=" + bizId + ":bizSource=" + bizSource + ":", e);
			}
			// 返还商品库存
			Boolean stock = context.get(DECR_STOCK);
			if (stock != null && stock) {
				itemKeyStockService.rollbackStock(bizId, bizSource);
			}
			// 返还预分配库存
			Boolean preStock = context.get(DECR_PRE_STOCK);
			if (preStock != null && stock) {
				itemKeyStockService.rollbackPreStock(bizId);
			}
			// 返还定向库存
			Boolean appSpecifyStock = context.get(DECR_APP_SPECIFY_STOCK);
			if (appSpecifyStock != null && appSpecifyStock) {
				String redirectBizType = redirectType(bizSource);
				itemKeyStockService.rollbackAppSpecifyStock(bizId, redirectBizType);
			}
			// 返还每日限量库存
			Boolean everydayStock = context.get(DECR_EVERDAY_STOCK);
			if (everydayStock != null && everydayStock) {
				itemKeyStockService.rollbackEverydayStock(itemKeyDto);
			}
		}
		return false;
	}

	@Override
	public Boolean rollbackStock(ItemKeyDto itemKeyDto, String bizId, String bizSource) {
		if (itemKeyDto.getItemDtoType().equals(ItemDO.TypeObject) || itemKeyDto.getItemDtoType().equals(ItemDO.TypeVirtual)) {
			// 返还商品库存
			itemKeyStockService.rollbackStock(bizId, bizSource);
		}
		if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
			ItemEntity itemEntity = itemService.find(itemKeyDto.getItem().getId());
			// 返还预分配库存
			PreStockPointEntity pointStock = preStockService.getPointStock(itemEntity, itemKeyDto.getAppId());
			if (pointStock != null) {
				itemKeyStockService.rollbackPreStock(bizId);
			}
			// 返还定向库存
			if (ItemDO.TypeObject.equals(itemKeyDto.getItemDtoType()) || ItemDO.TypeCoupon.equals(itemEntity.getType())) {
				itemKeyStockService.rollbackAppSpecifyStock(bizId, bizSource);
			}
		}
		if (everydayLimitService.isEverydayLimit(itemKeyDto)) {
			// 返还每日限量库存
			itemKeyStockService.rollbackEverydayStock(itemKeyDto);
		}
		return true;
	}

	/**
	 * 定向库存类型
	 * 
	 * @param bizSource
	 * @return
	 */
	private String redirectType(String bizSource) {
		if (ItemStockConsumeDto.BIZ_SOURCE_NORMAL.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_HDTOOL.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_HDTOOL_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_ACTIVITY.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_ACTIVITY_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_NGAME.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_NGAME_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_GAME.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_GAME_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_GUESS.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_GUESS_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_QUESTION.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_QUESTION_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_QUIZZ.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_QUIZZ_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_SINGLE.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SINGLE_SPECIFY;
		}
		if (ItemStockConsumeDto.BIZ_SOURCE_TURNTABLE.equals(bizSource)) {
			return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_TURNTABLE_SPECIFY;
		}
		return ItemStockSpecifyConsumeEntity.BIZ_SOURCE_SPECIFY;
	}

}
