/**
 * Project Name:goods-center-biz
 * File Name:GoodsCouponBOImpl.java
 * Package Name:cn.com.duiba.goods.center.biz.bo.impl
 * Date:2016年5月24日下午4:40:02
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.bo.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ACGStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.ConsumeStockTypeUtil;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

/**
 * ClassName:GoodsCouponBOImpl <br/>
 * Date:     2016年5月24日 下午4:40:02 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("goodsCouponBO")
public class GoodsCouponBOImpl implements GoodsCouponBO{
    
    private static Logger log=LoggerFactory.getLogger(GoodsCouponBOImpl.class);
    
    private LoadingCache<Long, Lock> lockmap=CacheBuilder.newBuilder().expireAfterAccess(1, TimeUnit.HOURS).build(new CacheLoader<Long, Lock>(){
        @Override
    	public Lock load(Long key) throws Exception {
            return new ReentrantLock();
        }
    });
    
    @Autowired
    @Qualifier("goodsCouponService4Speed")
    private GoodsCouponService goodsCouponService;
    @Autowired
    private GoodsBatchService goodsBatchService;
    @Autowired
    private RemoteStockService remoteStockService;
    @Autowired
    private StockService stockService;
    
    @Override
    public GoodsCouponEntity takeCoupon(GoodsTypeEnum gtype, long gid, long consumerId,String bizNum) {
        GoodsBatchEntity using=goodsBatchService.getUsingBatch(gtype, gid);//已知此方法有缓存
        if(using==null){
            //此处访问量较大,假设库存已经全部没有了，为了避免大量请求进入刷新批次方法中，需要先进行一次快速库存判断，防止大量请求进入刷新批次方法中
            boolean saleout=fastCheckSaleout(gtype, gid);
            if(!saleout){
                using=retryLockRefreshBatchUsing(gtype, gid);
            }
            if(using==null){
                return null;
            }
        }
        return getCouponInCycle(using, consumerId,bizNum,1);
    }
    
    private GoodsCouponEntity getCouponInCycle(GoodsBatchEntity using, long consumerId,String bizNum,int tryCount) {
        GoodsTypeEnum gtype=GoodsTypeEnum.getGoodsTypeEnum(using.getGtype());
        long gid=using.getGid();
        if (tryCount > 2) {
			log.error("getCouponInCycle old error gtype=" + gtype.getGtype() + ",gid=" + gid + ",consumerId=" + consumerId + ",bizNum=" + bizNum + " getCouponInCycle 重试次数大于" + tryCount);
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
        int counter = tryCount + 1;
        //调用取券方法
        try {
            DBTimeProfile.enter("GoodsCouponBOImpl.getCouponInCycle invoke takeXXX");
            if(using.getBatchType()==GoodsBatchEntity.BatchTypeLink){
                try{
                    DBTimeProfile.enter("takeLinkCoupon");
                    return goodsCouponService.takeLinkCoupon(gtype, using, consumerId, bizNum);
                }finally{
                    DBTimeProfile.release();
                }
            }else if(using.getBatchType()==GoodsBatchEntity.BatchTypeRepeat){
                return goodsCouponService.takeRepeatCoupon(gtype, using, consumerId, bizNum);
            }else if(using.getBatchType()==GoodsBatchEntity.BatchTypeNormal){
                return goodsCouponService.takeNormalCoupon(gtype, using, consumerId,bizNum);
            }
            
            return null;
        } catch (RuntimeGoodsException e) {
            if(e.getErrorCode()==ErrorCode.E0202005){
                //如果批次已售罄，标记售罄，设置下一个批次
				GoodsBatchEntity batch = retryLockRefreshBatchUsing(gtype, gid);
				if (batch != null) {
					// 递归再次执行
					return getCouponInCycle(batch, consumerId, bizNum, counter);
				} else {
					throw new RuntimeGoodsException(ErrorCode.E0202006);
				}
                
            }else{
                //取券异常，返回
                log.error("takeOneCouponInBatch error gtype="+gtype.getGtype()+",gid="+gid+",consumerId="+consumerId,e);
                return null;
            }
        } finally{
            DBTimeProfile.release();
        }
        
    }
    
    /**
     * 内存锁，强制刷新正在使用的批次，并返回这个批次
     *
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @return
     * @since JDK 1.6
     */
    private GoodsBatchEntity retryLockRefreshBatchUsing(GoodsTypeEnum gtype,long gid){
        Lock lock=lockmap.getUnchecked(gid);
        if(lock!=null && lock.tryLock()){
            try{
                return forceGetUsingBatch(gtype, gid);
            }finally{
                lock.unlock();
            }
        }
        return null;
    }
    
    /**
     * 快速检测是否已经售罄
     *
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @return
     * @since JDK 1.6
     */
	private boolean fastCheckSaleout(GoodsTypeEnum gtype, long gid) {
		try {
			DBTimeProfile.enter("fastCheckSaleout");
			List<GoodsBatchEntity> batchs = goodsBatchService.findNormalBatchs(gtype, gid);
			boolean saleout = true;
			for (GoodsBatchEntity e : batchs) {
				if (e.getStatus() == GoodsBatchEntity.StatusNotUse || e.getStatus() == GoodsBatchEntity.StatusUsing) {
					saleout = false;
					break;
				}
			}
			return saleout;
		} finally {
			DBTimeProfile.release();
		}
	}
    
    private GoodsBatchEntity forceGetUsingBatch(GoodsTypeEnum gtype,long gid){
        try{
            DBTimeProfile.enter("forceGetUsingBatch");
			// 1.查询未删除和未过期的批次
			List<GoodsBatchEntity> batchs = goodsBatchService.findNormalBatchs(gtype, gid);
			List<Long> stockIds = new ArrayList<>();
			for (GoodsBatchEntity gbe : batchs) {
				stockIds.add(gbe.getStockId());
			}
			// 2.查询批次的库存
			Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
			for (GoodsBatchEntity e : batchs) {
				StockDto stock = stockMap.get(e.getStockId());
				if (stock != null && stock.getStock() <= 0) {
					// 3.批次库存为0的标记为已使用
					goodsBatchService.markBatchStatusUsed(e.getId());
				}
			}
            
            // 3.选取下一个可用批次
            batchs=goodsBatchService.findNormalBatchs(gtype, gid);
			List<GoodsBatchEntity> backups = new ArrayList<>();
            Date min=null;
            for(GoodsBatchEntity e:batchs){
                if(e.getStatus()==GoodsBatchEntity.StatusNotUse || e.getStatus()==GoodsBatchEntity.StatusUsing){
                    backups.add(e);
					if (min == null) {
						min = e.getStartDay();
					} else {
						if (min.getTime() > e.getStartDay().getTime()) {
							min = e.getStartDay();
						}
					}
                }
            }
    
			for (GoodsBatchEntity e : backups) {
				if (e.getStartDay().equals(min)) {
					// 4.批次设置为使用中
					goodsBatchService.markBatchStatusUsing(e.getId());
					return goodsBatchService.find(e.getId());
				}
			}
        
            return null;
        }finally{
            DBTimeProfile.release();
        }
    }
    
    @Override
    public void completeCoupon(Long couponId, long orderId) {
        goodsCouponService.completeCoupon(couponId, orderId);
    }

    @Override
    public Boolean rollbackCoupon(Long couponId,String bizNum) {
        GoodsCouponEntity coupon=goodsCouponService.find(couponId);
        long goodsBatchId=coupon.getGoodsBatchId();
        GoodsBatchEntity batch=goodsBatchService.find(goodsBatchId);
        
        if(batch.getBatchType()==GoodsBatchEntity.BatchTypeNormal){
            Boolean succ=goodsCouponService.rollbackNormalCoupon(couponId);
            if(!succ){
                return succ;
            }
        }
        //所有的类型都需要尝试到库存中心返还库存，除了主库存，还有定向库存等等需要返还
        DubboResult<Boolean> stockRet=remoteStockService.rollbackStock(ConsumeStockTypeUtil.getConsumeStockTypes(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype())).getType(), bizNum);
        if(!stockRet.isSuccess()){
            log.error("remoteStockService.rollbackStock fail,couponId="+couponId+",bizNum="+bizNum);
            return false;
        }
        return stockRet.getResult();
    }

	@Override
	public GoodsCouponEntity findCoupon(Long goodsCouponId) {
		return goodsCouponService.find(goodsCouponId);
	}

	@Override
	public Long findGoodsStock(GoodsTypeEnum gtype, long gid) {
		return goodsBatchService.getSumBatchStockBatch(gtype, gid);
	}

	@Override
	public List<GoodsStockDto> findGoodsStockByBatch(List<GoodsStockDto> goodsDtos) {
		List<GoodsStockDto> stocks = new ArrayList<>();
		if (goodsDtos == null) {
			return stocks;
		}
		Map<GoodsTypeEnum, List<Long>> map = new HashMap<>();
		for (GoodsStockDto goods : goodsDtos) {
			List<Long> gids = map.get(goods.getGtype());
			if (gids == null) {
				gids = new ArrayList<>();
			}
			gids.add(goods.getGid());
			map.put(goods.getGtype(), gids);
		}
		// 已知GoodsTypeEnum类型4 for不会太多
		for (Map.Entry<GoodsTypeEnum, List<Long>> entry : map.entrySet()) {
			if (entry.getValue().isEmpty()) {
				continue;
			}
			List<ACGStockDto> dtos = goodsBatchService.findStockByGids(entry.getKey(), entry.getValue());
			for (ACGStockDto acg : dtos) {
				GoodsStockDto dto = new GoodsStockDto();
				dto.setGtype(entry.getKey());
				dto.setGid(acg.getAcgId());
				dto.setStock(acg.getStock());
				dto.setTotalStock(acg.getTotalStock());
				stocks.add(dto);
			}
		}
		return stocks;
	}

	@Override
	public GoodsBatchEntity findBatch(Long goodsBatchId) {
		return goodsBatchService.find(goodsBatchId);
	}

	@Override
	public GoodsBatchEntity findUsingBatch(GoodsTypeEnum gtype, long gid) {
		List<GoodsBatchEntity> batchs = goodsBatchService.findNormalBatchs(gtype, gid);
		for(GoodsBatchEntity gd : batchs){
			if(gd.getStatus() == GoodsBatchEntity.StatusUsing){
				return gd;
			}
		}
		return null;
	}

	@Override
	public GoodsCouponEntity takeCoupon(GoodsTypeEnum gtype, long gid, Long consumerId, Long appId, String bizNum) {
		// 1.获取当前使用中的批次
		GoodsBatchEntity using=goodsBatchService.getUsingBatch(gtype, gid);
        if(using==null){
        	//2.如果没有使用中的批次进行切换下一个可用批次
            boolean saleout=fastCheckSaleout(gtype, gid);
            if(!saleout){
                using=retryLockRefreshBatchUsing(gtype, gid);
            }
            if(using==null){
                return null;
            }
        }
        // 3.从使用中的批次发一个券码
        return getCouponInCycle(using, consumerId, appId, bizNum, 1);
	}
	
	
	private GoodsCouponEntity getCouponInCycle(GoodsBatchEntity using, Long consumerId, Long appId, String bizNum, int tryCount) {
		GoodsTypeEnum gtype = GoodsTypeEnum.getGoodsTypeEnum(using.getGtype());
		long gid = using.getGid();
		if (tryCount > 2) {
			log.error("getCouponInCycle error gtype=" + gtype.getGtype() + ",gid=" + gid + ",consumerId=" + consumerId + ",appId=" + appId + ",bizNum=" + bizNum + " getCouponInCycle 重试次数大于" + tryCount);
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
		int counter = tryCount + 1;
		try {
			DBTimeProfile.enter("getCouponInCycle");
			// 1. 从当前使用中的批次发一个券码
			if (using.getBatchType() == GoodsBatchEntity.BatchTypeLink) {
				return goodsCouponService.takeLinkCoupon(gtype, using, consumerId, appId, bizNum);
			} else if (using.getBatchType() == GoodsBatchEntity.BatchTypeRepeat) {
				return goodsCouponService.takeRepeatCoupon(gtype, using, consumerId, appId, bizNum);
			} else if (using.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
				return goodsCouponService.takeNormalCoupon(gtype, using, consumerId, appId, bizNum);
			}
			return null;
		} catch (RuntimeGoodsException e) {
			// 2. 如果当前使用中的批次已售罄
			if (e.getErrorCode() == ErrorCode.E0202005) {
				// 3. 刷新批次切换到下一个可用批次
				GoodsBatchEntity batch = retryLockRefreshBatchUsing(gtype, gid);
				if (batch != null) {
					// 4.再次执行一次发券
					return getCouponInCycle(batch, consumerId, appId, bizNum, counter);
				} else {
					throw new RuntimeGoodsException(ErrorCode.E0202006);
				}
			} else {
				// 取券异常，返回
				log.error("getCouponInCycle error gtype=" + gtype.getGtype() + ",gid=" + gid + ",consumerId=" + consumerId + ",appId=" + appId + ",bizNum=" + bizNum, e);
				return null;
			}
		} finally {
			DBTimeProfile.release();
		}
	}

}

