/**
 * Project Name:goods-center-biz File Name:GoodsBatchServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午4:43:49 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ACGStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.enums.CouponType;
import cn.com.duiba.goods.center.biz.dao.GoodsBatchDao;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponRetrieveLogEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponRetrieveLogService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.AppendUploadUtil;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

import com.alibaba.ttl.threadpool.TtlExecutors;

/**
 * ClassName:GoodsBatchServiceImpl <br/>
 * Date: 2016年5月25日 下午4:43:49 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
public class GoodsBatchServiceImpl implements GoodsBatchService {

    private static Logger             log = LoggerFactory.getLogger(GoodsBatchServiceImpl.class);
    @Autowired
    private GoodsBatchDao             goodsBatchDao;
    @Autowired
    private RemoteStockService        remoteStockService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;
    @Autowired
    private GoodsCouponRetrieveLogService goodsCouponRetrieveLogService;
    @Autowired
    private StockService stockService;
    @Autowired
    private AppendUploadUtil appendUploadUtil;
    @Autowired
    private GoodsCouponService goodsCouponService;
    
    private static ExecutorService executorService = TtlExecutors.getTtlExecutorService(Executors.newSingleThreadExecutor());

    private Date date;
    
    private Long position = 0L;
    
    private static final long PERRETRIEVEGOODSCOUPON = 999;
    
    private static final String FILEURLPREFIX = "http://yun.duiba.com.cn/";
    
    private String timeStart = " 00:00:00";
    private String timeEnd = " 23:59:59";

    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay,Long stockId) {
		Date startDate = DateUtils.getSecondDate(DateUtils.getDayStr(startDay) + timeStart);
		Date endDate = DateUtils.getSecondDate(DateUtils.getDayStr(endDay) + timeEnd);
        return goodsBatchDao.insert(gtype, gid, CouponType.Normal, startDate, endDate, stockId);
    }
    
    @Override
    public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
    	Date startDate = DateUtils.getSecondDate(DateUtils.getDayStr(startDay) + timeStart);
		Date endDate = DateUtils.getSecondDate(DateUtils.getDayStr(endDay) + timeEnd);
        return goodsBatchDao.insert(gtype, gid, CouponType.Link, startDate, endDate, stockId);
    }

    @Override
    public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
    	Date startDate = DateUtils.getSecondDate(DateUtils.getDayStr(startDay) + timeStart);
		Date endDate = DateUtils.getSecondDate(DateUtils.getDayStr(endDay) + timeEnd);
        return goodsBatchDao.insert(gtype, gid, CouponType.Repeat, startDate, endDate, stockId);
    }

    @Override
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.findNormalBatchs");
            return goodsBatchDao.selectNotDeletedNotExpiredGtypeGid(gtype, gid);
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.findNotDeletedBatchs");
            return goodsBatchDao.selectNotDeletedByGtypeGid(gtype, gid);
        }finally{
            DBTimeProfile.release();
        }
        
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        GoodsBatchEntity e = goodsBatchDao.select(batchId);
        if (e.getImporting()) {
            return false;
        }
		if (e.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
			boolean existUsedCoupon = goodsCouponService.findBatchExsitUsedCoupon(gtype, gid, batchId);
			if (existUsedCoupon) {
				return false;
			}
		}else{
			DubboResult<StockDto> ret= remoteStockService.findDto(e.getStockId());
            if(ret.isSuccess()){
            	StockDto dto = ret.getResult();
                Long stock = dto.getStock();
                Long totalStock = dto.getTotalStock();
                if(!stock.equals(totalStock)){
                	return false;
                }
            }else{
                throw new RuntimeGoodsException(ErrorCode.E9999999);
            }
		}
        int ret=goodsBatchDao.deleteBatch(gtype,gid,batchId);
        if(ret==1){
        	//删除批次后将库存此批次的库存置为0
        	DubboResult<Long> dubbo = remoteStockService.find(e.getStockId());
        	remoteStockBackendService.decreaseItemStock(e.getStockId(), dubbo.getResult());
            return true;
        } else {
            return false;
        }
    }

    @Override
    public GoodsBatchEntity find(Long goodsBatchId) {
        return goodsBatchDao.select(goodsBatchId);
    }

    @Override
    public GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid) {
        try {
            DBTimeProfile.enter(getClass().getSimpleName() + ".refreshBatchUsing");

            List<GoodsBatchEntity> batchs = findNormalBatchs(gtype, gid);
			List<Long> stockIds = new ArrayList<>();
			for (GoodsBatchEntity gbe : batchs) {
				stockIds.add(gbe.getStockId());
			}
			Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
			for (GoodsBatchEntity e : batchs) {
				StockDto stock = stockMap.get(e.getStockId());
				if (stock != null && stock.getStock() <= 0) {
					//标记无库存
                    goodsBatchDao.updateStatusUsed(e.getId());
                    e.setStatus(GoodsBatchEntity.StatusUsed);
                }
            }

            //选当前使用用的批次
            for(GoodsBatchEntity e:batchs){
	            if(e.getStatus()==GoodsBatchEntity.StatusUsing){
	        		return e;
	        	}
            }

            //选下一个批次
            batchs=findNormalBatchs(gtype, gid);
            List<GoodsBatchEntity> backups=new ArrayList<>();
            Date min=null;
            for(GoodsBatchEntity e:batchs){
                if(e.getStatus()==GoodsBatchEntity.StatusNotUse){
                    backups.add( e);
                    if(min==null){
                        min=e.getStartDay();
                    }else{
                        if(min.getTime()>e.getStartDay().getTime()){
                            min=e.getStartDay();
                        }
                    }
                }
            }

            for (GoodsBatchEntity e : backups) {
                if (e.getStartDay().equals(min)) {
                    // 选中
                    goodsBatchDao.updateStatusUsing(e.getId());
                    return find(e.getId());
                }
            }

            return null;
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public Long getSumBatchStock(GoodsTypeEnum gtype, long gid) {
    	return getSumBatchStockBatch(gtype, gid);
    }
    
    @Override
    public Long getSumBatchStockBatch(GoodsTypeEnum gtype, long gid) {
    	long total = 0L;
    	List<Long> stockIds = new ArrayList<>();
    	try {
			DBTimeProfile.enter("findNormalBatchs");
			List<GoodsBatchEntity> list = findNormalBatchs(gtype, gid);
			for (GoodsBatchEntity e : list) {
				//已删除，或者已使用的，跳过
				if (e.getStatus() != GoodsBatchEntity.StatusDeleted && e.getStatus()!=GoodsBatchEntity.StatusUsed) {
					stockIds.add(e.getStockId());
				}
			}
    	}finally{
			DBTimeProfile.release();
		}
    	try{
			DBTimeProfile.enter("remoteStockService.findBatch");
			if(!stockIds.isEmpty()){
				DubboResult<Map<Long, Long>> stockRet = remoteStockService.findBatch(stockIds);
				if(!stockRet.isSuccess()){
					log.error("remoteStockService.findBatch error,msg="+stockRet.getMsg());
				}
				for (Map.Entry<Long, Long> entry : stockRet.getResult().entrySet()) {
					total += entry.getValue();
				}
			}
		}finally{
			DBTimeProfile.release();
		}
    	return total;
    }

    @Override
    public Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchServiceImpl.getTotalAllBatchStock");
            List<GoodsBatchEntity> list = findNotDeletedBatchs(gtype, gid);
            List<Long> stockIds = new ArrayList<>();
            for (GoodsBatchEntity e : list) {
                if (e.getStatus() != GoodsBatchEntity.StatusDeleted) {
                	stockIds.add(e.getStockId());
                }
            }
			Map<Long, StockDto> stocks = stockService.findStockByStockIds(stockIds);
			long total = 0L;
			for (GoodsBatchEntity e : list) {
				if (e.getStatus() != GoodsBatchEntity.StatusDeleted) {
					StockDto dto = stocks.get(e.getStockId());
					long tatalStock = dto == null ? 0 : dto.getTotalStock();
					total += tatalStock;
				}
			}
			return total;
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public void markBatchStatusUsed(long batchId) {
        goodsBatchDao.updateStatusUsed(batchId);
    }

    @Override
    public void markBatchStatusUsing(long batchId) {
        goodsBatchDao.updateStatusUsing(batchId);
    }

    @Override
    public void markBatchStatusNotUsed(long batchId) {
        goodsBatchDao.updateStatusNotUsed(batchId);
    }

    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay) {
    	Date startDate = DateUtils.getSecondDate(DateUtils.getDayStr(startDay) + timeStart);
		Date endDate = DateUtils.getSecondDate(DateUtils.getDayStr(endDay) + timeEnd);
        if (goodsBatchDao.updateValidDate(batchId, startDate, endDate) == 1) {
            return true;
        }
        return false;
    }

    @Override
    public boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType) {
        if (gtype.equals(GoodsTypeEnum.ADVERT)) {
            int row = goodsBatchDao.updateBatchType(gid, dstType);
            if (row >= 1) {
                return true;
            }
            return false;
        }
        return false;
    }

    @Override
    public Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
    	GoodsBatchEntity e = goodsBatchDao.select(batchId);
        int row = goodsBatchDao.physicalDeleteBatch(gtype, gid, batchId);
        if (row >= 1) {
        	//删除批次后将库存此批次的库存置为0
        	DubboResult<Long> ret = remoteStockService.find(e.getStockId());
        	remoteStockBackendService.decreaseItemStock(e.getStockId(), ret.getResult());
            return true;
        }
        return false;
    }

	@Override
	public List<GoodsBatchEntity> findGoodsBatchs(GoodsTypeEnum gtype,
			List<Long> gids) {
		return goodsBatchDao.selectGoodsBatchs(gtype,gids);
	}
   

	@Override
	public void refreshUsingBatch(GoodsTypeEnum gtype, long gid) {
		try {
			DBTimeProfile.enter(getClass().getSimpleName() + ".refreshBatchUsing");

			List<GoodsBatchEntity> batchs = findNormalBatchs(gtype, gid);

			// 选当前使用用的批次
			for (GoodsBatchEntity e : batchs) {
				if (e.getStatus() == GoodsBatchEntity.StatusUsing) {
					return;
				}
			}

			// 选下一个批次
			batchs = findNormalBatchs(gtype, gid);
			List<GoodsBatchEntity> backups = new ArrayList<>();
			Date min = null;
			for (GoodsBatchEntity e : batchs) {
				if (e.getStatus() == GoodsBatchEntity.StatusNotUse) {
					backups.add(e);
					if (min == null) {
						min = e.getStartDay();
					} else {
						if (min.getTime() > e.getStartDay().getTime()) {
							min = e.getStartDay();
						}
					}
				}
			}

			for (GoodsBatchEntity e : backups) {
				if (e.getStartDay().equals(min)) {
					// 选中
					goodsBatchDao.updateStatusUsing(e.getId());
					return;
				}
			}
		} finally {
			DBTimeProfile.release();
		}
	}

    @Override
    public Long retrieveGoodsBatch(final GoodsTypeEnum gtype, final Long gid, final Long batchId, final Long count) throws GoodsException {

        //获取分布式锁
        if(!this.getBatchLock(batchId,gid,gtype)){
            throw new GoodsException("0","批次已被锁");
        }

        //检查批次状态
        final GoodsBatchEntity batch = this.find(batchId);
        Long rst;
        if(batch==null){
            throw new GoodsException("0","批次不存在");
        }
        final int oriStatus = batch.getStatus();

        if(GoodsBatchEntity.StatusNotUse!=oriStatus && GoodsBatchEntity.StatusUsing!=oriStatus){
            throw new GoodsException("0","[已使用,未使用]的批次才能回收");
        }
        date = new Date();

        try {
            // 冻结批次,更新成已使用
            this.markBatchStatusUsed(batchId);
            rst = this.createRetrieveLog(gtype, gid, batchId, 0, GoodsCouponRetrieveLogEntity.retrieveStatus_ing, "");
            final Long count1 = rst;
            executorService.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        doRetrieveGoodsCoupon(batch,oriStatus,gtype, gid, batchId, count, count1);
                    } catch (Exception e) {
                        log.error("",e);
                    }finally {
                        updateToOriStatus(batchId,oriStatus);
//                        释放分布式锁
                        releaseBatchLock(batchId,gid,gtype);

                    }
                }
            });
        }catch (Exception e){
            this.updateToOriStatus(batchId,oriStatus);
            log.error("回收批次出错:",e);
            throw e;
        }
        return rst;
    }

    @Override
    public boolean getBatchLock(Long goodsBatchId, Long gid, GoodsTypeEnum gtype) {
        //在GoodsBatchRedisService.java中获取锁
        return false;
    }

    @Override
    public void releaseBatchLock(Long goodsBatchId, Long gid, GoodsTypeEnum gtype) {
    	 // Do nothing because of X and Y.
    }

    private void updateToOriStatus(Long batchId,int oriStatus){
        if(oriStatus==GoodsBatchEntity.StatusNotUse){
            this.markBatchStatusNotUsed(batchId);
        }else if(oriStatus==GoodsBatchEntity.StatusUsing){
            this.markBatchStatusUsing(batchId);
        }
    }

    private void doRetrieveGoodsCoupon(GoodsBatchEntity batch ,int oriStatus,GoodsTypeEnum gtype, Long gid, Long batchId, Long count,Long logid) throws GoodsException {
        int successRetrieveCount = 0;
        position = 0L;
        String filePath = this.getFileName(gid,batchId);
        try {

            DubboResult<Long> dubboResult = remoteStockService.find(batch.getStockId());
            if(dubboResult.isSuccess()){
                Long stockNum = dubboResult.getResult();
                if(stockNum!=null){
                    if(stockNum.longValue()==0){
                        log.warn("库存为0");
                        goodsCouponRetrieveLogService.updateStatusFail(logid);
                        throw new GoodsException("0","库存为0");
                    }
                    //如果回收数量大于库存数量,则回收数量等于库存数量
                    if(stockNum.longValue()<count.longValue()){
                        count = stockNum;
                    }

                }else{
                    log.error("库存不存在");
                    goodsCouponRetrieveLogService.updateStatusFail(logid);
                    throw new GoodsException("0","库存不存在");
                }
            }

            long forCount = caculateForCount(count);
            String key = filePath+".txt";
            appendUploadUtil.setKey(key);

            boolean isLastPage = false;
            List<GoodsCouponEntity> list;
            if(forCount==1){
                list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                        0,(int)count.longValue());
                isLastPage = true;
                successRetrieveCount += this.deleteGoodsCoupons(gid,list,logid,isLastPage);
            }else{

                try {
                    for (int i = 0 ;i<forCount;i++){
                        //如果为最后一页
                        if(i==forCount-1){
                            list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                                    0,(int)(count.longValue()-successRetrieveCount));
                            isLastPage = true;
                        }else{
                            list = goodsCouponService.findPageByStatus(gtype, gid, batchId, GoodsCouponEntity.StatusNew,
                                    0,(int)PERRETRIEVEGOODSCOUPON);
                        }
                        successRetrieveCount += this.deleteGoodsCoupons(gid,list,logid,isLastPage);
                    }
                }catch (Exception e){
                    log.error("分页回收批次出错", e);
                }
            }

            //上传文件,更新库存,更新回收记录状态
            completeRetrieve(gid,batchId,count,batch,successRetrieveCount,logid,key);
            log.info("成功回收券码总数"+successRetrieveCount+"条");
        }catch (Exception e){
            log.error("回收券码失败。",e);
            throw  e;
        }
    }

    /**
     * @param gtype
     * @param gid
     * @param batchId
     * @param statusUsed
     * @param start
     * @param PERRETRIEVEGOODSCOUPON
     * 写入文件,并删除数据
     * @return
     */
    private int deleteGoodsCoupons(Long gid, List<GoodsCouponEntity> list ,Long logid,boolean isLastPage) {
        int rst = 0;

        //把优惠券写入文件 //删除优惠券
        if(!CollectionUtils.isEmpty(list)){
            log.info("分页回收券码数量:"+list.size());
            try {
                List<Long> deleteIds = appendContenToOss(list,isLastPage);
                rst = goodsCouponService.deleteGoodsCouponByIds(gid, deleteIds);
            }catch (Exception e){
                log.error("appendContenToOss, deleteGoodsCouponByIds", e);
                rst = 0;
            }
        }else {
            log.info("没有可回收的券码");
        }
        return rst;
    }

    private void completeRetrieve(Long gid, Long batchId,Long count,
                                  GoodsBatchEntity batch, int successRetrieveCount,
                                  Long logId,String filePath) {

        //更新库次及缓存数据
        remoteStockBackendService.decreaseItemStock(batch.getStockId(), successRetrieveCount);
        updateLogStatus(logId,successRetrieveCount,count,filePath);

    }

    private void updateLogStatus(Long logId,int successRetrieveCount,Long count,String fileUrl){
        String url = FILEURLPREFIX+fileUrl;
        log.info("update the fileUrl to the tb_goods_coupon_retrieve_log "+url);
        if(successRetrieveCount !=0 && count.longValue()==successRetrieveCount){
            goodsCouponRetrieveLogService.updateStatusSuccess(logId,url,successRetrieveCount);
        }else if(successRetrieveCount !=0 && count.longValue()>successRetrieveCount){
            goodsCouponRetrieveLogService.updateStatusPart(logId,url,successRetrieveCount);
        }
    }

    private long caculateForCount(Long count) {
        long forCount;
        if((count%PERRETRIEVEGOODSCOUPON)==0){
            forCount = count.longValue()/PERRETRIEVEGOODSCOUPON;
        }else{
            forCount = count.longValue()/PERRETRIEVEGOODSCOUPON+1;
        }
        return forCount;
    }

    private String getFileName(Long gid, Long batchId) {
        return "tuia/manager/coupon-retrieve/"+gid+"_"+batchId+"_"+date.getTime();
    }

    private Long createRetrieveLog(GoodsTypeEnum gtype, Long gid, Long batchId,
                                   int successRetrieveCount, Integer retrieveStatus,String filePath) {
        GoodsCouponRetrieveLogEntity entity = new GoodsCouponRetrieveLogEntity();
        entity.setBatchId(batchId);
        entity.setGid(gid);
        entity.setStatus(retrieveStatus);
        entity.setGoodsType(gtype.getGtype());
        entity.setRecoveryAmount(successRetrieveCount);
        entity.setFileUrl(filePath);
        return goodsCouponRetrieveLogService.create(entity);
    }

    private List<Long> appendContenToOss(List<GoodsCouponEntity> list,boolean isLastPage) throws GoodsException {
        List<Long> rst = new ArrayList<>(1000);
        StringBuilder content = new StringBuilder();
        int i = 0;
        for(GoodsCouponEntity entity:list){
            i++;
            if(!StringUtils.isEmpty(entity.getCode())){
                content.append(entity.getCode());
            }
            if(!StringUtils.isEmpty(entity.getPassword())){
                content.append("\t");
                content.append(entity.getPassword());
            }
            if(!(isLastPage && i==list.size())){
                content.append("\n");
            }
            rst.add(entity.getId());


        }
        position = appendUploadUtil.upload(content.toString(),position);
        return rst;
    }

	@Override
	public GoodsBatchEntity findByStartDayAndDay(GoodsTypeEnum gtype, Long gid, Date startDay, Date endDay) {
    	Date startDate = DateUtils.getSecondDate(DateUtils.getDayStr(startDay) + timeStart);
		Date endDate = DateUtils.getSecondDate(DateUtils.getDayStr(endDay) + timeEnd);
		return goodsBatchDao.selectByStartDayAndDay(gtype, gid, startDate, endDate);
	}

	@Override
	public List<ACGStockDto> findStockByGids(GoodsTypeEnum gtype, List<Long> gids) {
		List<GoodsBatchEntity> batchs = goodsBatchDao.selectGoodsBatchs(gtype, gids);
		List<Long> stockIds = new ArrayList<>();
		for (GoodsBatchEntity entity : batchs) {
			stockIds.add(entity.getStockId());
		}
		if (stockIds.isEmpty()) {
			return new ArrayList<>();
		}
		Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
		Map<Long, StockDto> batchStockMap = new HashMap<>();
		for (GoodsBatchEntity entity : batchs) {
			batchStockMap.put(entity.getId(), stockMap.get(entity.getStockId()));
		}
		// 分组计算商品库存
		Map<Long, StockDto> goodsStockMap = new HashMap<>();
		Date now = new Date();
		for (GoodsBatchEntity entity : batchs) {
			StockDto batchStock = batchStockMap.get(entity.getId());
			if (batchStock == null) {
				continue;
			}
			StockDto goodsStock = goodsStockMap.get(entity.getGid());
			if (goodsStock == null) {
				goodsStock = new StockDto();
			}
			Long stock = goodsStock.getStock();
			Long totalStock = goodsStock.getTotalStock();
			// 剩余库存不加入已使用，删除，过期的
			if (GoodsBatchEntity.StatusUsed != entity.getStatus() && GoodsBatchEntity.StatusDeleted != entity.getStatus() && entity.getEndDay().after(now)) {
				stock += batchStock.getStock();
			}
			// 总库存不加入删除的
			if (GoodsBatchEntity.StatusDeleted != entity.getStatus()) {
				totalStock += batchStock.getTotalStock();
			}
			goodsStock.setStock(stock);
			goodsStock.setTotalStock(totalStock);
			goodsStockMap.put(entity.getGid(), goodsStock);
		}
		List<ACGStockDto> acgs = new ArrayList<>();
		for (Long gid : gids) {
			StockDto stock = goodsStockMap.get(gid);
			if (stock == null) {
				stock = new StockDto();
			}
			ACGStockDto dto = new ACGStockDto();
			dto.setAcgId(gid);
			dto.setStock(stock.getStock());
			dto.setTotalStock(stock.getTotalStock());
			acgs.add(dto);
		}
		return acgs;
	}

	@Override
	public ACGStockDto findStockByGid(GoodsTypeEnum gtype, Long gid) {
		List<GoodsBatchEntity> batchs = goodsBatchDao.selectNotDeletedByGtypeGid(gtype, gid);
		List<Long> stockIds = new ArrayList<>();
		for (GoodsBatchEntity entity : batchs) {
			stockIds.add(entity.getStockId());
		}
		if (stockIds.isEmpty()) {
			return new ACGStockDto();
		}
		Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
		Map<Long, StockDto> batchStockMap = new HashMap<>();
		for (GoodsBatchEntity entity : batchs) {
			batchStockMap.put(entity.getId(), stockMap.get(entity.getStockId()));
		}
		Date now = new Date();
		Long stock = 0L;
		Long totalStock = 0L;
		for (GoodsBatchEntity entity : batchs) {
			StockDto batchStock = batchStockMap.get(entity.getId());
			if (batchStock == null) {
				continue;
			}
			// 剩余库存不加入已使用，删除，过期的
			if (GoodsBatchEntity.StatusUsed != entity.getStatus() && GoodsBatchEntity.StatusDeleted != entity.getStatus() && entity.getEndDay().after(now)) {
				stock += batchStock.getStock();
			}
			// 总库存不加入删除的
			if (GoodsBatchEntity.StatusDeleted != entity.getStatus()) {
				totalStock += batchStock.getTotalStock();
			}
		}
		ACGStockDto gstock = new ACGStockDto();
		gstock.setAcgId(gid);
		gstock.setStock(stock);
		gstock.setTotalStock(totalStock);
		return gstock;
	}

	@Override
	public List<GoodsBatchDto> findBatchsNoCode(GoodsTypeEnum gtype, List<Long> gids) {
		List<GoodsBatchEntity> ret = goodsBatchDao.selectGoodsBatchs(gtype, gids);
		List<GoodsBatchEntity> batchs = new ArrayList<>();
		List<Long> stockIds = new ArrayList<>();
		for (GoodsBatchEntity entity : ret) {
			if (entity.getStatus() != GoodsBatchEntity.StatusDeleted) {
				batchs.add(entity);
				stockIds.add(entity.getStockId());
			}
		}
		if (stockIds.isEmpty()) {
			return new ArrayList<>();
		}
		Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
		Map<Long, StockDto> batchStockMap = new HashMap<>();
		for (GoodsBatchEntity entity : batchs) {
			batchStockMap.put(entity.getId(), stockMap.get(entity.getStockId()));
		}
		List<GoodsBatchDto> batchDtos = BeanUtils.copyList(batchs, GoodsBatchDto.class);
		for (GoodsBatchDto dto : batchDtos) {
			StockDto batchStock = batchStockMap.get(dto.getId());
			if (batchStock == null) {
				batchStock = new StockDto();
			}
			dto.setStock(batchStock.getStock());
			dto.setTotalStock(batchStock.getTotalStock());
		}
		return batchDtos;
	}
}
