/**
 * Project Name:goods-center-biz File Name:GoodsCouponService4Speed.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午1:07:36 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RedisKeyTool;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName:GoodsCouponService4Speed <br/>
 * Date: 2016年5月25日 下午1:07:36 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
@Service("goodsCouponService4Speed")
public class GoodsCouponService4Speed extends GoodsCouponServiceImpl implements GoodsCouponService, InitializingBean {
	
    private static Logger               log = LoggerFactory.getLogger(GoodsCouponService4Speed.class);
	
    private static final int              REDISBATCHSIZE = 10000;
    /** 存放队列，内部将GoodsCouponEntity使用json转换为String */
    @Autowired
    private RedisTemplate<String, String> redisStringTemplate;
    /** redis券队列缓存管理类 */
    private CouponQueueRedisCache         couponQueueRedisCache;
    /** 批次是否售罄的redis缓存管理类 */
    private BatchSaleoutRedisCache        batchSaleoutRedisCache;

    @Override
    public void afterPropertiesSet() throws Exception {
        batchSaleoutRedisCache=new BatchSaleoutRedisCache(redisStringTemplate);
        couponQueueRedisCache=new CouponQueueRedisCache(redisStringTemplate,batchSaleoutRedisCache);
        
    }
    
    @Override
    public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
    	boolean ret = super.deleteBatchUnusedCoupons(batch);
    	couponQueueRedisCache.clearCache(batch.getId());
    	return ret;
    }
    
    @Override
    public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
    	boolean ret = super.deleteUnusedCoupon(gtype, gid, couponId, batchId);
    	couponQueueRedisCache.clearCache(batchId);
    	return ret;
    }
    

    @Override
    public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                              String bizNum) {
        try {
            DBTimeProfile.enter("GoodsCouponService4Speed.takeNormalCoupon");

            ConsumerDO c = remoteConsumerService.find(consumerId);

            if (!consumeStocks(gtype, batch, consumerId, bizNum, c)) {
                // 如果减库存失败，返还null
                return null;
            }

            if (!batchSaleoutRedisCache.isSaleOut(batch.getId())) {
                return tryTakeCoupon(gtype, batch, consumerId, 1);
            } else {
                // 该批次已售罄
                throw new RuntimeGoodsException(ErrorCode.E0202005);
            }
        } finally {
            DBTimeProfile.release();
        }
    }
    
	@Override
	public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		Boolean consumeSuccess = false;
		try {
			DBTimeProfile.enter("GoodsCouponService4Speed.takeNormalCoupon");
			// 1.库存中心扣库存
			consumeSuccess = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSuccess) {
				return null;
			}
			// 2.取一个券码
			return tryTakeCoupon(gtype, batch, consumerId, 1, bizNum);
		} catch (Exception e) {
			// 4.出现异常如果库存中心扣库存成功返还库存
			if (consumeSuccess) {
				rollbackStocks(gtype, bizNum);
			}
			throw e;
		} finally {
			DBTimeProfile.release();
		}
	}
	
    /**
     * 从redis队列中取数据 执行lock动作 如果lock动作失败，递归最多尝试3次
     *
     * @author xuhengfei
     * @param gtype
     * @param batch
     * @param consumerId
     * @param tryCount
     * @return
     * @since JDK 1.6
     */
    private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, int tryCount) {
        if (tryCount > 3) {
            throw new RuntimeGoodsException(ErrorCode.E0202003);
        }
        try {
        	int counter = tryCount + 1;
            DBTimeProfile.enter("GoodsCouponService4Speed.tryTakeCoupon");

            long goodsBatchId = batch.getId();
            long gid = batch.getGid();

            long size = couponQueueRedisCache.getQueueSize(goodsBatchId);
            if (size > 0) {
                GoodsCouponEntity entity = couponQueueRedisCache.pop(goodsBatchId);
                boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId);
                if (success) {
                    return entity;
                } else {
                    return tryTakeCoupon(gtype, batch, consumerId, counter);
                }
            } else {
                couponQueueRedisCache.tryLoadCouponByBatchId(gtype, gid, goodsBatchId, REDISBATCHSIZE);
                return tryTakeCoupon(gtype, batch, consumerId, counter);
            }
        } finally {
            DBTimeProfile.release();
        }
    }

    /**
     * BatchSaleoutRedisCache
     */
    public static class BatchSaleoutRedisCache {

        private RedisTemplate<String, String> redisTemplate;

        /**
         * BatchSaleoutRedisCache
         * @param redis
         */
        public BatchSaleoutRedisCache(RedisTemplate<String, String> redis) {
            this.redisTemplate = redis;
        }

        /**
         * 是否售罄，如果缓存中没有数据，默认返回还没有售罄
         *
         * @author xuhengfei
         * @param batchId
         * @return
         * @since JDK 1.6
         */
		public boolean isSaleOut(long batchId) {
			try {
				String saleoutStr = redisTemplate.opsForValue().get(getSaleOutKey(batchId));
				Boolean saleout = null;
				if (saleoutStr != null) {
					saleout = Boolean.valueOf(saleoutStr);
				}
				if (saleout == null) {
					return false;
				}
				return Boolean.TRUE.equals(saleout);
			} catch (Exception e) {
				log.error("isSaleOut error: batchId=" + batchId, e);
			}
			return false;
		}

        /**
         * 设置一个批次是否售罄的缓存值，1分钟后失效删除
         *
         * @author xuhengfei
         * @param batchId
         * @param saleout
         * @since JDK 1.6
         */
        public void setSaleOut(long batchId, boolean saleout) {
            redisTemplate.opsForValue().set(getSaleOutKey(batchId), String.valueOf(saleout));
            redisTemplate.expire(getSaleOutKey(batchId), 1, TimeUnit.MINUTES);
        }

        private String getSaleOutKey(long batchId) {
            return "goods-" + getClass().getSimpleName() + "-saleout-" + batchId;
        }
    }

	/**
	 * 优惠券队列
	 * 
	 * @author xuhengfei
	 */
	public class CouponQueueRedisCache {

		private RedisTemplate<String, String> redisTemplate;
		private BatchSaleoutRedisCache batchSaleoutRedisCache;

		/**
		 * 优惠券队列
		 * 
		 * @param redis
		 * @param batchSaleoutRedisCache
		 */
		public CouponQueueRedisCache(RedisTemplate<String, String> redis, BatchSaleoutRedisCache batchSaleoutRedisCache) {
			this.redisTemplate = redis;
			this.batchSaleoutRedisCache = batchSaleoutRedisCache;
		}

		/**
		 * 查询该批次的队列长度
		 *
		 * @author xuhengfei
		 * @param goodsBatchId
		 * @return
		 * @since JDK 1.6
		 */
		public long getQueueSize(long goodsBatchId) {
			return redisTemplate.opsForSet().size(getBatchCouponsKey(goodsBatchId));
		}

		/**
		 * 从队列中取出一条
		 *
		 * @author xuhengfei
		 * @param goodsBatchId
		 * @return
		 * @since JDK 1.6
		 */
		public GoodsCouponEntity pop(long goodsBatchId) {
			return JSONObject.parseObject(redisTemplate.opsForSet().pop(getBatchCouponsKey(goodsBatchId)), GoodsCouponEntity.class);
		}

		/**
		 * 清理缓存
		 * 
		 * @param goodsBatchId
		 */
		public void clearCache(Long goodsBatchId) {
			String key = getBatchCouponsKey(goodsBatchId);
			redisTemplate.delete(key);
		}

		/**
		 * 从数据库中查询一批未使用的的券码列表放入队列
		 * 
		 * @param gtype
		 * @param gid
		 * @param goodsBatchId
		 * @param limit
		 */
		public void tryLoadCouponByBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId, int limit) {
			try {
				DBTimeProfile.enter("redis tryLoadCouponByBatchId");
				// 1.获取一个分布式锁防止并发
				boolean flag = redisTemplate.opsForValue().setIfAbsent(getBatchLockKey(goodsBatchId), "1");
				if (!flag) {
					throw new RuntimeGoodsException(ErrorCode.E0202004);
				}
				// 2.给锁设置30秒超时失效防止死锁
				redisTemplate.expire(getBatchLockKey(goodsBatchId), 30, TimeUnit.SECONDS);
				// 3.从数据库查询一批未使用的券码
				List<GoodsCouponEntity> list = GoodsCouponService4Speed.this.loadCouponByBatchId(gtype, gid, goodsBatchId, limit);
				if (list.isEmpty()) {
					// 4.如果查不出数据，标记当前批次已经售罄
					batchSaleoutRedisCache.setSaleOut(goodsBatchId, true);
				} else {
					// 5.将查询的券码数据放入队列
					String[] array = new String[list.size()];
					for (int i = 0; i < list.size(); i++) {
						array[i] = JSONObject.toJSONString(list.get(i));
					}
					redisTemplate.opsForSet().add(getBatchCouponsKey(goodsBatchId), array);
					redisTemplate.expire(getBatchCouponsKey(goodsBatchId), 1, TimeUnit.HOURS);
					// 6.标记当前批次未售罄可用
					batchSaleoutRedisCache.setSaleOut(goodsBatchId, false);
				}
			} finally {
				// 7.操作完后释放锁
				redisTemplate.delete(getBatchLockKey(goodsBatchId));
				DBTimeProfile.release();
			}
		}

		/**
		 * REDIS 存放队列的key
		 * 
		 * @param goodsBatchId
		 * @return
		 */
		private String getBatchCouponsKey(Long goodsBatchId) {
			return RedisKeyTool.getRedisKey(this.getClass(), "batch_coupons", goodsBatchId + "");
		}

		/**
		 * 批次的分布式锁，加锁后从数据库捞取数据
		 * 
		 * @param goodsBatchId
		 * @return
		 */
		private String getBatchLockKey(Long goodsBatchId) {
			return RedisKeyTool.getRedisKey(this.getClass(), "batch_lock", goodsBatchId + "");
		}

	}

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		Boolean ret = super.deleteBatchUnusedCoupons(gtype, gid, batchId, couponIds);
		couponQueueRedisCache.clearCache(batchId);
		return ret;
	}
	
	/**
	 * 获取一个券码
	 * 
	 * @param gtype
	 * @param batch
	 * @param consumerId
	 * @param tryCount
	 * @param bizNum
	 * @return
	 */
	private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, int tryCount, String bizNum) {
		if (tryCount > 3) {
			throw new RuntimeGoodsException(ErrorCode.E0202003);
		}
		try {
			DBTimeProfile.enter("GoodsCouponService4Speed.tryTakeCoupon");
			int counter = tryCount + 1;
			long goodsBatchId = batch.getId();
			long gid = batch.getGid();
			// 1.判断批次是否已售罄
			if (batchSaleoutRedisCache.isSaleOut(batch.getId())) {
				throw new RuntimeGoodsException(ErrorCode.E0202005);
			}
			// 2.查看批次队列是有数据
			long size = couponQueueRedisCache.getQueueSize(goodsBatchId);
			if (size > 0) {
				// 3.从批次队列里取一个券码
				GoodsCouponEntity entity = couponQueueRedisCache.pop(goodsBatchId);
				// 4.更新数据库里券码的状态
				boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId, bizNum);
				if (success) {
					return entity;
				} else {
					// 5.如果数据库更新失败继续从队列取下一个券码
					return tryTakeCoupon(gtype, batch, consumerId, counter, bizNum);
				}
			} else {
				// 6.队列里无数据从数据库里拉取券码放入队列
				couponQueueRedisCache.tryLoadCouponByBatchId(gtype, gid, goodsBatchId, REDISBATCHSIZE);
				// 7.继续从队列取一个券码
				return tryTakeCoupon(gtype, batch, consumerId, counter, bizNum);
			}
		} finally {
			DBTimeProfile.release();
		}
	}

}
