package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.cache.EventHomeCache;
import cn.com.duiba.goods.center.biz.dao.item.AppItemDao;
import cn.com.duiba.goods.center.biz.entity.AppItemEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.wolf.cache.CacheClient;

/**
 * 
 * AppItemServiceImpl
 *
 */
@Service("appItemService")
public class AppItemServiceImpl implements AppItemService {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(AppItemServiceImpl.class);

	@Autowired
	private AppItemDao appItemDao;

	@Autowired
	private ItemService itemService;

	@Autowired
	private GoodsCouponBO goodsCouponBO;

	@Autowired
	private GoodsBatchService goodsBatchService;
	
    @Resource
    private EventHomeCache eventHomeCache;
    
    @Resource
    private CacheClient    cacheClient;
    
    /**
     * 获取AppItem缓存Key
     * @param id
     * @return
     */
    private String getKeyById(Long id) {
        return CacheConstants.KEY_APP_ITEM_BY_ID + id;
    }

	@Override
	public AppItemEntity find(Long id) {
		AppItemEntity entity = appItemDao.find(id);
		if (entity == null) {
			return null;
		}
		if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.APP, entity.getId());
			if (stock != null) {
				entity.setRemaining(Integer.valueOf(stock.toString()));
			}
		}
		return entity;
	}

	@Override
	public List<AppItemEntity> findByIds(List<Long> ids) {
		if(ids == null || ids.isEmpty()){
			return Collections.emptyList();
		}
		List<AppItemEntity> entitys = appItemDao.findByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		for (AppItemEntity entity : entitys) {
			if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.APP);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Integer> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), Integer.valueOf(dto.getStock().toString()));
			}
			for (AppItemEntity entity : entitys) {
				if (entity.isOpTypeAppItem(ItemDto.OpTypeGoodsCoupon)) {
					Integer stock = map.get(entity.getId());
					entity.setRemaining(stock);
				}
			}
		}
		return entitys;
	}

	@Override
	public Long insert(AppItemEntity appItemEntity) {
		appItemDao.insert(appItemEntity);
        eventHomeCache.invalidApp(appItemEntity.getAppId());
		return appItemEntity.getId();
	}

	@Override
	public Boolean update(AppItemEntity appItemEntity) {
		int ret = appItemDao.update(appItemEntity);
		if (ret > 0) {
			eventHomeCache.invalidApp(appItemEntity.getAppId());
			cacheClient.remove(getKeyById(appItemEntity.getId()));
			return true;
		}
		return false;
	}

	@Override
	public Boolean delete(Long appId, Long appItemId) {
		int ret = appItemDao.delete(appId, appItemId);
		if (ret > 0) {
			eventHomeCache.invalidApp(appId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Integer deleteByIds(Long appId, List<Long> appItemIds) {
		int ret = appItemDao.deleteByIds(appId, appItemIds);
		eventHomeCache.invalidAppItem(appItemIds);
		for (Long id : appItemIds) {
			cacheClient.remove(getKeyById(id));
		}
		return ret;
	}

	@Override
	public Boolean updateStatus(Long appId, Long appItemId, String status) {
		int ret = appItemDao.updateStatus(appId, appItemId, status);
		if (ret > 0) {
			eventHomeCache.invalidAppItem(appItemId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Integer updateStatusByIds(Long appId, List<Long> appItemIds, String status) {
		int ret = appItemDao.updateStatusByIds(appId, appItemIds, status);
		eventHomeCache.invalidAppItem(appItemIds);
		for (Long id : appItemIds) {
			cacheClient.remove(getKeyById(id));
		}
		return ret;
	}

	@Override
	public Boolean decrStock(Long appId, Long appItemId) {
		int ret = appItemDao.decrStock(appItemId);
		if (ret > 0) {
	        eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}
	
	@Override
	public void removeCache(Long appItemId) {
		try {
			eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClient.remove(getKeyById(appItemId));
		} catch (Exception e) {
			LOGGER.error("removeCache:appItemId=" + appItemId, e);
		}
	}
	

	@Override
	public Boolean incrStock(Long appItemId) {
		int ret = appItemDao.incrStock(appItemId);
		if (ret > 0) {
	        eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean appendStock(Long appId, Long appItemId, Long number) {
		int ret = appItemDao.appendStock(appItemId, number);
		if (ret > 0) {
	        eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean deductStock(Long appId, Long appItemId, Long number) {
		int ret = appItemDao.deductStock(appItemId, number);
		if (ret > 0) {
	        eventHomeCache.invalidAppItemRemaining(appItemId);
			cacheClient.remove(getKeyById(appItemId));
			return true;
		}
		return false;
	}

	@Override
	public AppItemEntity findByAppIdAndItemId(Long appId, Long itemId) {
		return appItemDao.findByAppIdAndItemId(appId, itemId);
	}

	@Override
	public int updateRemainingAndvalidEndDate(Long appItemId, Long totalStock, Date validEndDate) {
		int ret = appItemDao.updateRemainingAndvalidEndDate(appItemId, totalStock, validEndDate);
		eventHomeCache.invalidAppItem(appItemId);
		cacheClient.remove(getKeyById(appItemId));
		return ret;
	}

	@Override
	public int updateExpiedAppItem(Long itemId) {
		int ret = appItemDao.updateExpiedAppItem(itemId);
		List<Long> ids = appItemDao.findIdByItemId(itemId);
        for (Long id : ids) {
            cacheClient.remove(getKeyById(id));
        }
		return ret;
	}

	@Override
	public int updateExpiedAppItemById(Long appItemId) {
		int ret = appItemDao.updateExpiedAppItemById(appItemId);
		eventHomeCache.invalidAppItem(appItemId);
		cacheClient.remove(getKeyById(appItemId));
		return ret;
	}
	
    @Override
    public List<Long> findAppIdsByInActivityId(Long activityId, Integer type) {
        return appItemDao.findAppIdsByInActivityId(activityId, type);
    }

    @Override
    public List<Long> findAppIdsByActivityId(Long relationId) {
        return appItemDao.findAppIdsByActivityId(relationId);
    }

    @Override
    public List<Long> findAppIdByItemIds(List<Long> itemIds) {
        return appItemDao.findAppIdByItemIds(itemIds);
    }

    @Override
    public List<Long> findAppIdsByIds(List<Long> ids) {
        return appItemDao.findAppIdsByIds(ids);
    }

    @Override
    public int updateStatusByActivityIds(String status, Boolean delete, List<Long> oids) {
        int ret = appItemDao.updateStatusByActivityIds(status,delete, oids);
        List<Long> appItemIds = appItemDao.findIdsByActivityIds(oids);
        eventHomeCache.invalidAppItem(appItemIds);
        for (Long id : appItemIds) {
            cacheClient.remove(getKeyById(id));
        }
        return ret;
    }
}
