package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.AppItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.CouponBatchDao;
import cn.com.duiba.goods.center.biz.dao.item.CouponDao;
import cn.com.duiba.goods.center.biz.entity.CouponBatchEntity;
import cn.com.duiba.goods.center.biz.entity.CouponEntity;
import cn.com.duiba.goods.center.biz.service.item.AppItemService;
import cn.com.duiba.goods.center.biz.service.item.CouponFlowInnerService;
import cn.com.duiba.goods.center.biz.service.item.CouponFlowService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.util.DataSource;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.CouponBatchDO;

/**
 * 优惠券接口实现
 */
@Service
public class CouponFlowServiceImpl implements CouponFlowService {

	@Autowired
	private CouponFlowInnerService couponFlowInnerService;
	@Autowired
	private CouponBatchDao couponBatchDao;
	@Autowired
	private CouponDao couponDao;
	@Autowired
	private ItemService itemService;
	@Autowired
	private AppItemService appItemService;

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public CouponEntity getOneCoupon(ItemKeyDto itemKey) {
		// 批次验证
		couponFlowInnerService.checkCouponBatch(itemKey);
		Long batchId = null;
		AppItemDto appItem = itemKey.getAppItem();
		ItemDto item = itemKey.getItem();
		// 商品加锁
		if (itemKey.isSelfAppItemMode()) {
			batchId = appItem.getBatchId();
		} else if (itemKey.isDuibaAppItemMode() || itemKey.isItemMode()) {
			batchId = item.getBatchId();
		}
		if (batchId == null) {
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
		// 批次加锁
		CouponBatchEntity batch = couponBatchDao.findForUpdate(batchId);
		if (batch == null || batch.getRemaining() <= 0) {
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		} else if (batch.getValidEndDate().before(new Date())) {
			// 切换批次
			couponFlowInnerService.switchBatch(itemKey);
			// 刷新商品信息
			couponFlowInnerService.refreshItemKeyCoupon(itemKey);
			// 批次过期触发刷新直接返回 null，抛异常会回滚刷新
			return null;
		}
		CouponEntity coupon = couponDao.getOneCouponByBatchForUpdate(batch.getId());
		if (coupon != null) {
			// 锁定券码
			int couponRet = couponDao.lockCoupon(coupon.getId());
			if (couponRet < 1) {
				throw new RuntimeGoodsException(ErrorCode.E0202006);
			}
			couponBatchDao.reduceCouponBatchRemaining(batch.getId());
			if (coupon.getItemId() != null) {
				itemService.decrStock(coupon.getItemId());
				itemService.removeCache(coupon.getItemId());
			} else if (coupon.getAppItemId() != null && coupon.getItemId() == null) {
				appItemService.decrStock(itemKey.getAppId(), coupon.getAppItemId());
				appItemService.removeCache(coupon.getAppItemId());
			}
		}
		// 切换批次
		couponFlowInnerService.switchBatch(itemKey);
		// 刷新商品信息
		couponFlowInnerService.refreshItemKeyCoupon(itemKey);
		return coupon;
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void returnOneCoupon(long couponId) {
		CouponEntity coupon = couponDao.find(couponId);
		if (!coupon.getLocked()) {
			return;
		}
		// 批次加锁
		CouponBatchEntity couponBatch = couponBatchDao.findForUpdate(coupon.getBatchId());
		// 优惠券加锁
		int ret = couponDao.unlockCoupon(couponId);
		if (ret < 1) {
			throw new RuntimeGoodsException(ErrorCode.E0102002);
		}
		couponFlowInnerService.refreshBatch(couponBatch);
		if (coupon.getItemId() != null) {
			couponFlowInnerService.switchItemBatch(coupon.getItemId());
			couponFlowInnerService.refreshItemCoupon(coupon.getItemId());
		} else if (coupon.getAppItemId() != null) {
			couponFlowInnerService.switchAppItemBatch(coupon.getAppItemId());
			couponFlowInnerService.refreshAppItemCoupon(coupon.getAppItemId());
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public void deleteCouponBatch(long batchId) {
		// 批次加锁
		CouponBatchEntity batch = couponBatchDao.findForUpdate(batchId);
		if (batch != null && !batch.getActive().equals(CouponBatchDO.ActiveDeleted)) {
			// 删除批次 下券码（未锁定，未使用）
			couponDao.deleteByBatchId(batchId);
			if (batch.getItemId() != null) {
				// 刷新批次
				couponFlowInnerService.refreshBatch(batch);
				// 切换item批次
				couponFlowInnerService.switchItemBatch(batch.getItemId());
				// 刷新item商品信息
				couponFlowInnerService.refreshItemCoupon(batch.getItemId());
			} else if (batch.getAppItemId() != null && batch.getItemId() == null) {
				// 刷新批次
				couponFlowInnerService.refreshBatch(batch);
				// 切换AppItem批次
				couponFlowInnerService.switchAppItemBatch(batch.getAppItemId());
				// 刷新AppItem商品信息
				couponFlowInnerService.refreshAppItemCoupon(batch.getAppItemId());
			}
		}
	}

	@Override
	@Transactional(value = DataSource.CREDITS, propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
	public int deleteCouponCode(long couponId) {
		CouponEntity coupon = couponDao.find(couponId);
		if (coupon.getLocked() || coupon.getUsed()) {
			return 0;
		}
		// 批次加锁
		CouponBatchEntity couponBatch = couponBatchDao.findForUpdate(coupon.getBatchId());
		// 删除券码
		int ret = couponDao.deleteOneCoupon(couponId);
		if (ret < 1) {
			throw new RuntimeGoodsException(ErrorCode.E0102002);
		}
		if (coupon.getItemId() != null) {
			// 刷新批次
			couponFlowInnerService.refreshBatch(couponBatch);
			// 切换item批次
			couponFlowInnerService.switchItemBatch(coupon.getItemId());
			// 刷新item商品信息
			couponFlowInnerService.refreshItemCoupon(coupon.getItemId());
		} else if (coupon.getAppItemId() != null && coupon.getItemId() == null) {
			// 刷新批次
			couponFlowInnerService.refreshBatch(couponBatch);
			// 切换AppItem批次
			couponFlowInnerService.switchAppItemBatch(coupon.getAppItemId());
			// 刷新AppItem商品信息
			couponFlowInnerService.refreshAppItemCoupon(coupon.getAppItemId());
		}
		return ret;
	}

	@Override
	public void completeCoupon(Long couponId, Long orderId) {
		couponDao.useCoupon(couponId, orderId);
	}

}
