package cn.com.duiba.goods.center.biz.service.item.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.collections.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemDto;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.cache.EventHomeCache;
import cn.com.duiba.goods.center.biz.dao.item.AppItemMaskDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemAppSpecifyDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemAutoRecommendDao;
import cn.com.duiba.goods.center.biz.dao.item.ItemDao;
import cn.com.duiba.goods.center.biz.entity.ItemEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.item.ItemService;
import cn.com.duiba.goods.center.biz.service.item.PreStockService;
import cn.com.duiba.goods.center.biz.util.CacheConstants;
import cn.com.duiba.wolf.cache.CacheClient;

import com.google.common.base.Preconditions;

/**
 * itemservice Created by gyf . 16/10/25 .
 */
@Service("itemService")
public class ItemServiceImpl implements ItemService {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(ItemServiceImpl.class);

	@Autowired
	private ItemDao itemDao;
	@Autowired
	private ItemAppSpecifyDao itemAppSpecifyDao;
	@Autowired
	private GoodsCouponBO goodsCouponBO;
	@Autowired
	private GoodsBatchService goodsBatchService;
	@Autowired
	private PreStockService preStockService;
	@Autowired
	private ItemAutoRecommendDao itemAutoRecommendDao;
	@Autowired
	private AppItemMaskDao appItemMaskDao;
    @Resource
    private EventHomeCache eventHomeCache;
    @Resource
    private CacheClient cacheClient;
    
    /**
     * 获取item缓存ID
     * @param id
     * @return
     */
    private String getCacheKeyById(Long id){
        return CacheConstants.KEY_ITEM_BY_ID + id;
    }

	@Override
	public ItemEntity findCorrectStockItem(Long id) {
		Preconditions.checkNotNull(id, "id 不能为 null");
		ItemEntity item = itemDao.find(id);
		if (item == null) {
			return null;
		}
		if (item.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.DUIBA, item.getId());
			item.setRemaining(Integer.valueOf(stock.toString()));
		}
		return item;
	}

	@Override
	public ItemEntity find(Long id) {
		ItemEntity entity = itemDao.find(id);
		if (entity == null) {
			return null;
		}
		if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
			Long stock = goodsBatchService.getSumBatchStockBatch(GoodsTypeEnum.DUIBA, entity.getId());
			entity.setRemaining(Integer.valueOf(stock.toString()));
		}
		return entity;
	}

	@Override
	public List<ItemEntity> findByIds(List<Long> ids) {
		List<ItemEntity> entitys = itemDao.findAllByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		for (ItemEntity entity : entitys) {
			if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.DUIBA);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Long> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), dto.getStock());
			}
			for (ItemEntity entity : entitys) {
				Long stock = map.get(entity.getId());
				if (stock != null && entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
					entity.setRemaining(Integer.valueOf(stock.toString()));
				}
			}
		}
		return entitys;
	}
	
	@Override
	public Map<Long, ItemEntity> findByIdsToMap(List<Long> ids) {
		List<ItemEntity> entitys = itemDao.findAllByIds(ids);
		List<GoodsStockDto> stocks = new ArrayList<>();
		Map<Long, ItemEntity> maps = new HashMap<>();
		for (ItemEntity entity : entitys) {
			maps.put(entity.getId(), entity);
			if (entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
				GoodsStockDto stock = new GoodsStockDto();
				stock.setGid(entity.getId());
				stock.setGtype(GoodsTypeEnum.DUIBA);
				stocks.add(stock);
			}
		}
		if (!stocks.isEmpty()) {
			List<GoodsStockDto> gst = goodsCouponBO.findGoodsStockByBatch(stocks);
			Map<Long, Long> map = new HashMap<>();
			for (GoodsStockDto dto : gst) {
				map.put(dto.getGid(), dto.getStock());
			}
			for (ItemEntity entity : entitys) {
				Long stock = map.get(entity.getId());
				if (stock != null && entity.isOpTypeItem(ItemDto.OpTypeGoodsCoupon)) {
					entity.setRemaining(Integer.valueOf(stock.toString()));
				}
			}
		}
		return maps;
	}

	@Override
	public Long insert(ItemEntity itemEntity) {
		itemDao.insert(itemEntity);
        eventHomeCache.invalidItem(itemEntity.getId());
		return itemEntity.getId();
	}

	@Override
	public Boolean update(ItemEntity itemEntity) {
		int ret = itemDao.update(itemEntity);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemEntity.getId());
			cacheClient.remove(getCacheKeyById(itemEntity.getId()));
			return true;
		}
		return false;
	}

	@Override
	public Boolean delete(Long itemId) {
		int ret = itemDao.delete(itemId);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean updateEnable(Long itemId, Boolean enable) {
		int ret = itemDao.updateEnable(itemId, enable);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean decrStock(Long itemId) {
		int ret = itemDao.decrStock(itemId);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean incrStock(Long itemId) {
		int ret = itemDao.incrStock(itemId);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean appendStock(Long itemId, Long number) {
		int ret = itemDao.appendStock(itemId, number);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@Override
	public Boolean deductStock(Long itemId, Long number) {
		int ret = itemDao.deductStock(itemId, number);
		if (ret > 0) {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
			return true;
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<ItemEntity> findAutoRecommend(Long appId) {
		// 自动推荐&非定向
		List<Long> allList = itemAutoRecommendDao.findItemAutoRecommendNoSpecify(true, true);
		// 自动推荐&定向关系存在
		List<Long> spcefyList = itemAutoRecommendDao.findItemSpecify(appId);
		// app 黑名单
		List<Long> appMaskingList = appItemMaskDao.findItemAppMasking(appId);
		// tag黑名单
		List<Long> tagsMaskingList = itemAutoRecommendDao.findItemAppTagsMasking(appId);
		// 开发者库黑名单
		List<Long> inAppList = itemAutoRecommendDao.findItemInApp(appId, false, false);

		List<Long> itemIdList = ListUtils.sum(allList, spcefyList);
		itemIdList = ListUtils.subtract(itemIdList, appMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, tagsMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, inAppList);

		// 避免自动推荐过多
		if (itemIdList != null && itemIdList.size() > 20) {
			itemIdList = itemIdList.subList(0, 20);
		}
		List<ItemEntity> entitys = new ArrayList<>();
		if (itemIdList != null && !itemIdList.isEmpty()) {
			entitys = findByIds(itemIdList);
		}
		return entitys;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<ItemEntity> findAutoRecommendMap(Long appId, Map<String, Object> query) {
		query.put("appId", appId);
		query.put("autoRecommend", true);
		query.put("enable", true);
		// 自动推荐&非定向
		List<Long> allList = itemAutoRecommendDao.findItemAutoRecommendNoSpecifyMap(query);
		// 自动推荐&定向关系存在
		List<Long> spcefyList = itemAutoRecommendDao.findItemSpecifyMap(query);
		// APP 黑名单
		List<Long> appMaskingList = appItemMaskDao.findItemAppMasking(appId);
		// tag黑名单
		List<Long> tagsMaskingList = itemAutoRecommendDao.findItemAppTagsMasking(appId);
		// 开发者库黑名单
		List<Long> inAppList = itemAutoRecommendDao.findItemInApp(appId, false, false);

		List<Long> itemIdList = ListUtils.sum(allList, spcefyList);
		itemIdList = ListUtils.subtract(itemIdList, appMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, tagsMaskingList);
		itemIdList = ListUtils.subtract(itemIdList, inAppList);
		// 避免自动推荐过多
		if (itemIdList != null && itemIdList.size() > 20) {
			itemIdList = itemIdList.subList(0, 20);
		}
		List<ItemEntity> entitys = new ArrayList<>();
		if (itemIdList != null && !itemIdList.isEmpty()) {
			entitys = findByIds(itemIdList);
		}
		return entitys;
	}

	@Override
	public Boolean isRechargeGoods(ItemEntity entity) {
		if (entity == null) {
			return false;
		}
		if (ItemDto.TypeAlipay.equals(entity.getType())) {
			return true;
		}
		if (ItemDto.TypeAlipayCode.equals(entity.getType())) {
			return true;
		}
		if (ItemDto.TypeAlipayFast.equals(entity.getType())) {
			return true;
		}
		if (ItemDto.TypePhonebill.equals(entity.getType())) {
			return true;
		}
		if (ItemDto.TypePhoneflow.equals(entity.getType())) {
			return true;
		}
		if (ItemDto.TypeQB.equals(entity.getType())) {
			return true;
		}
		return false;
	}

	@Override
	public Long findSpecifyRemaining(Long appId, Long itemId) {
		return itemAppSpecifyDao.findSpecifyRemaining(itemId, appId);
	}

	@Override
	public int updateRemainingAndvalidEndDate(Long itemId, Long totalStock, Date validEndDate) {
		int ret = itemDao.updateRemainingAndvalidEndDate(itemId, totalStock, validEndDate);
		eventHomeCache.invalidItem(itemId);
        cacheClient.remove(getCacheKeyById(itemId));
        return ret;
	}

    @Override
    public List<ItemDto> findHomeItems(List<Long> itemIds) {
        return itemDao.findHomeItems(itemIds);
    }

	@Override
	public void removeCache(Long itemId) {
		try {
			eventHomeCache.invalidItem(itemId);
			cacheClient.remove(getCacheKeyById(itemId));
		} catch (Exception e) {
			LOGGER.error("removeCache:itemId=" + itemId, e);
		}
	}

}
