package cn.com.duiba.goods.center.biz.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.util.RedisKeyFactory;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.alibaba.fastjson.JSONObject;

/**
 * 商品批次缓存服务实现类
 * 批次变更时清除缓存
 */
@Service("goodsBatchRedisService")
public class GoodsBatchRedisService extends GoodsBatchServiceImpl implements GoodsBatchService {

	@Override
	public boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType) {
		boolean ret = super.updateGoodsType(gtype, gid, dstType);
		if (ret) {
			clearBatchsCache(gtype, gid);
		}
		return ret;
	}

	@Override
	public Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
		boolean ret = super.physicalDeleteBatch(gtype, gid, batchId);
		if (ret) {
			clearBatchsCache(gtype, gid);
		}
		return ret;
	}

	@Override
	public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid) {
		List<GoodsBatchEntity> batchs = findNotDeletedBatchs(gtype, gid);
		List<GoodsBatchEntity> ret = new ArrayList<>();
		Date today = new Date();
		for (GoodsBatchEntity e : batchs) {
			if (!e.getEndDay().before(today)) {
				ret.add(e);
			}
		}
		return ret;
	}

	@Override
	public List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid) {
		try {
			DBTimeProfile.enter("findNotDeletedBatchs");
			String value = getBatchsCache(gtype, gid);
			if (StringUtils.isEmpty(value)) {
				List<GoodsBatchEntity> batchs = super.findNotDeletedBatchs(gtype, gid);
				value = JSONObject.toJSONString(batchs);
				setBatchsCache(gtype, gid, value);
			}
			return JSONObject.parseArray(value, GoodsBatchEntity.class);
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId) {
		Long ret = super.createNormalBatch(gtype, gid, start, end, stockId);
		clearBatchsCache(gtype, gid);
		return ret;
	}

	@Override
	public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId) {
		Long ret = super.createLinkBatch(gtype, gid, start, end, stockId);
		clearBatchsCache(gtype, gid);
		return ret;
	}

	@Override
	public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId) {
		Long ret = super.createRepeatBatch(gtype, gid, start, end, stockId);
		clearBatchsCache(gtype, gid);
		return ret;
	}

	@Override
	public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
		Boolean ret = super.deleteBatch(gtype, gid, batchId);
		clearBatchsCache(gtype, gid);
		return ret;
	}

	@Override
	public void markBatchStatusNotUsed(long batchId) {
		super.markBatchStatusNotUsed(batchId);
		GoodsBatchEntity batch = super.find(batchId);
		clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
	}

	@Override
	public void markBatchStatusUsed(long batchId) {
		super.markBatchStatusUsed(batchId);
		GoodsBatchEntity batch = super.find(batchId);
		clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
	}

	@Override
	public void markBatchStatusUsed(GoodsTypeEnum gtype, long gid, long batchId) {
		try {
			DBTimeProfile.enter("markBatchStatusUsed");
			super.markBatchStatusUsed(batchId);
			clearBatchsCache(gtype, gid);
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public void markBatchStatusUsing(long batchId) {
		super.markBatchStatusUsing(batchId);
		GoodsBatchEntity batch = super.find(batchId);
		clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
	}

	@Override
	public void markBatchStatusUsing(GoodsTypeEnum gtype, long gid, long batchId) {
		try {
			DBTimeProfile.enter("markBatchStatusUsing");
			super.markBatchStatusUsing(batchId);
			clearBatchsCache(gtype, gid);
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay) {
		Boolean result = super.updateValidDate(gtype, gid, batchId, startDay, endDay);
		clearBatchsCache(gtype, gid);
		return result;
	}

	@Override
	public GoodsBatchEntity refreshGoodsBatchs(GoodsTypeEnum gtype, long gid) {
		// 1.查询商品批次（不包含已删除和已过期批次）
		List<GoodsBatchEntity> normals = findNormalBatchs(gtype, gid);
		// 2.判断是否有可切换批次
		boolean saleout = isSaleout(normals);
		if (saleout) {
			return null;
		}
		// 3.获取一个切换批次锁，防止并发切换批次，并发超过锁等待忽略
		getBatchSwitchLock(gtype, gid);
		// 4.切换批次
		try {
			return super.refreshGoodsBatchs(gtype, gid);
		} catch (Exception e) {
			throw e;
		} finally {
			releaseBatchSwitchLock(gtype, gid);
		}
	}

	/**
	 * 简单判断是否有可用批次
	 * 
	 * @param batchs
	 * @return
	 */
	private boolean isSaleout(List<GoodsBatchEntity> batchs) {
		boolean saleout = true;
		for (GoodsBatchEntity e : batchs) {
			boolean available = e.getStatus() == GoodsBatchEntity.StatusNotUse || e.getStatus() == GoodsBatchEntity.StatusUsing;
			if (available) {
				saleout = false;
				break;
			}
		}
		return saleout;
	}
	
	/**
	 * 获取批次切换分布式锁
	 * 
	 * @param gtype
	 * @param gid
	 * @return
	 */
	private boolean getBatchSwitchLock(GoodsTypeEnum gtype, long gid) {
		String key = getBatchSwitchLockKey(gtype, gid);
		return redisCacheService.tryGetLock(key, 5);
	}

	/**
	 * 释放批次切换分布式锁
	 * 
	 * @param gtype
	 * @param gid
	 */
	private void releaseBatchSwitchLock(GoodsTypeEnum gtype, long gid) {
		String key = getBatchSwitchLockKey(gtype, gid);
		redisCacheService.releaseLock(key);
	}

	/**
	 * 根据商品ID查询商品批次缓存
	 * 
	 * @param gtype
	 * @param gid
	 * @return
	 */
	private String getBatchsCache(GoodsTypeEnum gtype, long gid) {
		String key = getRedisBatchsKey(gtype, gid);
		String val = redisCacheService.get(key);
		if (StringUtils.isEmpty(val)) {
			return null;
		}
		return val;
	}

	/**
	 * 根据商品ID删除批次缓存
	 * 
	 * @param gtype
	 * @param gid
	 */
	private void clearBatchsCache(GoodsTypeEnum gtype, long gid) {
		String key = getRedisBatchsKey(gtype, gid);
		redisCacheService.delete(key);
	}

	/**
	 * 设置缓存
	 * 
	 * @param gtype
	 * @param gid
	 * @param batchs
	 */
	private void setBatchsCache(GoodsTypeEnum gtype, long gid, String value) {
		String key = getRedisBatchsKey(gtype, gid);
		redisCacheService.set(key, value, 3600);
	}

	/**
	 * 商品批次缓存KEY
	 * 
	 * @param gtype
	 * @param gid
	 * @return
	 */
	private String getRedisBatchsKey(GoodsTypeEnum gtype, long gid) {
		return RedisKeyFactory.K201 + String.valueOf(gtype.getGtype()) + "_" + gid;
	}

	/**
	 * 获取批次切换分布式锁KEY
	 * 
	 * @param gtype
	 * @param gid
	 * @return
	 */
	private String getBatchSwitchLockKey(GoodsTypeEnum gtype, long gid) {
		return RedisKeyFactory.K202 + String.valueOf(gtype.getGtype()) + "_" + gid;
	}

}
