package cn.com.duiba.goods.center.biz.service.impl;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RedisKeyTool;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.alibaba.fastjson.JSONObject;

/**
 * 发放券码实现类
 */
@Service("goodsCouponService4Speed")
public class GoodsCouponService4Speed extends GoodsCouponServiceImpl implements GoodsCouponService {

	private static final int QUEUE_SIZE = 10000;

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
		boolean ret = super.deleteBatchUnusedCoupons(batch);
		clearQueueCache(batch.getId());
		return ret;
	}

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		Boolean ret = super.deleteBatchUnusedCoupons(gtype, gid, batchId, couponIds);
		clearQueueCache(batchId);
		return ret;
	}

	@Override
	public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
		boolean ret = super.deleteUnusedCoupon(gtype, gid, couponId, batchId);
		clearQueueCache(batchId);
		return ret;
	}

	@Override
	public Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link) {
		Boolean ret = super.updateLinkBatch(gtype, gid, batchId, link);
		clearLinkOrRepeat(batchId);
		return ret;
	}

	@Override
	public void importLinkCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String link) {
		super.importLinkCoupon(gtype, gid, goodsBatchId, goodsCouponId, link);
		clearLinkOrRepeat(goodsBatchId);
	}

	@Override
	public void importRepeatCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String code, String password) {
		super.importRepeatCoupon(gtype, gid, goodsBatchId, goodsCouponId, code, password);
		clearLinkOrRepeat(goodsBatchId);
	}

	@Override
	public Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password) {
		Boolean ret = super.updateRepeatBatch(gtype, gid, batchId, code, password);
		clearLinkOrRepeat(batchId);
		return ret;
	}

	@Override
	public Integer importNormalCoupons(GoodsBatchEntity batch, List<CouponFormat> coupons) {
		Integer ret = super.importNormalCoupons(batch, coupons);
		clearQueueCache(batch.getId());
		return ret;
	}

	@Override
	public boolean deleteGoodsCoupon(GoodsTypeEnum gtype, long gid, long batchId) {
		boolean ret = super.deleteGoodsCoupon(gtype, gid, batchId);
		clearQueueCache(batchId);
		clearLinkOrRepeat(batchId);
		return ret;
	}

	@Override
	public int deleteGoodsCouponByIds(long gid, Long batchId, List<Long> ids) {
		int ret = super.deleteGoodsCouponByIds(gid, batchId, ids);
		clearQueueCache(batchId);
		return ret;
	}

	@Override
	public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		Boolean consumeSuccess = false;
		try {
			DBTimeProfile.enter("takeNormalCoupon");
			// 1.库存中心扣库存
			consumeSuccess = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSuccess) {
				return null;
			}
			// 2.取一个券码
			return tryTakeCoupon(gtype, batch, consumerId, bizNum, 1);
		} catch (Exception e) {
			// 3.出现异常如果库存中心扣库存成功返还库存
			if (consumeSuccess) {
				rollbackStocks(gtype, bizNum);
			}
			throw e;
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 从队列获取一个券码<br/>
	 * 如果队列没有从数据库取一批券码放到队列，重新充队列取一张券码
	 * 
	 * @param gtype
	 * @param batch
	 * @param consumerId
	 * @param tryCount
	 * @param bizNum
	 * @return
	 */
	private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, String bizNum, int count) {
		if (count > 3) {
			throw new RuntimeGoodsException(ErrorCode.E0404002);
		}
		int counter = count + 1;
		try {
			long goodsBatchId = batch.getId();
			long gid = batch.getGid();
			// 1.从批次队列里取一个券码
			GoodsCouponEntity entity = pop(goodsBatchId);
			// 2.队列里无数据从数据库里查询一批券码放入队列
			if (entity == null) {
				tryLoadCouponByBatchId(gtype, gid, goodsBatchId, QUEUE_SIZE);
				// 3.重新取一张券码
				throw new RuntimeGoodsException(ErrorCode.E0404012);
			}
			// 5.更新数据库里券码的状态
			boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId, bizNum);
			if (!success) {
				throw new RuntimeGoodsException(ErrorCode.E0404012);
			}
			return entity;
		} catch (RuntimeGoodsException e) {
			// 4.如果没有取到券新从队列里取，减少失败率
			boolean retry = ErrorCode.E0404012 == e.getErrorCode() || ErrorCode.E0202004 == e.getErrorCode();
			if (retry) {
				return tryTakeCoupon(gtype, batch, consumerId, bizNum, counter);
			}
			throw e;
		}
	}

	/**
	 * 从数据库中查询一批未使用的的券码列表放入队列<br/>
	 * 1.先获取一个分布式锁（未获取到等待30ms重试）<br/>
	 * 2.从数据库查询数据放到队列
	 * 
	 * @param gtype
	 * @param gid
	 * @param goodsBatchId
	 * @param limit
	 */
	public void tryLoadCouponByBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId, int limit) {
		queueCheck(goodsBatchId);
		// 1.获取一个分布式锁防止并发
		boolean lock = redisCacheService.tryGetLock(getQueueLockKey(goodsBatchId), 30);
		if (!lock) {
			throw new RuntimeGoodsException(ErrorCode.E0202004);
		}
		queueCheck(goodsBatchId);
		try {
			DBTimeProfile.enter("selectCouponAddQueue");
			// 2.从数据库 查询一批未使用的券码
			List<GoodsCouponEntity> list = loadCouponByBatchId(gtype, gid, goodsBatchId, limit);
			if (list.isEmpty()) {
				// 4.如果没有未使用的券码，标记当前批次已经售罄
				setSaleOut(goodsBatchId, true);
				goodsBatchService.markBatchStatusUsed(gtype, gid, goodsBatchId);
				throw new RuntimeGoodsException(ErrorCode.E0202005);
			} else {
				// 5.将查询的券码数据放入队列
				String[] array = new String[list.size()];
				for (int i = 0; i < list.size(); i++) {
					array[i] = JSONObject.toJSONString(list.get(i));
				}
				redisCacheService.addQueue(getCouponQueueKey(goodsBatchId), array, 3600);
				// 6.标记当前批次未售罄可用
				setSaleOut(goodsBatchId, false);
			}
		} finally {
			// 7.操作完后释放锁
			redisCacheService.releaseLock(getQueueLockKey(goodsBatchId));
			DBTimeProfile.release();
		}
	}
	
	/**
	 * 批次队列检测，防止并发多个线程锁排队前后重复处理
	 * 
	 * @param goodsBatchId
	 */
	private void queueCheck(Long goodsBatchId) {
		if (getQueueSize(goodsBatchId) > 0) {
			throw new RuntimeGoodsException(ErrorCode.E0404012);
		}
		if (isSaleOut(goodsBatchId)) {
			throw new RuntimeGoodsException(ErrorCode.E0202005);
		}
	}
	
	/**
	 * 批次券码是否售罄，如果缓存中没有数据，默认返回还没有售罄
	 *
	 * @param batchId
	 * @return
	 * @since JDK 1.6
	 */
	private boolean isSaleOut(long batchId) {
		String value = redisCacheService.get(getSaleOutKey(batchId));
		if (StringUtils.isEmpty(value)) {
			return Boolean.valueOf(value);
		}
		return false;
	}

	/**
	 * 设置一个批次是否售罄的缓存值，1分钟后失效删除
	 *
	 * @param batchId
	 * @param saleout
	 */
	private void setSaleOut(long batchId, boolean saleout) {
		redisCacheService.set(getSaleOutKey(batchId), String.valueOf(saleout), 60);
	}

	/**
	 * 是否售罄KEY
	 * 
	 * @param batchId
	 * @return
	 */
	private String getSaleOutKey(long batchId) {
		return "goods-" + getClass().getSimpleName() + "-saleout-" + batchId;
	}

	/**
	 * 查询该批次的队列长度
	 *
	 * @param goodsBatchId
	 * @return
	 */
	public long getQueueSize(long goodsBatchId) {
		return redisCacheService.getQueueSize(getCouponQueueKey(goodsBatchId));
	}

	/**
	 * 从队列中取出一条
	 *
	 * @param goodsBatchId
	 * @return
	 */
	public GoodsCouponEntity pop(long goodsBatchId) {
		return JSONObject.parseObject(redisCacheService.pop(getCouponQueueKey(goodsBatchId)), GoodsCouponEntity.class);
	}

	/**
	 * 清理批次的队列
	 * 
	 * @param goodsBatchId
	 */
	public void clearQueueCache(Long goodsBatchId) {
		String key = getCouponQueueKey(goodsBatchId);
		redisCacheService.delete(key);
	}

	/**
	 * REDIS 存放队列的key
	 * 
	 * @param goodsBatchId
	 * @return
	 */
	private String getCouponQueueKey(Long goodsBatchId) {
		return RedisKeyTool.getRedisKey(this.getClass(), "batch_coupons", goodsBatchId + "");
	}

	/**
	 * 批次的分布式锁，加锁后从数据库捞取数据
	 * 
	 * @param goodsBatchId
	 * @return
	 */
	private String getQueueLockKey(Long goodsBatchId) {
		return RedisKeyTool.getRedisKey(this.getClass(), "batch_lock", goodsBatchId + "");
	}

}
