package cn.com.duiba.goods.center.biz.service.item.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.ItemKeyUtils;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.dao.item.CouponBatchDao;
import cn.com.duiba.goods.center.biz.dao.item.CouponDao;
import cn.com.duiba.goods.center.biz.entity.CouponBatchEntity;
import cn.com.duiba.goods.center.biz.entity.CouponEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.item.CouponFlowService;
import cn.com.duiba.goods.center.biz.service.item.ItemCouponGoodsService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.ItemDO;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 商品GTYPE 为DUIBA和APP类的优惠券相关服务
 */
@Service
public class ItemCouponGoodsServiceImpl implements ItemCouponGoodsService {
	
    private static final Logger LOGGER = LoggerFactory.getLogger(ItemCouponGoodsServiceImpl.class);

	@Autowired
	private GoodsCouponBO goodsCouponBO;
	@Autowired
	private GoodsBatchService goodsBatchService;
	@Autowired
	private CouponFlowService couponFlowService;
	@Autowired
	private CouponDao couponDao;
	@Autowired
	private CouponBatchDao couponBatchDao;

	@Override
	public GoodsCouponDto takeCoupon(ItemKeyDto itemKeyDto, Long consumerId, String bizNum) {
		// 兼容老的券码发放逻辑
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			return takeOldCoupon(itemKeyDto);
		}
		GoodsTypeEnum gtype;
		Long gid;
		if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
			gtype = GoodsTypeEnum.DUIBA;
			gid = itemKeyDto.getItem().getId();
		} else if (itemKeyDto.isSelfAppItemMode()) {
			gtype = GoodsTypeEnum.APP;
			gid = itemKeyDto.getAppItem().getId();
		} else {
			throw new RuntimeGoodsException(ErrorCode.E0203013);
		}
		return takeGoodsCoupon(gtype, gid, consumerId, itemKeyDto.getAppId(), bizNum);
	}

	@Override
	public Boolean completeCoupon(ItemKeyDto itemKeyDto, Long couponId, long orderId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			couponFlowService.completeCoupon(couponId, orderId);
		} else {
			// 新版商品中心优惠券
			goodsCouponBO.completeCoupon(couponId, orderId);
		}
		return true;
	}

	@Override
	public Boolean rollbackCoupon(ItemKeyDto itemKeyDto, Long couponId, String bizNum) {
		boolean ret = true;
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			couponFlowService.returnOneCoupon(couponId);
		} else {
			// 发新版商品中心优惠券
			ret = goodsCouponBO.rollbackCoupon(couponId, bizNum);
		}
		return ret;
	}

	@Override
	public GoodsCouponDto findCoupon(ItemKeyDto itemKeyDto, Long couponId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			// 兼容老的券码逻辑
			CouponEntity ce = couponDao.find(couponId);
			GoodsCouponDto dto = convert(itemKeyDto, ce);
			CouponBatchEntity cbe = couponBatchDao.find(ce.getBatchId());
			dto.setStartDay(cbe.getValidStartDate());
			dto.setOverDue(cbe.getValidEndDate());
			return dto;
		}
		GoodsCouponEntity entity = goodsCouponBO.findCoupon(couponId);
		GoodsCouponDto dto = BeanUtils.copy(entity, GoodsCouponDto.class);
		GoodsBatchEntity batch = goodsCouponBO.findBatch(entity.getGoodsBatchId());
		dto.setStartDay(batch.getStartDay());
		dto.setOverDue(batch.getEndDay());
		return dto;
	}
	
	@Override
	public GoodsBatchDto findCurrentBatch(ItemKeyDto itemKeyDto) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			CouponBatchEntity entity = couponBatchDao.find(getOldBatchId(itemKeyDto));
			return convertBatch(itemKeyDto, entity);
		} 
		GoodsBatchEntity entity = goodsCouponBO.findUsingBatch(getGtype(itemKeyDto), getGid(itemKeyDto));
		return BeanUtils.copy(entity, GoodsBatchDto.class);
	}

	@Override
	public GoodsBatchDto findCouponBatch(ItemKeyDto itemKeyDto, Long batchId) {
		if (!ItemKeyUtils.isGoodsCoupon(itemKeyDto)) {
			CouponBatchEntity entity = couponBatchDao.find(batchId);
			return convertBatch(itemKeyDto, entity);
		}
		GoodsBatchEntity entity = goodsCouponBO.findBatch(batchId);
		return BeanUtils.copy(entity, GoodsBatchDto.class);
	}
	
	/**
	 * 获取老的商品的批次ID
	 * @param itemKey
	 * @return
	 */
	private Long getOldBatchId(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return itemKey.getItem().getBatchId();
		}
		if (itemKey.isSelfAppItemMode()) {
			return itemKey.getAppItem().getBatchId();
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}
	
	/**
	 * ItemKey获取商品Gtype
	 * @param itemKey
	 * @return
	 */
	private GoodsTypeEnum getGtype(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return GoodsTypeEnum.DUIBA;
		}
		if (itemKey.isSelfAppItemMode()) {
			return GoodsTypeEnum.APP;
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}
	
	/**
	 * ItemKey获取商品Gtype
	 * @param itemKey
	 * @return
	 */
	private Long getGid(ItemKeyDto itemKey) {
		if (itemKey.isItemMode() || itemKey.isDuibaAppItemMode()) {
			return itemKey.getItem().getId();
		}
		if (itemKey.isSelfAppItemMode()) {
			return itemKey.getAppItem().getId();
		}
		throw new RuntimeGoodsException(ErrorCode.E0202008);
	}

	/**
	 * 老的单表发券逻辑
	 * @param itemKeyDto
	 * @return
	 */
	private GoodsCouponDto takeOldCoupon(ItemKeyDto itemKeyDto) {
		Long appItemId = itemKeyDto.isSelfAppItemMode() ? itemKeyDto.getAppItem().getId() : null;
		Long itemId = itemKeyDto.getItem() != null ? itemKeyDto.getItem().getId() : null;
		LOGGER.warn("take oldCoupon appId:" + itemKeyDto.getAppId() + " appItemId:" + appItemId + " itemId:" + itemId);
		CouponEntity ce = couponFlowService.getOneCoupon(itemKeyDto);
		return convert(itemKeyDto, ce);
	}

	/**
	 * 新版发券逻辑
	 * @param gtype
	 * @param gid
	 * @param consumerId
	 * @param bizNum
	 * @return
	 */
	private GoodsCouponDto takeGoodsCoupon(GoodsTypeEnum gtype, Long gid, Long consumerId, Long appId, String bizNum) {
		GoodsCouponDto dto = goodsCouponBO.takeCoupon(gtype, gid, consumerId, appId, bizNum);
		if (dto == null) {
			throw new RuntimeGoodsException(ErrorCode.E0202006);
		}
		return dto;
	}
	
	/**
	 * 是否是链接券
	 * 
	 * @param itemKeyDto
	 * @return
	 */
	private Integer getCouponType(ItemKeyDto itemKeyDto) {
		Integer linkType = null;
		if (itemKeyDto.isDuibaAppItemMode() || itemKeyDto.isItemMode()) {
			linkType = itemKeyDto.getItem().getSubType();
		} else if (itemKeyDto.isSelfAppItemMode()) {
			linkType = itemKeyDto.getAppItem().getSubType();
		}
		return linkType;
	}
	
	/**
	 * 老的优惠券封装到新的优惠券
	 * @param itemKey
	 * @param ce
	 * @return
	 */
	private GoodsCouponDto convert(ItemKeyDto itemKey, CouponEntity ce) {
		if (ce == null) {
			return null;
		}
		GoodsCouponDto dto = new GoodsCouponDto();
		Long gid = ce.getItemId() != null ? ce.getItemId() : ce.getAppItemId();
		Integer gtype = ce.getItemId() != null ? GoodsTypeEnum.DUIBA.getGtype() : GoodsTypeEnum.APP.getGtype();
		dto.setGid(gid);
		dto.setGtype(gtype);
		dto.setGmtCreate(ce.getGmtCreate());
		dto.setGmtModified(ce.getGmtModified());
		dto.setBatchId(ce.getBatchId());
		dto.setGoodsCouponId(ce.getId());
		dto.setGoodsBatchId(ce.getBatchId());
		if (ItemDO.SubTypeLink == getCouponType(itemKey)) {
			dto.setLink(ce.getCode());
			dto.setPassword(ce.getCode());
		} else {
			dto.setCode(ce.getCode());
			dto.setPassword(ce.getPassword());
		}
		return dto;
	}
	
	/**
	 * 老的批次封装到新的批次
	 * @param itemKey
	 * @param cbe
	 * @return
	 */
	private GoodsBatchDto convertBatch(ItemKeyDto itemKey, CouponBatchEntity cbe) {
		if (cbe == null) {
			return null;
		}
		GoodsBatchDto dto = new GoodsBatchDto();
		dto.setId(cbe.getId());
		dto.setGid(getGid(itemKey));
		dto.setGtype(getGtype(itemKey).getGtype());
		dto.setBatchType(getCouponType(itemKey));
		dto.setStartDay(cbe.getValidStartDate());
		dto.setEndDay(cbe.getValidEndDate());
		dto.setGmtCreate(cbe.getGmtCreate());
		dto.setGmtModified(cbe.getGmtModified());
		return dto;
	}

}
