/**
 * Project Name:goods-center-biz
 * File Name:GoodsBatchCouponBackendBOImpl.java
 * Package Name:cn.com.duiba.goods.center.biz.bo.impl
 * Date:2016年6月5日下午1:25:44
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.bo.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import net.sf.cglib.beans.BeanCopier;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsBatchImportLogDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.bo.GoodsBatchCouponBackendBO;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchImportLogEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchImportLogService;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.PlatformCouponGoodsService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.IdSuffixTool;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.goods.center.common.UnicodeReader;
import cn.com.duiba.idmaker.service.api.remoteservice.RemoteIDMakerBackendService;
import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;

import com.alibaba.ttl.threadpool.TtlExecutors;

/**
 * ClassName:GoodsBatchCouponBackendBOImpl <br/>
 * Date:     2016年6月5日 下午1:25:44 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("goodsBatchCouponBackendBO")
public class GoodsBatchCouponBackendBOImpl implements GoodsBatchCouponBackendBO{
    
    private static final Logger log=LoggerFactory.getLogger(GoodsBatchCouponBackendBOImpl.class);
    
    public static final int COUPON_MAX_LENGTH = 200;
	private ExecutorService es = TtlExecutors.getTtlExecutorService(Executors.newCachedThreadPool());
    private AtomicInteger concurrent=new AtomicInteger();
    private static BeanCopier goodsBatchCopier=BeanCopier.create(GoodsBatchEntity.class, GoodsBatchDto.class, false);
    private static BeanCopier goodsCouponCopier=BeanCopier.create(GoodsCouponEntity.class, GoodsCouponDto.class, false);

    @Autowired
    protected GoodsCouponService goodsCouponService;
    @Autowired
    protected GoodsBatchService goodsBatchService;
    @Autowired
    protected PlatformCouponGoodsService platformCouponGoodsService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;
    @Autowired
    private GoodsBatchImportLogService goodsBatchImportLogService;
    @Autowired
    private RemoteIDMakerBackendService remoteIDMakerBackendService;
    @Autowired
    private StockService stockService;
    
    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end) throws GoodsException{
        // 查询现有日期的批次是否存在
    	GoodsBatchEntity entity = goodsBatchService.findByStartDayAndDay(gtype, gid, start, end);
        if(entity != null){
			// 如果是删除的返回批次ID，上传文件导入成功后会标记为未删除
			if (entity.getStatus() == GoodsBatchEntity.StatusDeleted) {
				return entity.getId();
			}
			throw new GoodsException(ErrorCode.E0404001);
        }
        DubboResult<Long> stockRet=remoteStockBackendService.newStock(gid, 0);
        if(!stockRet.isSuccess()){
        	throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
        }
        Long batchId=goodsBatchService.createNormalBatch(gtype, gid, start, end,stockRet.getResult());
        if (GoodsTypeEnum.PLATFORM == gtype) {
        	platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeNormal);
        }
        return batchId;
    }

	@Override
	public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, String link, long stock) throws GoodsException {
		Long batchId;
		// 查询现有日期的批次是否存在
		GoodsBatchEntity entity = goodsBatchService.findByStartDayAndDay(gtype, gid, start, end);
		if (entity != null) {
			batchId = entity.getId();
			// 如果是删除的标记为未使用
			if (entity.getStatus() == GoodsBatchEntity.StatusDeleted) {
				goodsBatchService.markBatchStatusNotUsed(entity.getId());
				// 修改链接信息
				goodsCouponService.updateLinkBatch(gtype, gid, batchId, link);
				// 更新库存
				DubboResult<Boolean> ret = remoteStockBackendService.increaseItemStock(entity.getStockId(), stock);
				if (!ret.isSuccess()) {
					throw new GoodsException(ErrorCode.E0203003.getErrorCode(), "增加库存失败");
				}
			} else {
				throw new GoodsException(ErrorCode.E0404001);
			}
		} else {
			// 新建一个库存
			DubboResult<Long> stockRet = remoteStockBackendService.newStock(gid, stock);
			if (stockRet.isSuccess()) {
				batchId = goodsBatchService.createLinkBatch(gtype, gid, start, end, stockRet.getResult());
				if (GoodsTypeEnum.PLATFORM == gtype) {
					platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeLink);
				}
				goodsCouponService.importLinkCoupon(gtype, gid, batchId, getNewGoodsCouponId(gid), link);
			} else {
				log.error("newStock error, because of:" + stockRet.getMsg());
				throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
			}
		}
		// 刷新选择一个可用批次
		goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
		return batchId;
	}

	@Override
	public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, String code, String password, long stock) throws GoodsException {
		Long batchId;
		// 查询现有日期的批次是否存在
		GoodsBatchEntity entity = goodsBatchService.findByStartDayAndDay(gtype, gid, start, end);
		if (entity != null) {
			batchId = entity.getId();
			// 如果是删除的标记为未使用
			if (entity.getStatus() == GoodsBatchEntity.StatusDeleted) {
				goodsBatchService.markBatchStatusNotUsed(entity.getId());
				// 修改重复券码信息
				goodsCouponService.updateRepeatBatch(gtype, gid, batchId, code, password);
				// 更新库存
				DubboResult<Boolean> ret = remoteStockBackendService.increaseItemStock(entity.getStockId(), stock);
				if (!ret.isSuccess()) {
					throw new GoodsException(ErrorCode.E0203003.getErrorCode(), "增加库存失败");
				}
			} else {
				throw new GoodsException(ErrorCode.E0404001);
			}
		} else {
			// 更新库存
			DubboResult<Long> stockRet = remoteStockBackendService.newStock(gid, stock);
			if (stockRet.isSuccess()) {
				batchId = goodsBatchService.createRepeatBatch(gtype, gid, start, end, stockRet.getResult());
				if (GoodsTypeEnum.PLATFORM == gtype) {
					platformCouponGoodsService.editPlatformCouponType(gid, GoodsBatchEntity.BatchTypeRepeat);
				}
				goodsCouponService.importRepeatCoupon(gtype, gid, batchId, getNewGoodsCouponId(gid), code, password);
				// 新增批次，后选择一个可用批次
				goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
				return batchId;
			} else {
				log.error("newStock error, because of:" + stockRet.getMsg());
				throw new GoodsException(ErrorCode.E0203003.getErrorCode(), stockRet.getMsg());
			}
		}
		// 刷新选择一个可用批次
		goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
		return batchId;
	}

    @Override
    public Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link, long changeStock) {
        boolean success=goodsCouponService.updateLinkBatch(gtype, gid, batchId, link);
        if(!success){
            return success;
        }
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        // 更新库存
        if(changeStock>0){
            DubboResult<Boolean> ret=remoteStockBackendService.increaseItemStock(e.getStockId(), changeStock);
            if(!ret.isSuccess()){
                return false;
            }
            //如果该批次已经标记为已使用，更新批次为未使用
            GoodsBatchEntity batch=goodsBatchService.find(batchId);
            if(batch.getStatus()==GoodsBatchEntity.StatusUsed){
                goodsBatchService.markBatchStatusNotUsed(batchId);
            }
        }else if(changeStock<0){
            DubboResult<Boolean> ret=remoteStockBackendService.decreaseItemStock(e.getStockId(), changeStock*(-1));
            if(!ret.isSuccess()){
                return false;
            }
        }
        // 切换批次，刷新批次库存
		goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
        return true;
    }

	@Override
	public Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password, long changeStock) {
		boolean success = goodsCouponService.updateRepeatBatch(gtype, gid, batchId, code, password);
		if (!success) {
			return success;
		}
		GoodsBatchEntity e = goodsBatchService.find(batchId);
		// 更新库存
		if (changeStock > 0) {
			DubboResult<Boolean> ret = remoteStockBackendService.increaseItemStock(e.getStockId(), changeStock);
			if (!ret.isSuccess()) {
				return false;
			}
			// 如果该批次已经标记为已使用，更新批次为未使用
			GoodsBatchEntity batch = goodsBatchService.find(batchId);
			if (batch.getStatus() == GoodsBatchEntity.StatusUsed) {
				goodsBatchService.markBatchStatusNotUsed(batchId);
			}
		} else if (changeStock < 0) {
			DubboResult<Boolean> ret = remoteStockBackendService.decreaseItemStock(e.getStockId(), changeStock * (-1));
			if (!ret.isSuccess()) {
				return false;
			}
		}
		// 切换批次，刷新批次库存
		goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
		return success;
	}

	@Override
	public List<GoodsBatchDto> findBatchs(GoodsTypeEnum gtype, long gid) {
		List<GoodsBatchEntity> list = goodsBatchService.findNotDeletedBatchs(gtype, gid);
		List<Long> stockIds = new ArrayList<>();
		for (GoodsBatchEntity entity : list) {
			stockIds.add(entity.getStockId());
		}
		Map<Long, StockDto> stockMap = stockService.findStockByStockIds(stockIds);
		List<GoodsBatchDto> ret = new ArrayList<>(list.size());
		Date now = new Date();
		for (GoodsBatchEntity e : list) {
			GoodsBatchDto dto = convert(e);
			StockDto stockDto = stockMap.get(e.getStockId());
			Long stock = 0L;
			// 剩余库存不加入已使用，删除，过期的
			if (GoodsBatchEntity.StatusUsed != e.getStatus() && GoodsBatchEntity.StatusDeleted != e.getStatus() && e.getEndDay().after(now)) {
				stock = stockDto == null ? 0 : stockDto.getStock();
			}
			Long totalStock = 0L;
			// 总库存不加入删除的
			if (GoodsBatchEntity.StatusDeleted != e.getStatus()) {
				totalStock = stockDto == null ? 0 : stockDto.getTotalStock();
			}
			dto.setStock(stock);
			dto.setTotalStock(totalStock);
			if (GoodsBatchEntity.BatchTypeLink == e.getBatchType() || GoodsBatchEntity.BatchTypeRepeat == e.getBatchType()) {
				// 根据批次找券没找到,那就换下个批次看看
				GoodsCouponEntity c = goodsCouponService.findOneByGoodsBatchId(gtype, gid, e.getId());
				if (c == null) {
					continue;
				}
				dto.setLink(c.getLink());
				dto.setCode(c.getCode());
				dto.setPassword(c.getPassword());
			}
			ret.add(dto);
		}
		return ret;
	}


    @Override
    public Long importNormalCoupons(final GoodsTypeEnum gtype,final long gid,final Long batchId, final String downloadUrl) {
        GoodsBatchImportLogEntity e=new GoodsBatchImportLogEntity();
        e.setGoodsBatchId(batchId);
        final Long importLogId=goodsBatchImportLogService.insert(e);
        
        int data=concurrent.incrementAndGet();
        if(data>5){
            concurrent.decrementAndGet();
            throw new RuntimeGoodsException(ErrorCode.E0404002);
        }
        
        es.submit(new Runnable() {
            
            @Override
            public void run() {
				Integer successCount = 0;
				Integer failCount = 0;
                try{
					List<String> lines = downloadTxt(downloadUrl);
					if (lines.isEmpty()) {
						throw new RuntimeGoodsException(ErrorCode.E0404004);
					}
					long s = System.currentTimeMillis();
					failCount = lines.size();
                    List<CouponFormat> cfs=new ArrayList<>(lines.size());
                    Map<String, String> couponMap=new HashMap<>(lines.size());
                    List<Long> goodsCouponIds=getNewBatchGoodsCouponIds(gid, lines.size());
                    String split="\t";
                    for(int i=0;i<lines.size();i++){
                        String l=lines.get(i);
                        String[] ss=l.split(split);
                        if (ss.length == 0) {
                            break;
                        }
                        String code=ss[0].trim();
                        String password = StringUtils.EMPTY;
                        if (ss.length == 2) {
                            password = ss[1];
                        }
						if (StringUtils.isNotEmpty(code) && code.length() > COUPON_MAX_LENGTH) {
							throw new RuntimeGoodsException(ErrorCode.E0404003);
						}
						if (StringUtils.isNotEmpty(password) && password.length() > COUPON_MAX_LENGTH) {
							throw new RuntimeGoodsException(ErrorCode.E0404003);
						}
                        couponMap.put(code, password);
                        CouponFormat cf=new CouponFormat();
                        cf.setCode(code);
                        cf.setGoodsCouponId(goodsCouponIds.get(i));
                        cf.setPassword(password);
                        cfs.add(cf);
                    }
                    GoodsBatchEntity batch=goodsBatchService.find(batchId);
                    successCount=goodsCouponService.importNormalCoupons(batch, cfs);
                    long e = System.currentTimeMillis();
					logTime("batchInsertCoupon", s, e);
					failCount = failCount - successCount;
                    if(successCount>0){
                        //如果该批次已经标记为已使用，更新批次为未使用
                        if(batch.getStatus()==GoodsBatchEntity.StatusUsed){
                            goodsBatchService.markBatchStatusNotUsed(batchId);
                        }
                    	// 切换批次，刷新批次库存
                    	goodsBatchService.refreshGoodsBatchsNow(gtype, gid);
                    }
				} catch (RuntimeGoodsException e) {
					log.warn("importNormalCoupons thread error:" + e.getMessage());
				} catch (Exception e) {
					log.error("importNormalCoupons thread error:", e);
				} finally {
					concurrent.decrementAndGet();
					goodsBatchImportLogService.updateSuccessAndFail(importLogId, successCount, failCount);
				}
            }
        });
        return importLogId;
    }

    @Override
    public GoodsBatchImportLogDto findBatchImportLog(Long goodsBatchImportLogId) {
        GoodsBatchImportLogEntity e=goodsBatchImportLogService.select(goodsBatchImportLogId);
        return BeanUtils.copy(e, GoodsBatchImportLogDto.class);
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        return goodsBatchService.deleteBatch(gtype, gid, batchId);
    }

    @Override
    public Page<GoodsCouponDto> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize, int pageIndex) {
        GoodsBatchEntity batch=goodsBatchService.find(batchId);
        DubboResult<Long> stockRet=remoteStockBackendService.findTotalStock(batch.getStockId());
        if(!stockRet.isSuccess()){
           return new Page<>(pageSize, pageIndex);
        }
        
        Page<GoodsCouponEntity> page=goodsCouponService.findPage(gtype, gid, batchId, pageSize, pageIndex,stockRet.getResult().intValue());
        
        Page<GoodsCouponDto> ret=new Page<>(pageSize, pageIndex);
        ret.setTotalPages(page.getTotalPages());
        ret.setTotalCount(page.getTotalCount());
        
        List<GoodsCouponDto> list=new ArrayList<>(page.getList().size());
        for(GoodsCouponEntity e:page.getList()){
            list.add(convert(e));
        }
        ret.setList(list);
        
        return ret;
    }

    @Override
    public GoodsBatchDto findBatchStock(GoodsTypeEnum gtype, long gid, long batchId) {
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        GoodsBatchDto dto=convert(e);
        if(e.getStockId()==null){
            dto.setStock(0L);
            dto.setTotalStock(0L);
            return dto;
        }
        StockDto stock = stockService.findStock(e.getStockId());
        dto.setStock(stock.getStock());
        dto.setTotalStock(stock.getTotalStock());
        return dto;
    }
    
    @Override
    public List<GoodsCouponDto> findCouponByCode(GoodsTypeEnum gtype, long gid, long batchId, String code) {
        List<GoodsCouponEntity> list=goodsCouponService.searchByCode(gtype, gid, batchId, code);
        List<GoodsCouponDto> ret=new ArrayList<>();
        for(GoodsCouponEntity e:list){
            ret.add(convert(e));
        }
        return ret;
    }

    @Override
    public GoodsCouponDto findCoupon(GoodsTypeEnum gtype, long gid, long couponId) {
        GoodsCouponEntity entity=goodsCouponService.find(couponId);
        if(entity!=null){
            return convert(entity);
        }
        return null;
    }
    
    @Override
    public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, Long couponId, Long batchId) {
        boolean del = goodsCouponService.deleteUnusedCoupon(gtype, gid, couponId, batchId);
        if(del){
        	GoodsBatchEntity e=goodsBatchService.find(batchId);
	        remoteStockBackendService.decreaseItemStock(e.getStockId(), 1);
        }
        return del;
    }
    
    @Override
    public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, Long batchId) {
        GoodsBatchEntity e=goodsBatchService.find(batchId);
        return goodsCouponService.deleteBatchUnusedCoupons(e);
    }
    
    private Long getNewGoodsCouponId(Long gid){
        List<Long> ids=getNewBatchGoodsCouponIds(gid, 1);
        return ids.get(0);
    }
    
    private List<Long> getNewBatchGoodsCouponIds(Long gid,int count){
        DubboResult<Long> ret=remoteIDMakerBackendService.getBatchNextID(null, count);
        if(ret.isSuccess()){
            List<Long> ids=new ArrayList<>(count);
            for(int i=0;i<count;i++){
                ids.add(Long.valueOf((ret.getResult()+i)+IdSuffixTool.getTableSuffix(gid)));
            }
            return ids;
        }
        throw new RuntimeGoodsException(ErrorCode.E0203001);
    }
    

    private GoodsBatchDto convert(GoodsBatchEntity e){
        GoodsBatchDto d=new GoodsBatchDto();
        goodsBatchCopier.copy(e, d, null);
        return d;
    }
    private GoodsCouponDto convert(GoodsCouponEntity e){
        GoodsCouponDto d=new GoodsCouponDto();
        goodsCouponCopier.copy(e, d, null);
        return d;
    }

	private static List<String> downloadTxt(String url) {
		long s = System.currentTimeMillis();
		File f = null;
		try {
			f = File.createTempFile("coupon-", "txt");
			URL httpurl = new URL(url);
			FileUtils.copyURLToFile(httpurl, f);
			FileInputStream in = null;
			BufferedReader br = null;
			try {
				in = new FileInputStream(f);
				br = new BufferedReader(new UnicodeReader(in, "UTF-8"));
				String line = "";
				List<String> lines = new ArrayList<>();
				while (StringUtils.isNotBlank(line = br.readLine())) {
					lines.add(line);
				}
				return lines;
			} finally {
				if (br != null) {
					br.close();
				}
				if (in != null) {
					in.close();
				}
			}
		} catch (Exception e) {
			log.error("downloadTxt error: url=" + url, e);
		} finally {
			long e = System.currentTimeMillis();
			logTime("download url:" + url, s, e);
			if (f != null && !f.delete()) {
				log.warn("delete file fail");
			}
		}
		return Collections.emptyList();
	}
	
	private static void logTime(String text, long start, long end) {
		long s = end - start;
		if (s > 500) {
			log.info(text + " timeProfile: " + s + "ms");
		}
	}

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, Long batchId, List<Long> couponIds) {
        return goodsCouponService.deleteBatchUnusedCoupons(gtype, gid, batchId, couponIds);
	}

	@Override
	public List<GoodsCouponDto> findUnusedCoupons(GoodsTypeEnum gtype, long gid, Long batchId, int limit) {
		List<GoodsCouponEntity> entity = goodsCouponService.findUnusedCoupons(gtype, gid, batchId, limit);
		List<GoodsCouponDto> ret = new ArrayList<>();
		for (GoodsCouponEntity e : entity) {
			ret.add(convert(e));
		}
		return ret;
	}
    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long acgId, long batchId, Date startDay, Date endDay) {
        return goodsBatchService.updateValidDate(gtype, acgId, batchId, startDay, endDay);
    }
}

