/**
 * Project Name:goods-center-biz
 * File Name:GoodsCouponBOImpl.java
 * Package Name:cn.com.duiba.goods.center.biz.bo.impl
 * Date:2016年5月24日下午4:40:02
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.bo.impl;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.dto.ACGStockDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsCouponDto;
import cn.com.duiba.goods.center.api.remoteservice.dto.GoodsStockDto;
import cn.com.duiba.goods.center.biz.bo.GoodsCouponBO;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.ConsumeStockTypeUtil;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * ClassName:GoodsCouponBOImpl <br/>
 * Date:     2016年5月24日 下午4:40:02 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("goodsCouponBO")
public class GoodsCouponBOImpl implements GoodsCouponBO{
    
    private static Logger log=LoggerFactory.getLogger(GoodsCouponBOImpl.class);
    
    @Autowired
    @Qualifier("goodsCouponService4Speed")
    private GoodsCouponService goodsCouponService;
    @Autowired
    private GoodsBatchService goodsBatchService;
    @Autowired
    private RemoteStockService remoteStockService;
    @Autowired
    private StockService stockService;
    
    @Override
    public void completeCoupon(Long couponId, long orderId) {
        goodsCouponService.completeCoupon(couponId, orderId);
    }

    @Override
    public Boolean rollbackCoupon(Long couponId,String bizNum) {
        GoodsCouponEntity coupon=goodsCouponService.find(couponId);
        long goodsBatchId=coupon.getGoodsBatchId();
        GoodsBatchEntity batch=goodsBatchService.find(goodsBatchId);
        
        if(batch.getBatchType()==GoodsBatchEntity.BatchTypeNormal){
            Boolean succ=goodsCouponService.rollbackNormalCoupon(couponId);
            if(!succ){
                return succ;
            }
        }
        //所有的类型都需要尝试到库存中心返还库存，除了主库存，还有定向库存等等需要返还
        DubboResult<Boolean> stockRet=remoteStockService.rollbackStock(ConsumeStockTypeUtil.getConsumeStockTypes(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype())).getType(), bizNum);
        if(!stockRet.isSuccess()){
            log.error("remoteStockService.rollbackStock fail,couponId="+couponId+",bizNum="+bizNum);
            return false;
        }
        return stockRet.getResult();
    }

	@Override
	public GoodsCouponEntity findCoupon(Long goodsCouponId) {
		return goodsCouponService.find(goodsCouponId);
	}

	@Override
	public Long findGoodsStock(GoodsTypeEnum gtype, long gid) {
		return goodsBatchService.getSumBatchStockBatch(gtype, gid);
	}

	@Override
	public List<GoodsStockDto> findGoodsStockByBatch(List<GoodsStockDto> goodsDtos) {
		List<GoodsStockDto> stocks = new ArrayList<>();
		if (goodsDtos == null) {
			return stocks;
		}
		EnumMap<GoodsTypeEnum, List<Long>> map = new EnumMap<>(GoodsTypeEnum.class);
		for (GoodsStockDto goods : goodsDtos) {
			List<Long> gids = map.get(goods.getGtype());
			if (gids == null) {
				gids = new ArrayList<>();
			}
			gids.add(goods.getGid());
			map.put(goods.getGtype(), gids);
		}
		// 已知GoodsTypeEnum类型4 for不会太多
		for (Map.Entry<GoodsTypeEnum, List<Long>> entry : map.entrySet()) {
			if (entry.getValue().isEmpty()) {
				continue;
			}
			List<ACGStockDto> dtos = goodsBatchService.findStockByGids(entry.getKey(), entry.getValue());
			for (ACGStockDto acg : dtos) {
				GoodsStockDto dto = new GoodsStockDto();
				dto.setGtype(entry.getKey());
				dto.setGid(acg.getAcgId());
				dto.setStock(acg.getStock());
				dto.setTotalStock(acg.getTotalStock());
				stocks.add(dto);
			}
		}
		return stocks;
	}

	@Override
	public GoodsBatchEntity findBatch(Long goodsBatchId) {
		return goodsBatchService.find(goodsBatchId);
	}

	@Override
	public GoodsBatchEntity findUsingBatch(GoodsTypeEnum gtype, long gid) {
		List<GoodsBatchEntity> batchs = goodsBatchService.findNormalBatchs(gtype, gid);
		for (GoodsBatchEntity entity : batchs) {
			if (entity.getStatus() == GoodsBatchEntity.StatusUsing) {
				return entity;
			}
		}
		return null;
	}

	@Override
	public GoodsCouponDto takeCoupon(GoodsTypeEnum gtype, long gid, Long consumerId, Long appId, String bizNum) {
		// 1.获取当前使用中的批次
		GoodsBatchEntity using = getUsingBatch(gtype, gid);
		if (using == null) {
			return null;
		}
		// 2.扣批次库存发券
		return takeCouponInCycle(using, consumerId, appId, bizNum, 1);
	}

	/**
	 * 获取使用中的批次<br/>
	 * 如果没有切换下一个批次
	 * 
	 * @param gtype
	 * @param gid
	 * @return
	 */
	private GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid) {
		try {
			DBTimeProfile.enter("getUsingBatch");
			// 1.获取当前使用中的批次
			GoodsBatchEntity using = findUsingBatch(gtype, gid);
			if (using != null) {
				return using;
			}
			// 2.切换下一个批次
			GoodsBatchEntity nextUsing = goodsBatchService.refreshGoodsBatchs(gtype, gid);
			if (nextUsing != null) {
				return nextUsing;
			}
			return null;
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 扣批次库存，如果库存不足切换下一个批次<br/>
	 * 
	 * @param using
	 * @param consumerId
	 * @param appId
	 * @param bizNum
	 * @return
	 */
	private GoodsCouponDto takeCouponInCycle(GoodsBatchEntity using, Long consumerId, Long appId, String bizNum, int count) {
		if (count > 5) {
			throw new RuntimeGoodsException(ErrorCode.E0404002);
		}
		GoodsTypeEnum gtype = GoodsTypeEnum.getGoodsTypeEnum(using.getGtype());
		int innerCount = count + 1;
		long gid = using.getGid();
		try {
			// 1.扣批次库存发券
			GoodsCouponEntity coupon = takeGoodsCoupon(using, consumerId, appId, bizNum);
			GoodsCouponDto dto = BeanUtils.copy(coupon, GoodsCouponDto.class);
			dto.setStartDay(using.getStartDay());
			dto.setOverDue(using.getEndDay());
			return dto;
		} catch (RuntimeGoodsException e) {
			// 2.切换到下一个批次
			if (e.getErrorCode() == ErrorCode.E0202005) {
				GoodsBatchEntity nextBatch = goodsBatchService.refreshGoodsBatchs(gtype, gid);
				if (nextBatch == null) {
					return null;
				}
				// 3.重新发下一个批次的券
				return takeCouponInCycle(nextBatch, consumerId, appId, bizNum, innerCount);
			}
			throw e;
		}
	}
	
	/**
	 * 扣批次库存发券
	 * 
	 * @param using
	 * @param consumerId
	 * @param appId
	 * @param bizNum
	 * @return
	 */
	private GoodsCouponEntity takeGoodsCoupon(GoodsBatchEntity using, Long consumerId, Long appId, String bizNum) {
		GoodsTypeEnum gtype = GoodsTypeEnum.getGoodsTypeEnum(using.getGtype());
		GoodsCouponEntity coupon;
		// 1.扣批次库存发一个券
		if (using.getBatchType() == GoodsBatchEntity.BatchTypeLink) {
			// 链接券
			coupon = goodsCouponService.takeLinkCoupon(gtype, using, consumerId, appId, bizNum);
		} else if (using.getBatchType() == GoodsBatchEntity.BatchTypeRepeat) {
			// 重复券
			coupon = goodsCouponService.takeRepeatCoupon(gtype, using, consumerId, appId, bizNum);
		} else if (using.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
			// 普通券
			coupon = goodsCouponService.takeNormalCoupon(gtype, using, consumerId, appId, bizNum);
		} else {
			throw new RuntimeGoodsException(ErrorCode.E0202008);
		}
		return coupon;
	}

}

