/**
 * Project Name:goods-center-biz
 * File Name:GoodsCouponService.java
 * Package Name:cn.com.duiba.goods.center.biz.service
 * Date:2016年5月25日下午12:23:22
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;

import java.util.List;

/**
 * ClassName:GoodsCouponService <br/>
 * Date:     2016年5月25日 下午12:23:22 <br/>
 * @author   xuhengfei
 * @version
 * @since    JDK 1.6
 * @see
 */
public interface GoodsCouponService {
    /**
     * 更新链接券的批次上的链接信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId  批次id
     * @param link  链接内容
     * @return true/false
     * @since JDK 1.6
     */
    Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link);
    /**
     * 更新重复券的批次上的卡密信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId 批次id
     * @param code  卡号
     * @param password  密码
     * @return true/false 更新是否成功
     * @since JDK 1.6
     */
    Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password);

    /**
     * 导入普通券的券码
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param batchId 批次id
     * @param coupons 券码列表
     * @return 成功导入条数
     * @since JDK 1.6
     */
    Integer importNormalCoupons(GoodsBatchEntity batch, List<CouponFormat> coupons);
    /**
     * 导入链接券的券码<br/>
     * 此方法只在创建链接批次时，自动执行插入一条记录
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param goodsBatchId  批次id
     * @param goodsCouponId 券号
     * @param link 链接内容
     * @since JDK 1.6
     */
    void importLinkCoupon(GoodsTypeEnum gtype,long gid,long goodsBatchId,long goodsCouponId,String link);
    /**
     * 导入重复券的券码<br/>
     * 此方法只在创建重复批次是，自动执行插入一条记录
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param goodsBatchId 批次id
     * @param goodsCouponId 券号
     * @param code 卡号
     * @param password 密码
     * @since JDK 1.6
     */
    void importRepeatCoupon(GoodsTypeEnum gtype,long gid,long goodsBatchId,long goodsCouponId,String code,String password);
    /**
     * 分页查询券码信息
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @param pageSize 分页大小
     * @param pageIndex 页码(从1开始)
     * @param total 总共条数(请外部通过库存中心查询结果，然后传参进来)
     * @return 查询结果列表
     * @since JDK 1.6
     */
	Page<GoodsCouponEntity> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize, int pageIndex, int total);


    /**
     * 查询券码根据状态
     * @param gtype
     * @param gid
     * @param batchId
     * @param status
     * @param start
     * @param limit
     * @return
     */
	List<GoodsCouponEntity> findPageByStatus(GoodsTypeEnum gtype, long gid, long batchId, int status, int start, int limit);

    /**
     * 查询一张优惠劵
     *
     * @author xuhengfei
     * @param goodsCouponId 券号
     * @return 券信息
     * @since JDK 1.6
     */
    GoodsCouponEntity find(Long goodsCouponId);
    /**
     * 确认完成一张优惠劵的发放
     *
     * @author xuhengfei
     * @param couponId 券号
     * @param orderId 订单id
     * @since JDK 1.6
     */
    void completeCoupon(Long couponId, long orderId);
    /**
     * 回滚返还一张优惠劵
     *
     * @author xuhengfei
     * @param couponId 券号
     * @return 是否成功
     * @since JDK 1.6
     */
    Boolean rollbackNormalCoupon(Long couponId);


    /**
     * 链接券和重复券，根据批次，查询信息
     * @param gtype
     * @param gid
     * @param goodsBatchId
     * @return
     */
    GoodsCouponEntity findOneByGoodsBatchId(GoodsTypeEnum gtype,long gid,long goodsBatchId);
    /**
     * 根据批次id和卡号搜索
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid  商品id
     * @param goodsBatchId 批次id
     * @param code 卡号
     * @return 搜索结果集
     * @since JDK 1.6
     */
    List<GoodsCouponEntity> searchByCode(GoodsTypeEnum gtype,long gid,long goodsBatchId,String code);

    /**
     * 删除一张未使用的优惠劵
     * @param gtype
     * @param gid
     * @param couponId
     * @param batchId
     * @return
     */
    public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype,long gid,long couponId, long batchId);
    
    /**
     * 删除一个批次下未使用的优惠劵，内存锁，超过5W拒绝执行
     * @param batch
     * @return
     */
    public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch);
    /**
     * 根据优惠券ID删除批次指定的优惠券
     * @param gtype
     * @param gid
     * @param batchId
     * @param couponIds
     * @return
     */
    public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype,long gid,long batchId,List<Long> couponIds);

    /**
     * 查询批次下是否存在已使用的优惠券
     * @param gtype
     * @param gid
     * @param goodsBatchId 批次ID
     * @return
     */
    public Boolean findBatchExsitUsedCoupon(GoodsTypeEnum gtype,long gid,long goodsBatchId);

    /**
     * 查询批次下未使用的优惠券
     * @param gtype
     * @param gid
     * @param batchId
     * @param limit
     * @return
     */
    public List<GoodsCouponEntity> findUnusedCoupons(GoodsTypeEnum gtype,long gid,long batchId,int limit);

    /**
     *
     * 删除某个优惠券指定批次所有券码. <br/>
     *
     * @author ZFZ
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 是否成功
     * @since JDK 1.6
     */
    boolean deleteGoodsCoupon(GoodsTypeEnum gtype, long gid, long batchId);

    /**
     * 批量删除券码
     * @param gid
     * @param longs
     * @return
     */
    int deleteGoodsCouponByIds(long gid, Long batchId, List<Long> longs);
    
	/**
	 * 从数据库获取一条优惠劵
	 * @param gtype
	 * @param batch
	 * @param consumerId
	 * @param appId
	 * @param bizNum
	 * @return
	 */
	GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum);

	/**
	 * 领一张链接券
	 * @param gtype
	 * @param batch
	 * @param consumerI
	 * @param appId
	 * @param bizNum
	 * @return
	 */
	GoodsCouponEntity takeLinkCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerI, Long appId, String bizNum);

	/**
	 * 领一张重复券
	 * @param gtype
	 * @param batch
	 * @param consumerI
	 * @param appId
	 * @param bizNum
	 * @return
	 */
	GoodsCouponEntity takeRepeatCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerI, Long appId, String bizNum);
}

