/**
 * Project Name:goods-center-biz File Name:GoodsCouponServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午1:08:07 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.dao.GoodsBatchDao;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.RedisCacheService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.ConsumeStockTypeUtil;
import cn.com.duiba.goods.center.biz.util.RedisKeyFactory;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.GoodsException;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.idmaker.service.api.remoteservice.RemoteIDMakerBackendService;
import cn.com.duiba.stock.service.api.constant.STErrorCode;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

import com.alibaba.fastjson.JSONObject;

/**
 * ClassName:GoodsCouponServiceImpl <br/>
 * Date: 2016年5月25日 下午1:08:07 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
@Service("goodsCouponService")
public class GoodsCouponServiceImpl implements GoodsCouponService {

	private static Logger log = LoggerFactory.getLogger(GoodsCouponServiceImpl.class);

	private Lock lock = new ReentrantLock();

	@Autowired
	private GoodsCouponDao goodsCouponDao;
	@Autowired
	private GoodsBatchDao goodsBatchDao;
	@Autowired
	private RemoteStockService remoteStockService;
	@Autowired
	private StockService stockService;
	@Autowired
	protected GoodsCouponService goodsCouponService;
	@Autowired
	protected GoodsBatchService goodsBatchService;
	@Autowired
	private RemoteIDMakerBackendService remoteIDMakerBackendService;
	@Autowired
	private RemoteStockBackendService remoteStockBackendService;
	@Autowired
	protected RedisCacheService redisCacheService;

	@Override
	public GoodsCouponEntity find(Long goodsCouponId) {
		return goodsCouponDao.selectByCouponId(goodsCouponId);
	}

	@Override
	public Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link) {
		int ret = goodsCouponDao.updateLinkCoupon(gtype, gid, batchId, link);
		if (ret == 1) {
			return true;
		}
		return false;
	}

	@Override
	public void importLinkCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String link) {
		goodsCouponDao.insertLinkCoupon(gtype, gid, goodsBatchId, goodsCouponId, link);
	}

	@Override
	public void importRepeatCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String code, String password) {
		goodsCouponDao.insertRepeatCoupon(gtype, gid, goodsBatchId, goodsCouponId, code, password);
	}

	@Override
	public Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password) {
		int ret = goodsCouponDao.updateRepeatCoupon(gtype, gid, batchId, code, password);
		if (ret == 1) {
			return true;
		}
		return false;
	}

	@Override
	public Integer importNormalCoupons(GoodsBatchEntity batch, List<CouponFormat> coupons) {
		if (coupons.size() > 1000000) {
			throw new RuntimeGoodsException(ErrorCode.E9999999, new Throwable("超出100W，拒绝导入"));
		}

		final int maxRecords = 20000;

		int size = coupons.size();
		int batchs = size % maxRecords == 0 ? (size / maxRecords) : (size / maxRecords + 1);
		List<List<CouponFormat>> group = new ArrayList<>(batchs);
		Map<Integer, AtomicInteger> countMap = new HashMap<>();

		Set<String> set = new HashSet<>();

		for (int i = 0; i < batchs; i++) {
			List<CouponFormat> l = new ArrayList<>(maxRecords);
			group.add(l);
			countMap.put(i, new AtomicInteger());
		}

		// 先分组
		for (CouponFormat cf : coupons) {
			if (set.add(cf.getCode())) {
				for (int i = 0; i < group.size(); i++) {
					List<CouponFormat> cflist = group.get(i);
					if (countMap.get(i).get() < maxRecords) {
						countMap.get(i).incrementAndGet();
						cflist.add(cf);
						break;
					}
				}
			} else {
				log.warn("重复券码 code=" + cf.getCode());
			}
		}

		int totalSuccess = 0;
		// 对每一个分组进行一个批量插入动作
		for (List<CouponFormat> list : group) {
			if (!list.isEmpty()) {
				totalSuccess += goodsCouponDao.insertBatchImport(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid(), batch.getId(), list);
			}
		}
		// 更新库存中心的库存
		DubboResult<Boolean> stockRet = remoteStockBackendService.increaseItemStock(batch.getStockId(), totalSuccess);
		if (!stockRet.isSuccess() && !stockRet.getResult()) {
			log.error("remoteStockBackendService.increaseItemStock fail ,msg=" + stockRet.getMsg());
		}
		return totalSuccess;
	}

	@Override
	public Page<GoodsCouponEntity> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize, int pageIndex, int total) {
		Page<GoodsCouponEntity> page = new Page<>(pageSize, pageIndex);
		int count = total;
		int start = (pageIndex - 1) * pageSize;
		int limit = pageSize;
		List<GoodsCouponEntity> list = goodsCouponDao.selectPageByBatchId(gtype, gid, batchId, start, limit);
		page.setTotalPages(count % pageSize == 0 ? (count / pageSize) : (count / pageSize + 1));
		page.setList(list);
		page.setTotalCount(count);
		return page;
	}

	@Override
	public List<GoodsCouponEntity> findPageByStatus(GoodsTypeEnum gtype, long gid, long batchId, int status, int start, int limit) {
		return goodsCouponDao.selectPageByBatchIdAndStatus(gtype, gid, batchId, status, start, limit);
	}

	@Override
	public void completeCoupon(Long couponId, long orderId) {
		goodsCouponDao.updateCoupnoMarkUsed(couponId, orderId);
	}

	@Override
	public Boolean rollbackNormalCoupon(Long couponId) {
		int ret = goodsCouponDao.updateCouponMarkRollback(couponId);
		return ret == 1 ? true : false;
	}

	/**
	 * 从数据库查询一批${limit}未使用的优惠券
	 *
	 * @param gtype
	 * @param gid
	 * @param batchId
	 * @param limit
	 * @return
	 * @since JDK 1.6
	 */
	protected List<GoodsCouponEntity> loadCouponByBatchId(GoodsTypeEnum gtype, long gid, long batchId, int limit) {
		try {
			DBTimeProfile.enter("selectBatchNotUsed");
			return goodsCouponDao.selectBatchNotUsed(gtype, gid, batchId, limit);
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 修改普通券状态
	 * 
	 * @param gtype
	 * @param gid
	 * @param goodsBatchId
	 * @param goodsCouponId
	 * @param consumerId
	 * @param bizNum
	 * @return
	 */
	protected boolean takeCoupon4Point(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, long consumerId, String bizNum) {
		try {
			DBTimeProfile.enter("takeCoupon4Point");
			if (GoodsTypeEnum.ADVERT.getGtype() == gtype.getGtype()) {
				// 广告平台直接标记为已使用
				Long bizId = StringUtils.isNumeric(bizNum) ? Long.valueOf(bizNum) : null;
				int ret = goodsCouponDao.updateAdvertCouponMarkUsed(goodsCouponId, consumerId, bizId);
				return ret > 0 ? true : false;
			} else {
				int ret = goodsCouponDao.updateCouponMarkLocked(gtype, gid, goodsBatchId, goodsCouponId, consumerId);
				return ret == 1 ? true : false;
			}
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public GoodsCouponEntity findOneByGoodsBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
		return goodsCouponDao.selectOneByGoodsBatchId(gtype, gid, goodsBatchId);
	}

	@Override
	public List<GoodsCouponEntity> searchByCode(GoodsTypeEnum gtype, long gid, long goodsBatchId, String code) {
		return goodsCouponDao.selectSearchByCode(gtype, gid, goodsBatchId, code);
	}

	@Override
	public boolean deleteGoodsCoupon(GoodsTypeEnum gtype, long gid, long batchId) {
		GoodsBatchEntity e = goodsBatchDao.select(batchId);
		int ret = goodsCouponDao.deleteGoodsCoupon(gtype, gid, batchId);
		if (ret > 0) {
			remoteStockBackendService.decreaseItemStock(e.getStockId(), ret);
			return true;
		} else {
			return false;
		}
	}

	@Override
	public int deleteGoodsCouponByIds(long gid, Long batchId, List<Long> ids) {
		return goodsCouponDao.deleteGoodsCouponByIds(gid, ids);
	}

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
		try {
			if (!lock.tryLock()) {
				return false;
			}
			if (batch.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
				// 普通券删除未使用的券码
				int ret = goodsCouponDao.deleteBatchUnusedCoupons(batch.getGtype(), batch.getGid(), batch.getId());
				if (ret > 0) {
					remoteStockBackendService.decreaseItemStock(batch.getStockId(), ret);
					return true;
				}
			} else {
				// 重复券，连接券 库存减少为0
				DubboResult<Long> stockRet = remoteStockService.find(batch.getStockId());
				remoteStockBackendService.decreaseItemStock(batch.getStockId(), stockRet.getResult());
				return true;
			}
			return false;
		} finally {
			lock.unlock();
		}
	}

	@Override
	public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
		int ret = goodsCouponDao.deleteUnusedCoupon(gtype.getGtype(), gid, couponId);
		if (ret > 0) {
			return true;
		}
		return false;
	}

	@Override
	public Boolean findBatchExsitUsedCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
		Long ret = goodsCouponDao.selectByBatchExistUsed(gtype.getGtype(), gid, goodsBatchId);
		if (ret != null) {
			return true;
		}
		return false;
	}

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		if (couponIds == null || couponIds.isEmpty() || couponIds.size() > 5000) {
			throw new RuntimeGoodsException(ErrorCode.E9999999);
		}
		try {
			if (!lock.tryLock()) {
				throw new RuntimeGoodsException(ErrorCode.E9999999);
			}
			GoodsBatchEntity e = goodsBatchDao.select(batchId);
			int ret = goodsCouponDao.deleteBatchUnusedCouponsByIds(gtype.getGtype(), gid, batchId, couponIds);
			if (ret > 0) {
				remoteStockBackendService.decreaseItemStock(e.getStockId(), ret);
			}
		} finally {
			lock.unlock();
		}
		return true;
	}

	@Override
	public List<GoodsCouponEntity> findUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, int limit) {
		if (limit > 5000) {
			throw new RuntimeGoodsException(ErrorCode.E9999999);
		}
		return goodsCouponDao.findUnusedCoupons(gtype.getGtype(), gid, batchId, limit);
	}

	@Override
	public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		try {
			DBTimeProfile.enter("takeNormalCoupon");
			boolean consumeSucc = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSucc) {
				return null;
			}
			GoodsCouponEntity entity = goodsCouponDao.selectOneCouponNotUsed(gtype, batch.getGid(), batch.getId());
			boolean success = takeCoupon4Point(gtype, batch.getGid(), batch.getId(), entity.getGoodsCouponId(), consumerId, bizNum);
			if (success) {
				return entity;
			}
			return null;
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public GoodsCouponEntity takeLinkCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		try {
			DBTimeProfile.enter("takeLinkCoupon");
			boolean consumeSucc = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSucc) {
				return null;
			}
			GoodsCouponEntity e = getLinkOrRepeatCouponCache(batch.getId());
			if (e != null) {
				return e;
			}
			e = goodsCouponDao.selectOneByGoodsBatchId(gtype, batch.getGid(), batch.getId());
			if (e != null) {
				setLinkOrRepeatCouponCache(batch.getId(), e);
			}
			return e;
		} finally {
			DBTimeProfile.release();
		}
	}

	@Override
	public GoodsCouponEntity takeRepeatCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, Long consumerId, Long appId, String bizNum) {
		try {
			DBTimeProfile.enter("takeRepeatCoupon");
			boolean consumeSucc = consumeStocks(gtype, batch, bizNum, appId);
			if (!consumeSucc) {
				return null;
			}
			GoodsCouponEntity e = getLinkOrRepeatCouponCache(batch.getId());
			if (e != null) {
				return e;
			}
			e = goodsCouponDao.selectOneByGoodsBatchId(gtype, batch.getGid(), batch.getId());
			if (e != null) {
				setLinkOrRepeatCouponCache(batch.getId(), e);
			}
			return e;
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 扣库存
	 * 
	 * @param gtype
	 * @param batch
	 * @param consumerId
	 * @param bizNum
	 * @param c
	 * @return
	 */
	protected boolean consumeStocks(GoodsTypeEnum gtype, GoodsBatchEntity batch, String bizNum, Long appId) {
		// 需要扣库存的库存ID
		List<Long> stockIds = stockService.getNeedCountDownOtherStockIds(gtype.getGtype(), batch.getGid(), appId);
		stockIds.add(batch.getStockId());
		try {
			DBTimeProfile.enter("remoteStockService.consumeStock");
			// 调用库存中心扣库存
			DubboResult<Boolean> stockRet = remoteStockService.consumeStock(ConsumeStockTypeUtil.getConsumeStockTypes(gtype).getType(), bizNum, stockIds);
			if (stockRet.isSuccess() && stockRet.getResult()) {
				return true;
			}
			// 扣库存响应库存不足
			if (STErrorCode.SS_0400001.getCode().equals(stockRet.getReturnCode())) {
				goodsBatchService.markBatchStatusUsed(gtype, batch.getGid(), batch.getId());
				throw new RuntimeGoodsException(ErrorCode.E0202005);
			}
			// 抛扣库存失败异常
			throw new RuntimeGoodsException(ErrorCode.E0404011, new GoodsException(stockRet.getReturnCode(), stockRet.getMsg()));
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 领券失败后返还库存
	 * 
	 * @param gtype
	 * @param bizNum
	 */
	protected void rollbackStocks(GoodsTypeEnum gtype, String bizNum) {
		try {
			DBTimeProfile.enter("remoteStockService.rollbackStock");
			log.warn("rollbackStocks gtype=" + gtype + ",bizNum=" + bizNum);
			DubboResult<Boolean> stockRet = remoteStockService.rollbackStock(ConsumeStockTypeUtil.getConsumeStockTypes(gtype).getType(), bizNum);
			if (!stockRet.isSuccess()) {
				log.error("rollbackStocks error: gtype=" + gtype + ",bizNum=" + bizNum);
			}
		} finally {
			DBTimeProfile.release();
		}
	}

	/**
	 * 查询重复券，链接券缓存
	 * 
	 * @param batchId
	 * @return
	 */
	private GoodsCouponEntity getLinkOrRepeatCouponCache(long batchId) {
		String key = RedisKeyFactory.K205 + String.valueOf(batchId);
		String string = redisCacheService.get(key);
		if (string != null) {
			return JSONObject.parseObject(string, GoodsCouponEntity.class);
		}
		return null;
	}

	/**
	 * 设置重复券，链接券缓存
	 * 
	 * @param batchId
	 * @param entity
	 */
	private void setLinkOrRepeatCouponCache(Long batchId, GoodsCouponEntity entity) {
		String key = RedisKeyFactory.K205 + String.valueOf(batchId);
		String string = JSONObject.toJSONString(entity);
		redisCacheService.set(key, string, 3600);
	}

	/**
	 * 清理重复券，链接券缓存
	 * 
	 * @param batchId
	 */
	protected void clearLinkOrRepeat(long batchId) {
		String key = RedisKeyFactory.K205 + String.valueOf(batchId);
		redisCacheService.delete(key);
	}

}
