package cn.com.duiba.goods.center.biz.service.impl;

import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.biz.service.RedisCacheService;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

/**
 * 缓存服务类 目的提升系统性能，减轻数据库压力
 */
@Service
public class RedisCacheServiceImpl implements RedisCacheService {

	private static Logger log = LoggerFactory.getLogger(RedisCacheServiceImpl.class);

	@Autowired
	private StringRedisTemplate stringRedisTemplate;
	
	@Resource(name = "redisTemplate")
	private AdvancedCacheClient advancedCacheClient;
	
	@Override
	public <T> T getObject(String key){
		try {
			return advancedCacheClient.get(key);
		} catch (Exception e) {
			log.warn("getObject:key=" + key, e);
		}
		return null;
	}
	
	@Override
	public String get(String key) {
		try {
			return stringRedisTemplate.opsForValue().get(key);
		} catch (Exception e) {
			log.warn("get:key=" + key, e);
		}
		return null;
	}
	
	@Override
	public void setObject(String key, Object value, int seconds) {
		try {
			advancedCacheClient.set(key, value, seconds, TimeUnit.SECONDS);
		} catch (Exception e) {
			log.warn("setObject:key=" + key, e);
		}
	}

	@Override
	public void set(String key, String value, int seconds) {
		try {
			stringRedisTemplate.opsForValue().set(key, value, seconds, TimeUnit.SECONDS);
		} catch (Exception e) {
			log.warn("set:key=" + key, e);
		}
	}
	
	@Override
	public void delete(String key) {
		try {
			stringRedisTemplate.delete(key);
		} catch (Exception e) {
			log.warn("set:key=" + key, e);
		}
	}

	@Override
	public boolean getLock(String key, int seconds) {
		try {
			boolean lock = stringRedisTemplate.opsForValue().setIfAbsent(key, String.valueOf(seconds));
			if (lock) {
				stringRedisTemplate.expire(key, seconds, TimeUnit.SECONDS);
				return lock;
			}
		} catch (Exception e) {
			log.error("getLock:key=" + key, e);
		}
		return false;
	}

	@Override
	public boolean tryGetLock(String key, int seconds) {
		try {
			DBTimeProfile.enter("tryGetLock");
			for (int i = 0; i < 20; i++) {
				boolean lock = stringRedisTemplate.opsForValue().setIfAbsent(key, String.valueOf(seconds));
				if (lock) {
					stringRedisTemplate.expire(key, seconds, TimeUnit.SECONDS);
					return lock;
				}
				Thread.sleep(30);
			}
		} catch (Exception e) {
			log.error("tryGetLock error", e);
		} finally {
			DBTimeProfile.release();
		}
		return false;
	}

	@Override
	public void releaseLock(String key) {
		try {
			stringRedisTemplate.delete(key);
		} catch (Exception e) {
			log.warn("releaseLock error", e);
		}
	}

	@Override
	public String pop(String key) {
		return stringRedisTemplate.opsForSet().pop(key);
	}

	@Override
	public void addQueue(String key, String[] values, int seconds) {
		stringRedisTemplate.opsForSet().add(key, values);
		stringRedisTemplate.expire(key, seconds, TimeUnit.SECONDS);
	}

	@Override
	public Long getQueueSize(String key) {
		return stringRedisTemplate.opsForSet().size(key);
	}

}
