package cn.com.duiba.goods.common.dto;

import cn.com.duiba.goods.common.enums.goods.AccountType;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.*;

/**
 * 收货账户信息
 * Created by guoyanfei .
 * 2023/1/12 .
 */
public class AccountInfo implements Serializable {

    private static final long serialVersionUID = -458034431842518256L;

    /**
     * 收货账户类型
     * {@link AccountInfo}
     */
    private Integer type;

    /**
     * 账户值
     */
    private String value;

    public AccountInfo() {
    }

    public AccountInfo(Integer type, String value) {
        this.type = type;
        this.value = value;
    }

    @Override
    public String toString() {
        return AccountType.getByCode(type).desc() + ":" + value;
    }

    public static List<AccountInfo> parseList(String accountInfo) {
        if (StringUtils.isBlank(accountInfo)) {
            return Collections.emptyList();
        }
        List<AccountInfo> list = new ArrayList<>();
        JSONArray accountArray = JSONArray.parseArray(accountInfo);
        accountArray.stream().forEach(account -> {
            int type = ((JSONObject) account).getIntValue("type");
            String value = ((JSONObject) account).getString("value");
            list.add(new AccountInfo(AccountType.getByCode(type).code(), value));
        });
        return list;
    }

    public static Map<Integer, AccountInfo> parseMap(String accountInfo) {
        if (StringUtils.isBlank(accountInfo)) {
            return Collections.emptyMap();
        }
        Map<Integer, AccountInfo> map = new HashMap<>();
        JSONArray accountArray = JSONArray.parseArray(accountInfo);
        accountArray.stream().forEach(account -> {
            int type = ((JSONObject) account).getIntValue("type");
            String value = ((JSONObject) account).getString("value");
            map.put(type, new AccountInfo(AccountType.getByCode(type).code(), value));
        });
        return map;
    }

    public static String parseString(String accountInfo) {
        if (StringUtils.isBlank(accountInfo)) {
            return StringUtils.EMPTY;
        }
        List<AccountInfo> list = parseList(accountInfo);
        return Joiner.on(", ").join(list);
    }

    public static String toJSONString(List<AccountInfo> accountInfoList) {
        if (CollectionUtils.isEmpty(accountInfoList)) {
            return StringUtils.EMPTY;
        }
        return JSONArray.toJSONString(accountInfoList);
    }

    public Integer getType() {
        return type;
    }

    public void setType(Integer type) {
        this.type = type;
    }

    public String getValue() {
        return value;
    }

    public void setValue(String value) {
        this.value = value;
    }
}
