package cn.com.duiba.goods.inner.api.dto;

import cn.com.duiba.goods.common.dto.AttrValue;
import org.apache.commons.collections4.CollectionUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;

/**
 * spu保存DTO
 * Created by guoyanfei .
 * 2022/11/21 .
 */
public class SpuSaveDTO implements Serializable {

    private static final long serialVersionUID = 6859927092649161591L;

    /**
     * SPU id
     */
    private Long id;

    /**
     * 供应商id
     */
    @NotNull(message = "【供应商id】不能为空")
    private Long supplierId;

    /**
     * 商品名称
     */
    @NotBlank(message = "【商品名称】不能为空")
    @Size(max = 40, message = "【商品名称】不能大于40位")
    private String spuName;

    /**
     * 品牌名
     */
    @NotBlank(message = "【品牌名】不能为空")
    @Size(max = 20, message = "【品牌名】不能大于20位")
    private String brandName;

    /**
     * 类目id
     */
    @NotNull(message = "【类目id】不能为空")
    private Long categoryId;

    /**
     * 商品类型，1实物，2虚拟
     * {@code @cn.com.duiba.goods.common.enums.goods.SpuType}
     */
    private Integer spuType;

    /**
     * 商品主图链接，多张图用英文 逗号 分割
     */
    @NotBlank(message = "【商品主图】不能为空")
    private String mainImgs;

    /**
     * 商品详情图链接，多张图用英文 逗号 分割
     */
    @NotBlank(message = "【商品详情图】不能为空")
    private String detailImgs;

    /**
     * 税率，比如16%，那么存储1600
     */
    @NotNull(message = "【税率】不能为空")
    private Integer taxRate;

    /**
     * 商品描述
     */
    @NotBlank(message = "【商品描述】不能为空")
    private String spuDesc;

    /**
     * 69码
     */
    @NotNull(message = "【69码】不能为空")
    private String code69;

    /**
     * 商品资质图
     */
    private String credentialsImgs;

    /**
     * 商品物流信息
     */
    private GoodsExpress goodsExpress;

    /**
     * 规格明细
     */
    @NotEmpty(message = "【规格明细】不能为空")
    private List<Sku> skus;

    /**
     * 本次操作是创建操作
     *
     * @return
     */
    public boolean isCreateOp() {
        return this.id == null;
    }

    public Integer getMinSupplyPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> supplyPriceList = skus.stream().map(Sku::getSupplyPrice).sorted().collect(Collectors.toList());
        return supplyPriceList.get(0);
    }

    public Integer getMaxSupplyPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> supplyPriceList = skus.stream().map(Sku::getSupplyPrice).sorted().collect(Collectors.toList());
        return supplyPriceList.get(supplyPriceList.size() - 1);
    }

    public Integer getMinMarketPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> marketPriceList = skus.stream().map(Sku::getMarketPrice).sorted().collect(Collectors.toList());
        return marketPriceList.get(0);
    }

    public Integer getMaxMarketPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> marketPriceList = skus.stream().map(Sku::getMarketPrice).sorted().collect(Collectors.toList());
        return marketPriceList.get(marketPriceList.size() - 1);
    }

    public Integer getMinSellPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> sellPriceList = skus.stream().map(Sku::getSellPrice).sorted().collect(Collectors.toList());
        return sellPriceList.get(0);
    }

    public Integer getMaxSellPrice() {
        if (CollectionUtils.isEmpty(skus)) {
            return null;
        }
        List<Integer> sellPriceList = skus.stream().map(Sku::getSellPrice).sorted().collect(Collectors.toList());
        return sellPriceList.get(sellPriceList.size() - 1);
    }

    /**
     * 规格明细
     */
    public static class Sku implements Serializable {

        private static final long serialVersionUID = -3074185137528877718L;

        /**
         * SKU id，为空就是新建规格
         */
        public Long id;

        /**
         * 供应商的规格编码（用于展示）
         */
        @NotBlank(message = "【规格编码】不能为空")
        @Size(max = 20, message = "【规格编码】不能大于20位")
        private String supplierSkuCode;

        /**
         * 优先级最高的SKU图，是针对单个SKU单独设置的，没有针对上传，就没有值
         */
        private String skuImg;

        /**
         * 规格配置
         */
        @NotEmpty(message = "【规格配置】不能为空")
        private List<AttrValue> attrValueList;

        /**
         * 供货价，单位分，比如1元，数据存为100
         */
        @NotNull(message = "【供货价】不能为空")
        private Integer supplyPrice;

        /**
         * 市场指导价，划线价，单位分，比如1元，数据存为100
         */
        @NotNull(message = "【市场指导价】不能为空")
        private Integer marketPrice;

        /**
         * 建议售价，平台卖价，单位分，比如1元，数据存为100
         */
        @NotNull(message = "【建议售价】不能为空")
        private Integer sellPrice;

        /**
         * 修改前库存余量
         */
        @NotNull(message = "【修改前库存余量】不能为空")
        private Integer preRemaining;

        /**
         * 修改后库存余量
         */
        @NotNull(message = "【修改后库存余量】不能为空")
        private Integer postRemaining;

        public Long getId() {
            return id;
        }

        public void setId(Long id) {
            this.id = id;
        }

        public String getSupplierSkuCode() {
            return supplierSkuCode;
        }

        public void setSupplierSkuCode(String supplierSkuCode) {
            this.supplierSkuCode = supplierSkuCode;
        }

        public String getSkuImg() {
            return skuImg;
        }

        public void setSkuImg(String skuImg) {
            this.skuImg = skuImg;
        }

        public List<AttrValue> getAttrValueList() {
            return attrValueList;
        }

        public void setAttrValueList(List<AttrValue> attrValueList) {
            this.attrValueList = attrValueList;
        }

        public Integer getSupplyPrice() {
            return supplyPrice;
        }

        public void setSupplyPrice(Integer supplyPrice) {
            this.supplyPrice = supplyPrice;
        }

        public Integer getMarketPrice() {
            return marketPrice;
        }

        public void setMarketPrice(Integer marketPrice) {
            this.marketPrice = marketPrice;
        }

        public Integer getSellPrice() {
            return sellPrice;
        }

        public void setSellPrice(Integer sellPrice) {
            this.sellPrice = sellPrice;
        }

        public Integer getPreRemaining() {
            return preRemaining;
        }

        public void setPreRemaining(Integer preRemaining) {
            this.preRemaining = preRemaining;
        }

        public Integer getPostRemaining() {
            return postRemaining;
        }

        public void setPostRemaining(Integer postRemaining) {
            this.postRemaining = postRemaining;
        }
    }

    /**
     * 商品物流信息
     */
    public static class GoodsExpress implements Serializable {

        private static final long serialVersionUID = -2347079688489882828L;

        /**
         * 发货地
         */
        @NotBlank(message = "【发货地】不能为空")
        @Size(max = 20, message = "【发货地】不能大于20位")
        private String deliveryPlace;

        /**
         * 发货物流公司
         */
        @NotBlank(message = "【发货物流公司】不能为空")
        @Size(max = 20, message = "【发货物流公司】不能大于20位")
        private String deliveryExpressCompany;

        /**
         * 七天无理由，1支持，0不支持
         * {@code @cn.com.duiba.goods.common.enums.goods.SevenDaysNoReason}
         */
        @NotNull(message = "【是否支持七天无理由】不能为空")
        private Integer sevenDaysNoReason;

        /**
         * 退换货地址以及联系方式
         */
        @NotBlank(message = "【退换货地址以及联系方式】不能为空")
        @Size(max = 150, message = "【退换货地址以及联系方式】不能大于150位")
        private String details;

        public String getDeliveryPlace() {
            return deliveryPlace;
        }

        public void setDeliveryPlace(String deliveryPlace) {
            this.deliveryPlace = deliveryPlace;
        }

        public String getDeliveryExpressCompany() {
            return deliveryExpressCompany;
        }

        public void setDeliveryExpressCompany(String deliveryExpressCompany) {
            this.deliveryExpressCompany = deliveryExpressCompany;
        }

        public Integer getSevenDaysNoReason() {
            return sevenDaysNoReason;
        }

        public void setSevenDaysNoReason(Integer sevenDaysNoReason) {
            this.sevenDaysNoReason = sevenDaysNoReason;
        }

        public String getDetails() {
            return details;
        }

        public void setDetails(String details) {
            this.details = details;
        }
    }

    public String getCode69() {
        return code69;
    }

    public void setCode69(String code69) {
        this.code69 = code69;
    }

    public String getCredentialsImgs() {
        return credentialsImgs;
    }

    public void setCredentialsImgs(String credentialsImgs) {
        this.credentialsImgs = credentialsImgs;
    }

    public Long getSupplierId() {
        return supplierId;
    }

    public void setSupplierId(Long supplierId) {
        this.supplierId = supplierId;
    }

    public Long getCategoryId() {
        return categoryId;
    }

    public void setCategoryId(Long categoryId) {
        this.categoryId = categoryId;
    }

    public Integer getSpuType() {
        return spuType;
    }

    public void setSpuType(Integer spuType) {
        this.spuType = spuType;
    }

    public List<Sku> getSkus() {
        return skus;
    }

    public void setSkus(List<Sku> skus) {
        this.skus = skus;
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String getSpuName() {
        return spuName;
    }

    public void setSpuName(String spuName) {
        this.spuName = spuName;
    }

    public String getBrandName() {
        return brandName;
    }

    public void setBrandName(String brandName) {
        this.brandName = brandName;
    }

    public String getMainImgs() {
        return mainImgs;
    }

    public void setMainImgs(String mainImgs) {
        this.mainImgs = mainImgs;
    }

    public String getDetailImgs() {
        return detailImgs;
    }

    public void setDetailImgs(String detailImgs) {
        this.detailImgs = detailImgs;
    }

    public Integer getTaxRate() {
        return taxRate;
    }

    public void setTaxRate(Integer taxRate) {
        this.taxRate = taxRate;
    }

    public String getSpuDesc() {
        return spuDesc;
    }

    public void setSpuDesc(String spuDesc) {
        this.spuDesc = spuDesc;
    }

    public GoodsExpress getGoodsExpress() {
        return goodsExpress;
    }

    public void setGoodsExpress(GoodsExpress goodsExpress) {
        this.goodsExpress = goodsExpress;
    }
}
