package cn.com.duiba.goods.open.api.support;

import cn.com.duiba.goods.open.api.GoodsOpenApiProperties;
import cn.com.duiba.goods.open.api.constant.OpenApiConstant;
import cn.com.duiba.goods.open.api.exception.NoPermissionException;
import cn.com.duiba.goods.open.api.utils.SignTool;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;

/**
 * remote接口调用鉴权流程
 * Created by guoyanfei .
 * 2022/11/8 .
 */
@Aspect
public class OpenApiPermissionAspect {

    @Resource
    private GoodsOpenApiProperties goodsOpenApiProperties;

    @Pointcut("execution(public * cn.com.duiba.goods.open.api.remoteservice..*.*(..))")
    private void remoteCallOperation() {
    }

    @Around("remoteCallOperation()")
    public Object makePermission(ProceedingJoinPoint point) throws Throwable {
        // 判断是否是接口，经过测试，remote接口的实现类也会进入这段逻辑，所以需要区分一下
        // 如果是接口，说明是客户端调用，需要做后面流程
        // 如果不是接口，说明是服务端调用，直接略过
        if (!point.getSignature().getDeclaringType().isInterface()) {
            return point.proceed();
        }

        try {
            // 判断本地是否存在key、secret的配置，如果存在，那么拿来签名，并且设置到线程变量中
            if (goodsOpenApiProperties.isConfigExisted()) {
                long timestamp = System.currentTimeMillis();
                // 签名
                String sign = this.sign(timestamp);
                OpenApiPermissionContext.set(goodsOpenApiProperties.getApiKey(), timestamp, sign);
            }

            // 此时，内部应用调用，和openapi应用调用，都应该已经设置了线程变量
            // 还没有的话，不允许访问了
            if (!OpenApiPermissionContext.isExisted()) {
                throw new NoPermissionException("无权访问");
            }

            return point.proceed();
        } finally {
            OpenApiPermissionContext.remove();
        }
    }

    private String sign(long timestamp) {
        Map<String, String> params = new HashMap<>();
        params.put(OpenApiConstant.API_KEY, goodsOpenApiProperties.getApiKey());
        params.put(OpenApiConstant.API_SECRET, goodsOpenApiProperties.getApiSecret());
        params.put(OpenApiConstant.TIMESTAMP, timestamp + "");
        return SignTool.sign(params);
    }
}
