package cn.com.duiba.kjy.base.customweb.web.factory;

import cn.com.duiba.kjy.base.customweb.web.bean.ParameterBean;
import cn.com.duiba.kjy.base.customweb.web.bean.RequestTypeEnum;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMapping;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandler;
import cn.com.duiba.kjy.base.customweb.web.handler.mapping.controller.ControllerMappingHandlerBuilder;
import cn.com.duiba.kjy.base.customweb.web.handler.response.ResponseHandler;
import cn.com.duiba.kjy.base.customweb.web.processor.parameter.ParameterPostProcessor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.MethodParameter;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/3/23 7:44 下午
 */
@Slf4j
public class HandlerMappingBeanFactory {
    @Resource
    private ControllerMapping controllerMapping;

    private final List<ParameterPostProcessor> parameterPostProcessors;

    private final List<ResponseHandler> responseHandlers;

    public HandlerMappingBeanFactory(List<ParameterPostProcessor> parameterPostProcessors, List<ResponseHandler> responseHandlers) {
        this.parameterPostProcessors = parameterPostProcessors;
        this.responseHandlers = responseHandlers;
    }


    public void builderHandlerMappingAndRegister(Class<?> handlerClass, List<Method> handlerMethodList, Object bean){
        if (CollectionUtils.isEmpty(handlerMethodList)){
            return;
        }
        final RequestMapping requestMapping = handlerClass.getDeclaredAnnotation(RequestMapping.class);
        if (Objects.isNull(requestMapping)){
            log.error("there is no annotation of @RequestMapping in class {}! so skip register it!",handlerClass.getName());
            return;
        }
        final String startPath = getPathAndInsertSlashBefore(requestMapping.path(), handlerClass);
        for (Method method : handlerMethodList) {
            final ControllerMappingHandler controllerMappingHandler = buildOneHandlerMapping(handlerClass, startPath, method,bean);
            controllerMapping.registerHandler(controllerMappingHandler);
        }
    }

    private ControllerMappingHandler buildOneHandlerMapping(Class<?> handlerClass, String startPath, Method handlerMappingMethod, Object bean) {
        ControllerMappingHandlerBuilder controllerMappingHandlerBuilder = ControllerMappingHandlerBuilder.create(handlerClass, handlerMappingMethod);
        final GetMapping getAnnotation = handlerMappingMethod.getDeclaredAnnotation(GetMapping.class);
        if (Objects.nonNull(getAnnotation)){
            controllerMappingHandlerBuilder.setRequestType(RequestTypeEnum.GET);
            final String lastPath = getPathAndInsertSlashBefore(getAnnotation.value(), handlerClass);
            String fullPath = startPath + lastPath;
            controllerMappingHandlerBuilder.setUrl(fullPath);
        }else{
            final PostMapping postAnnotation = handlerMappingMethod.getDeclaredAnnotation(PostMapping.class);
            if (Objects.nonNull(postAnnotation)){
                final String lastPath = getPathAndInsertSlashBefore(postAnnotation.value(), handlerClass);
                String fullPath = startPath + lastPath;
                controllerMappingHandlerBuilder.setRequestType(RequestTypeEnum.POST);
                controllerMappingHandlerBuilder.setUrl(fullPath);
            }else{
                final RequestMapping requestMapping  = handlerMappingMethod.getDeclaredAnnotation(RequestMapping.class);
                if (Objects.nonNull(requestMapping)){
                    final String lastPath = getPathAndInsertSlashBefore(requestMapping.value(), handlerClass);
                    String fullPath = startPath + lastPath;
                    controllerMappingHandlerBuilder.setRequestType(RequestTypeEnum.ALL);
                    controllerMappingHandlerBuilder.setUrl(fullPath);
                }else{
                    return null;
                }
            }
        }
        final Class<?> returnType = handlerMappingMethod.getReturnType();
        controllerMappingHandlerBuilder.setReturnType(returnType);
        controllerMappingHandlerBuilder.setResponseHandler(getResponseHandler(handlerMappingMethod,returnType));
        controllerMappingHandlerBuilder.setHandlerObject(bean);
        fillParameterList(handlerClass, handlerMappingMethod, controllerMappingHandlerBuilder);
        return controllerMappingHandlerBuilder.build();
    }

    private void fillParameterList(Class<?> handlerClass, Method handlerMappingMethod, ControllerMappingHandlerBuilder controllerMappingHandlerBuilder) {
        final Parameter[] parameters = handlerMappingMethod.getParameters();
        List<ParameterBean> parameterBeans = new ArrayList<>();
        int index = 0;
        for (Parameter parameter : parameters) {
            ParameterBean parameterBean = new ParameterBean();
            final Class<?> type = parameter.getType();
            parameterBean.setType(type);
            parameterBean.setName(parameter.getName());
            parameterBean.setPrimitive(type.isPrimitive());
            parameterBean.setMethodParameter(new MethodParameter(handlerMappingMethod,index++));
            processorParameter(parameterBean, handlerClass, handlerMappingMethod);
            parameterBeans.add(parameterBean);
        }
        controllerMappingHandlerBuilder.setParamList(parameterBeans);
    }

    /**
     * build path
     * 开发者往往偷懒不写 / 此处补全。且顺道验证下messageMapping 的value，不允许多个路径！ 虽然spring支持，但咱不支持。多个路径，毫无意义。
     */
    private String getPathAndInsertSlashBefore(String[] value, Class<?> handlerClass) {
        if (Objects.isNull(value) || value.length == 0){
            return "";
        }
        if (value.length > 1){
            throw new UnsupportedOperationException("there is so many value of annotation mapping in class: "+handlerClass.getName());
        }
        String v = value[0];
        if (StringUtils.isNotBlank(v) && !v.startsWith("/")){
            return "/"+v;
        }
        return v;
    }

    private void processorParameter(ParameterBean parameterBean, Class<?> handlerBeanClass, Method method) {
        for (ParameterPostProcessor parameterPostProcessor : parameterPostProcessors) {
            parameterPostProcessor.postProcessorParameter(parameterBean,handlerBeanClass,method);
        }
    }


    private ResponseHandler getResponseHandler(Method method, Class<?> returnType){
        for (ResponseHandler responseHandler : responseHandlers) {
            if (responseHandler.canWrite(method,returnType)){
                return responseHandler;
            }
        }
        throw new UnsupportedOperationException("can not write this type "+returnType.getName()+"in method"+method.getName());
    }

}
