package cn.com.duiba.kjy.base.customweb.web.bean;

import cn.com.duiba.kjy.base.customweb.util.UrlHelper;
import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.entity.ContentType;
import org.jetbrains.annotations.Nullable;
import org.springframework.http.MediaType;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/3/26 11:00 上午
 */
@Slf4j
@Getter
@Setter
public class KjjHttpRequest {

    private final FullHttpRequest fullHttpRequest;
    private final URI uri;
    private final  MultiValueMap<String, String> paramMap = new LinkedMultiValueMap<>();
    private byte[] requestBody;
    private ContentType contentType;
    private MediaType mediaType;
    private boolean sync = true;
    private KjjHttpResponse response;
    private final ChannelHandlerContext context;
    private InetSocketAddress remoteAddress;
    private List<Cookie> cookies;
    private final Map<String,Object> attribute = new HashMap<>();

    public KjjHttpRequest(ChannelHandlerContext ctx, FullHttpRequest fullHttpRequest) {
        this.context = ctx;
        this.fullHttpRequest = fullHttpRequest;
        this.uri = getUri(fullHttpRequest);
        readyUri();
        readyBody();
        readyCookie();
        readyRemoteAddress();
    }

    private void readyCookie() {
        final String cookieString = getHeader(HttpHeaderNames.COOKIE.toString());
        if(StringUtils.isBlank(cookieString)){
            this.cookies = Collections.emptyList();
            return;
        }
        this.cookies = ServerCookieDecoder.STRICT.decodeAll(cookieString);
    }

    private void readyRemoteAddress() {
        final InetSocketAddress remoteAddress;
        final SocketAddress socketAddress = context.channel().remoteAddress();
        if (Objects.nonNull(socketAddress) && socketAddress instanceof InetSocketAddress){
            remoteAddress = (InetSocketAddress) socketAddress;
        }else{
            remoteAddress = null;
        }
        this.remoteAddress = remoteAddress;
    }

    private void readyUri() {
        if (Objects.nonNull(uri)){
            final String query = uri.getQuery();
            extractUrlParams(query);
        }
    }

    public HttpMethod method() {
        return fullHttpRequest.method();
    }

    public HttpHeaders headers() {
        return fullHttpRequest.headers();
    }

    public ByteBuf content() {
        return fullHttpRequest.content();
    }

    public String uri(){
        return fullHttpRequest.uri();
    }

    private void extractUrlParams(String paramsStr) {
        if (StringUtils.isBlank(paramsStr)) {
            return;
        }
        StringBuilder key = new StringBuilder();
        StringBuilder value = new StringBuilder();
        int pos = 0; //0:key 1:value
        for (char aChar : paramsStr.toCharArray()) {
            if (aChar == '='){
                pos=1;
            }else if(aChar == '&'){
                pos=0;
                addParameter(key.toString(),value.toString());
                key.delete(0,key.length());
                value.delete(0,value.length());
            }else{
                if (pos==0){
                    key.append(aChar);
                }else {
                    value.append(aChar);
                }
            }
        }
        addParameter(key.toString(),value.toString());
    }

    private void readyBody(){
        ByteBuf content = fullHttpRequest.content();
        if (content.isReadable()){
            requestBody = new byte[content.readableBytes()];
            content.readBytes(requestBody);
        }
    }

    private void addParameter(String key,String value){
        if (StringUtils.isNotBlank(key)){
            paramMap.add(key,value);
        }
    }

    @Nullable
    private URI getUri(FullHttpRequest fullHttpRequest) {
       return UrlHelper.decodeAndCleanUriString(fullHttpRequest);
    }

    public ContentType getContentType(){
        if (Objects.nonNull(contentType)){
            return contentType;
        }
        return contentType = ContentType.parse(headers().get(HttpHeaderNames.CONTENT_TYPE));
    }

    public MediaType getMediaType(){
        if (Objects.nonNull(mediaType)){
            return mediaType;
        }
        return mediaType = MediaType.parseMediaType(headers().get(HttpHeaderNames.CONTENT_TYPE));
    }

    public KjjContext startAsync(){
        this.sync = false;
        return new KjjContext(this,response);
    }

    public String getRequestURI() {
        return uri.getPath();
    }

    public String getMethod() {
        return fullHttpRequest.method().name();
    }

    public String getQueryString() {
        return uri.getQuery();
    }

    public String getHeader(String key) {
        return headers().get(key);
    }

    public String getIpAddr() {
        if (Objects.isNull(remoteAddress)){
            return null;
        }
        return remoteAddress.getAddress().getHostAddress();
    }

    public void addAttribute(String name, Object object){
        attribute.put(name,object);
    }

    public String getParameter(String parameterName) {
        return paramMap.getFirst(parameterName);
    }

    public Object getAttribute(String key) {
        return attribute.get(key);
    }

    @Override
    public String toString() {
        return "KjjHttpRequest{" +
                ", uri=" + uri +
                ", paramMap=" + paramMap +
                ", contentType=" + contentType +
                ", mediaType=" + mediaType +
                ", sync=" + sync +
                ", remoteAddress=" + getIpAddr() +
                ", cookies=" + cookies +
                ", attribute=" + attribute +
                '}';
    }
}
