package cn.com.duiba.kjy.base.customweb.web.filter;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.kjy.base.customweb.util.PerfTestUtil;
import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpRequest;
import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpResponse;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author dugq
 * @date 2021/3/30 5:53 下午
 */
@Slf4j
public class KjjAccessLogFilter implements CustomFilter{
    private static final String DONT_WRITE_CURRENT_ACCESSLOG="dont_write_current_accesslog";
    private static final Logger accessLog = LoggerFactory.getLogger("duiba_access_log");
    private static final String EX_ATTRIBUTE_KEY = "_accesslog_ex";
    private static final String APP_ID_ATTRIBUTE_KEY = "_accesslog_app_id";
    private static final String CONSUMER_ID_ATTRIBUTE_KEY = "_accesslog_consumer_id";

    @Value("${spring.application.name}")
    private String appName;


    @Override
    public void doFilter(CustomFilterChain customFilterChain, KjjHttpRequest httpRequest, KjjHttpResponse response) throws Throwable {
        long start = System.currentTimeMillis();
        try {
            customFilterChain.filter(httpRequest, response);
        }finally {
            doAfter(httpRequest, response, System.currentTimeMillis() - start);
        }
    }

    protected void doAfter(KjjHttpRequest req, KjjHttpResponse resp, long cost) {
        try {
            //判断是否性能压测url
            boolean isPerfUrl = isPerfTestRequest(req);

            if(!isPerfUrl && needLog(req)){
                processAccessLog(req,resp,cost);
            }
        }catch (Exception e){
            log.error("AccessLogFilter process error, message=", e);
        }
    }

    private void processAccessLog(KjjHttpRequest req, KjjHttpResponse resp,long cost) throws Exception{//NOSONAR
        //只记录get post请求
        String method = StringUtils.lowerCase(getMethod(req));
        if(!"get".equals(method) && !"post".equals(method)){
            return;
        }

        Map<String, Object> map = new HashMap<>();
        map.put("app_name", appName);//当前应用名称
        map.put("url_host", getHeader(req, "host"));
        map.put("url_path", getRequestURI(req));
        map.put("url_query", getQueryString(req));

        map.put("http_method",getMethod(req));
        map.put("rc",getStatus(resp));
        map.put("rt",cost);
        map.put("mip", NetUtils.getLocalIp());//记录服务器IP
        putIfNotNull(map, "ex", getExPair(req));

        //如果url中有callback参数,认为此url是一个JSONP请求,标记为POST类型,不作为PV统计
        String callback=getParameter(req, "callback");
        if(!StringUtils.isBlank(callback)){
            map.put("http_method","POST");
        }

        //优先从线程变量中取数据
        Long consumerId = getConsumerId(req);

        Long appId = getAppId(req);

        putIfNotNull(map, "consumer_id", consumerId);

        putIfNotNull(map, "app_id", appId);

        putUa(req, map);

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        map.put("time", sdf.format(new Date()));
        String referer = getHeader(req, "referer");
        putIfNotNull(map, "referer", referer);
        String ip = getIpAddr(req);
        putIfNotNull(map, "ip", ip);

        accessLog.info(JSON.toJSONString(map));
    }


    /**
     * 添加额外信息到线程变量,日志从线程变量获取信息进行打压输出
     * @param key
     * @param value
     */
    public void putExPair(KjjHttpRequest req, String key, Object value){
        ensureSafeCall();
        Map<String, Object> map = (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, EX_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }


    private Map<String, Object> getExPair(KjjHttpRequest req){
        return (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
    }

    /**
     * 设置consumerId
     * @param req
     * @param consumerId
     */
    public void setConsumerId(KjjHttpRequest req, Long consumerId) {
        addAttribute(req, CONSUMER_ID_ATTRIBUTE_KEY, consumerId);
    }

    public void setAppId(KjjHttpRequest request, Long sellerId) {
        addAttribute(request, APP_ID_ATTRIBUTE_KEY, sellerId);
    }

    private Long getAppId(KjjHttpRequest req) {
        return (Long)getAttribute(req, APP_ID_ATTRIBUTE_KEY);
    }

    private Long getConsumerId(KjjHttpRequest req){
        return (Long)getAttribute(req, CONSUMER_ID_ATTRIBUTE_KEY);
    }

    private void putUa(KjjHttpRequest req, Map<String, Object> map){
        String ua = StringUtils.trimToEmpty(getHeader(req, "user-agent"));
        if (ua.length() > 500) {
            ua = ua.substring(0, 499);
        }
        if (!StringUtils.isEmpty(ua)) {
            map.put("user_agent", ua);
        }
    }

    private void putIfNotNull(Map<String, Object> map, String key, Object value){
        if(value != null){
            map.put(key, value);
        }
    }

    private boolean needLog(KjjHttpRequest req){
        Boolean val = (Boolean)getAttribute(req, DONT_WRITE_CURRENT_ACCESSLOG);
        return val == null || !val;
    }

    protected String getRequestURI(KjjHttpRequest req) {
        return req.getRequestURI();
    }

    protected String getMethod(KjjHttpRequest req) {
        return req.getMethod();
    }

    protected String getQueryString(KjjHttpRequest req) {
        return req.getQueryString();
    }

    protected String getHeader(KjjHttpRequest req, String key) {
        return req.getHeader(key);
    }

    protected String getIpAddr(KjjHttpRequest req) {
        return req.getIpAddr();
    }

    protected boolean isPerfTestRequest(KjjHttpRequest req) {
        return PerfTestUtil.isPerfTestRequest(req);
    }

    protected void addAttribute(KjjHttpRequest req, String key, Object value) {
        req.addAttribute(key,value);
    }

    protected Object getAttribute(KjjHttpRequest req, String key) {
        return req.getAttribute(key);
    }

    protected String getParameter(KjjHttpRequest req, String key) {
        return req.getParameter(key);
    }


    protected int getStatus(KjjHttpResponse resp) {
        return resp.getResponse().status().code();
    }

    protected void ensureSafeCall() {
        String threadName = StringUtils.defaultString(Thread.currentThread().getName());
        if (!threadName.startsWith("netty")){
            log.error("can not use accessLogger out of netty work thread!");
        }
    }
}
