package cn.com.duiba.kjy.base.api.utils.login;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.kjy.base.api.bean.login.LoginBean;
import cn.com.duiba.kjy.base.api.enums.login.UserSourceEnum;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.util.Asserts;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * <h1>登陆状态还原方法</h1>
 *  <h2>使用步骤</h2>
 *  <li>新建一个loginInterceptor</li>
 *  <li>在loginInterceptor的preHandle方中调用：{@link #initHttp(HttpServletRequest, HttpServletResponse,UserSourceEnum...)}</li>
 *  <li>在loginInterceptor的afterCompletion方中调用：{@link #clearThreadLocally}</li>
 *  <li>之后在tomcat线程中，可利用{@link #isLogin(UserSourceEnum)} {@link #getUserInfo(UserSourceEnum)} 快速查询用户登陆信息</li>
 *
 *  <h2>其他建议</h2>
 *  <li>根据系统需求，UserSourceEnum 可多可少</li>
 *  <li>对于只限于一种UserSourceEnum的系统，建议新建一个RequestUtil对BaseRequestUtil进行封装，
 *  在RequestUtil中利用{@link #isLogin(UserSourceEnum)} 和 {@link #getUserInfo(UserSourceEnum)} 方法提供更多更好的方法</li>
 *
 */
@Slf4j
public class BaseRequestUtil {

    private static ThreadLocal<HttpRequestDto> local = new ThreadLocal<>();


    private BaseRequestUtil() {
        //to do something
    }


    /**
     * springMVC loginInterceptor 中初始化 用户信息
     */
    public static void initHttp(HttpServletRequest request, HttpServletResponse response,UserSourceEnum... userSourceEnums) {
        clearThreadLocally();//先清理
        Asserts.notNull(request, "RequestTool注入request为空");
        Asserts.notNull(response, "RequestTool注入response为空");
        get().initTokenUser(request, response,userSourceEnums);
    }

    /**
     * 清空线程本地变量,必须在处理完用户请求之后调用（一般是在LoginFilter调用链结束时）
     */
    public static void clearThreadLocally() {
        local.remove();
    }

    /**
     * 获得当前登录的用户的ip
     */
    public static String getIp() {
        return get().ip;
    }

    /**
     * 用户是否登录
     * @return true=已登录, false=未登录
     */
    public static boolean isLogin(UserSourceEnum userSourceEnum) {
        return get().getLoginStatus().getOrDefault(userSourceEnum,false);
    }

    /**
     * 获取当前登陆用户的uid
     * @return live_user表的ID
     */
    public static <T> T getUserInfo(UserSourceEnum userSourceEnum){
        if (!isLogin(userSourceEnum)){
            return null;
        }
        final HttpRequestDto httpRequestDto = get();
        return (T) httpRequestDto.getLoginBean().get(userSourceEnum);
    }

    public static HttpServletRequest getRequest() {
        return get().getRequest();
    }

    /**
     * 获取当前线程中存储的登陆信息
     */
    private static HttpRequestDto get() {
        HttpRequestDto httpRequestDto = local.get();
        if (null == httpRequestDto) {
            httpRequestDto = new HttpRequestDto();
            local.set(httpRequestDto);
        }
        return httpRequestDto;
    }

    @Data
    private static class HttpRequestDto{
        private String ip;
        private HttpServletRequest request;
        private HttpServletResponse response;
        private Map<UserSourceEnum, LoginBean> loginBean = new HashMap<>();
        private Map<UserSourceEnum, Boolean> loginStatus= new HashMap<>();

        /**
         * springMVC 请求 尝试从request中恢复用户信息
         *
         */
        void initTokenUser(HttpServletRequest request, HttpServletResponse response,UserSourceEnum... userSourceEnums) {
            this.request = request;
            this.response = response;
            if(ArrayUtils.isEmpty(userSourceEnums)){
                init(request,UserSourceEnum.values());
                return;
            }
            init(request,userSourceEnums);

        }
        /**
         * 在interceptor中执行
         */
        private void init(HttpServletRequest request,UserSourceEnum... supportUserSource) {
            this.ip = RequestTool.getIpAddr(request);
            for (UserSourceEnum userSourceEnum : supportUserSource) {
                if (Objects.isNull(userSourceEnum)){
                    continue;
                }
                final LoginBean loginBean = rollbackEncodeToken(request, userSourceEnum);
                this.loginBean.put(userSourceEnum, loginBean);
                loginStatus.put(userSourceEnum,Objects.nonNull(loginBean));
            }
        }

        private LoginBean rollbackEncodeToken(HttpServletRequest request,UserSourceEnum userSourceEnum){
            String encodeString  = RequestTool.getCookie(request,userSourceEnum.getCookieName());
            if (StringUtils.isBlank(encodeString)){
                encodeString  = request.getHeader(userSourceEnum.getCookieName());
            }
            if (!SpringEnvironmentUtils.isProdEnv()){
                encodeString  = request.getParameter(userSourceEnum.getCookieName());
            }
            if (StringUtils.isBlank(encodeString)){
                return null;
            }
            final LoginBean loginBean = TokenUtils.decodeToken(encodeString, userSourceEnum.getBeanClass());
            if (Objects.isNull(loginBean)){
                return null;
            }
            if (loginBean.getDisableTime()<System.currentTimeMillis()){
                return null;
            }
            return loginBean;
        }
    }


}
