package cn.com.duiba.kjy.base.reactive.convert;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.annotation.Order;
import org.springframework.validation.BindException;
import org.springframework.validation.Errors;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.method.annotation.MethodArgumentConversionNotSupportedException;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.reactive.BindingContext;
import org.springframework.web.reactive.result.method.HandlerMethodArgumentResolver;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Created by dugq on 2019-05-13.
 * 解析get方式提交参数中的加密ID
 */
@Order(Integer.MIN_VALUE)
@Slf4j
public class IdArgumentResolver extends IdDecodeBean implements HandlerMethodArgumentResolver {

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return (parameter.hasParameterAnnotation(IdDecode.class)  //基本类型
                || parameter.getParameterType().isAnnotationPresent(IdDecode.class))  //对象接受
                && !parameter.getParameterType().isAnnotationPresent(RequestBody.class)  //非post
                ;
    }

    @Override
    public Mono<Object> resolveArgument(MethodParameter parameter, BindingContext bindingContext, ServerWebExchange exchange) {
        final Object data;
        try {
            data = resolverParameterInRequestUrl(parameter, exchange, bindingContext);
            return Mono.justOrEmpty(data);
        } catch (Exception e) {
            log.error("-------------",e);
            return Mono.empty();
        }
    }


    private Object resolverParameterInRequestUrl(MethodParameter parameter, ServerWebExchange webRequest, BindingContext binderFactory) throws Exception {
        Class<?> parameterType = parameter.getParameterType();
        if(isPrimitiveOrString(parameterType)){    //参数 是 基本类型或者string
            return resolverPrimitiveAndStringValue(parameter, webRequest);
        }else if(isArray(parameterType) || isCollection(parameterType)){
            String parameterName = parameter.getParameterName();
            List<String> parameterValues = webRequest.getRequest().getQueryParams().get(parameterName);
            if(Objects.isNull(parameterValues)){
                return null;
            }
            IdDecode parameterAnnotation = parameter.getParameterAnnotation(IdDecode.class);
            List<Long> ids = parameterValues.stream().map(value -> decodeId(parameterAnnotation, value, Long.class,parameterName)).collect(Collectors.toList());
            if(isArray(parameterType)){
                return ids.toArray(new Long[0]);
            }
            if(Set.class.isAssignableFrom(parameterType)){
                return new HashSet<>(ids);
            }
            return ids;
        }
        else{
            List<Field> declaredFields = getAllFields(parameterType);
            Object instance = parameterType.newInstance();
            for(Field field : declaredFields){  //遍历属性赋值
                Object fieldValue = webRequest.getRequest().getQueryParams().getFirst(field.getName()); //获取参数值
                if(Objects.nonNull(fieldValue)){
                    resolverObjectFiled(parameter, webRequest, binderFactory, instance, field);//填充值
                }
            }
            validatorIfNecessary(parameter, webRequest, binderFactory, instance); //spring validator
            return instance;
        }
    }

    //spring validator
    private void validatorIfNecessary(MethodParameter parameter, ServerWebExchange webRequest, BindingContext binderFactory, Object instance) throws Exception {
        WebDataBinder binder = binderFactory.createDataBinder(webRequest, instance, parameter.getParameterName());
        if (binder.getTarget() != null) {
            validateIfApplicable(binder, parameter);
            if (binder.getBindingResult().hasErrors() && isBindExceptionRequired(parameter)) {
                throw new BindException(binder.getBindingResult());
            }
        }
    }

    //解析 object 的field
    private void resolverObjectFiled(MethodParameter parameter, ServerWebExchange webRequest, BindingContext binderFactory, Object instance, Field field) throws Exception {
        IdDecode annotation = field.getAnnotation(IdDecode.class);
        if(isPrimitiveOrString(field.getType()) && Objects.nonNull(annotation)) {
            String fieldValue = webRequest.getRequest().getQueryParams().getFirst(field.getName());
            Long id = decodeId(annotation, fieldValue, parameter.getParameterType(),field.getName());
            WebDataBinder binder = binderFactory.createDataBinder(webRequest, field.getName());
            Object valueObj = binder.convertIfNecessary(id, field.getType(), field);
            field.setAccessible(true);
            field.set(instance, valueObj);
        }else{
            Object fieldValue = webRequest.getRequest().getQueryParams().getFirst(field.getName());
            field.setAccessible(true);
            if (binderFactory != null) {
                WebDataBinder binder = binderFactory.createDataBinder(webRequest, null, field.getName());
                try {
                    fieldValue = binder.convertIfNecessary(fieldValue, field.getType(), field);
                    field.set(instance,fieldValue);
                }
                catch (ConversionNotSupportedException ex) {
                    throw new MethodArgumentConversionNotSupportedException(fieldValue, ex.getRequiredType(),
                            field.getName(), parameter, ex.getCause());
                }
                catch (TypeMismatchException ex) {
                    throw new MethodArgumentTypeMismatchException(fieldValue, ex.getRequiredType(),
                            field.getName(), parameter, ex.getCause());
                }
            }else{
                log.error("binderFactory is null ");
            }

        }
    }

    private Object resolverPrimitiveAndStringValue(MethodParameter parameter, ServerWebExchange webRequest) {
        String parameterName = parameter.getParameterName();
        String value = webRequest.getRequest().getQueryParams().getFirst(parameterName);
        IdDecode parameterAnnotation = parameter.getParameterAnnotation(IdDecode.class);
        return decodeId(parameterAnnotation, value,parameter.getParameterType(),parameterName);
    }


    private boolean isBindExceptionRequired( MethodParameter parameter) {
        int i = parameter.getParameterIndex();
        Class<?>[] paramTypes = parameter.getMethod().getParameterTypes();
        boolean hasBindingResult = (paramTypes.length > (i + 1) && Errors.class.isAssignableFrom(paramTypes[i + 1]));
        return !hasBindingResult;
    }

    private void validateIfApplicable(WebDataBinder binder, MethodParameter parameter) {
        Annotation[] annotations = parameter.getParameterAnnotations();
        for (Annotation ann : annotations) {
            Validated validatedAnn = AnnotationUtils.getAnnotation(ann, Validated.class);
            if (validatedAnn != null || ann.annotationType().getSimpleName().startsWith("Valid")) {
                Object hints = (validatedAnn != null ? validatedAnn.value() : AnnotationUtils.getValue(ann));
                Object[] validationHints = (hints instanceof Object[] ? (Object[]) hints : new Object[] {hints});
                binder.validate(validationHints);
                break;
            }
        }
    }

}
