package cn.com.duiba.kjy.base.customweb.web.condition;

import cn.com.duiba.kjy.base.customweb.web.bean.KjjHttpRequest;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * @author dugq
 * @date 2021/4/1 4:45 下午
 */
public class PatternsRequestCondition extends AbstractRequestCondition<PatternsRequestCondition> {

    private final Set<String> patterns;

    private final PathMatcher pathMatcher;

    private final boolean useTrailingSlashMatch = true;

    /**
     * Creates a new instance with the given URL patterns. Each pattern that is
     * not empty and does not start with "/" is prepended with "/".
     * @param patterns 0 or more URL patterns; if 0 the condition will match to
     * every request.
     */
    public PatternsRequestCondition(String... patterns) {
        this(Arrays.asList(patterns), null);
    }

    public PatternsRequestCondition(String[] patterns,
                                    @Nullable PathMatcher pathMatcher) {

        this(Arrays.asList(patterns), pathMatcher);
    }

    /**
     * Private constructor accepting a collection of patterns.
     */
    private PatternsRequestCondition(Collection<String> patterns, @Nullable PathMatcher pathMatcher) {
        this.patterns = Collections.unmodifiableSet(prependLeadingSlash(patterns));
        this.pathMatcher = pathMatcher != null ? pathMatcher : new AntPathMatcher();
    }

    /**
     * Private constructor for use when combining and matching.
     */
    private PatternsRequestCondition(Set<String> patterns, PatternsRequestCondition other) {
        this.patterns = patterns;
        this.pathMatcher = other.pathMatcher;
    }


    private static Set<String> prependLeadingSlash(Collection<String> patterns) {
        if (patterns.isEmpty()) {
            return Collections.singleton("");
        }
        Set<String> result = new LinkedHashSet<>(patterns.size());
        for (String pattern : patterns) {
            if (StringUtils.hasLength(pattern) && !pattern.startsWith("/")) {
                pattern = "/" + pattern;
            }
            result.add(pattern);
        }
        return result;
    }

    public Set<String> getPatterns() {
        return this.patterns;
    }

    @Override
    protected Collection<String> getContent() {
        return this.patterns;
    }

    @Override
    protected String getToStringInfix() {
        return " || ";
    }

    /**
     * Returns a new instance with URL patterns from the current instance ("this") and
     * the "other" instance as follows:
     * <ul>
     * <li>If there are patterns in both instances, combine the patterns in "this" with
     * the patterns in "other" using {@link PathMatcher#combine(String, String)}.
     * <li>If only one instance has patterns, use them.
     * <li>If neither instance has patterns, use an empty String (i.e. "").
     * </ul>
     */
    @Override
    public PatternsRequestCondition combine(PatternsRequestCondition other) {
        Set<String> result = new LinkedHashSet<>();
        if (!this.patterns.isEmpty() && !other.patterns.isEmpty()) {
            for (String pattern1 : this.patterns) {
                for (String pattern2 : other.patterns) {
                    result.add(this.pathMatcher.combine(pattern1, pattern2));
                }
            }
        }
        else if (!this.patterns.isEmpty()) {
            result.addAll(this.patterns);
        }
        else if (!other.patterns.isEmpty()) {
            result.addAll(other.patterns);
        }
        else {
            result.add("");
        }
        return new PatternsRequestCondition(result, this);
    }

    /**
     * Checks if any of the patterns match the given request and returns an instance
     * that is guaranteed to contain matching patterns, sorted via
     * {@link PathMatcher#getPatternComparator(String)}.
     * <p>A matching pattern is obtained by making checks in the following order:
     * <ul>
     * <li>Direct match
     * <li>Pattern match with ".*" appended if the pattern doesn't already contain a "."
     * <li>Pattern match
     * <li>Pattern match with "/" appended if the pattern doesn't already end in "/"
     * </ul>
     * @param request the current request
     * @return the same instance if the condition contains no patterns;
     * or a new condition with sorted matching patterns;
     * or {@code null} if no patterns match.
     */
    @Override
    @Nullable
    public boolean getMatchingCondition(KjjHttpRequest request) {
        if (this.patterns.isEmpty()) {
            return false;
        }
        String lookupPath = request.getRequestURI();
        List<String> matches = getMatchingPatterns(lookupPath);
        return CollectionUtils.isNotEmpty(matches);
    }

    /**
     * Find the patterns matching the given lookup path. Invoking this method should
     * yield results equivalent to those of calling {@link #getMatchingCondition}.
     * This method is provided as an alternative to be used if no request is available
     * (e.g. introspection, tooling, etc).
     * @param lookupPath the lookup path to match to existing patterns
     * @return a collection of matching patterns sorted with the closest match at the top
     */
    public List<String> getMatchingPatterns(String lookupPath) {
        List<String> matches = null;
        for (String pattern : this.patterns) {
            String match = getMatchingPattern(pattern, lookupPath);
            if (match != null) {
                matches = matches != null ? matches : new ArrayList<>();
                matches.add(match);
            }
        }
        if (matches == null) {
            return Collections.emptyList();
        }
        if (matches.size() > 1) {
            matches.sort(this.pathMatcher.getPatternComparator(lookupPath));
        }
        return matches;
    }

    @Nullable
    private String getMatchingPattern(String pattern, String lookupPath) {
        if (pattern.equals(lookupPath)) {
            return pattern;
        }
        if (this.pathMatcher.match(pattern, lookupPath)) {
            return pattern;
        }
        if (this.useTrailingSlashMatch) {
            if (!pattern.endsWith("/") && this.pathMatcher.match(pattern + "/", lookupPath)) {
                return pattern + "/";
            }
        }
        return null;
    }

    /**
     * Compare the two conditions based on the URL patterns they contain.
     * Patterns are compared one at a time, from top to bottom via
     * {@link PathMatcher#getPatternComparator(String)}. If all compared
     * patterns match equally, but one instance has more patterns, it is
     * considered a closer match.
     * <p>It is assumed that both instances have been obtained via
     *  to ensure they
     * contain only patterns that match the request and are sorted with
     * the best matches on top.
     */
    @Override
    public int compareTo(PatternsRequestCondition other, KjjHttpRequest request) {
        String lookupPath =request.getRequestURI();
        Comparator<String> patternComparator = this.pathMatcher.getPatternComparator(lookupPath);
        Iterator<String> iterator = this.patterns.iterator();
        Iterator<String> iteratorOther = other.patterns.iterator();
        while (iterator.hasNext() && iteratorOther.hasNext()) {
            int result = patternComparator.compare(iterator.next(), iteratorOther.next());
            if (result != 0) {
                return result;
            }
        }
        if (iterator.hasNext()) {
            return -1;
        }
        else if (iteratorOther.hasNext()) {
            return 1;
        }
        else {
            return 0;
        }
    }

}
