package cn.com.duiba.kjy.base.customweb.web.processor;

import cn.com.duiba.kjy.base.customweb.autoconfig.EnableKjjNettyServer;
import cn.com.duibaboot.ext.autoconfigure.data.redis.DuibaRedisCondition;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertySource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/6/8 1:45 下午
 * {@link cn.com.duibaboot.ext.autoconfigure.config.DuibaEnvironmentPostProcessor}
 */
public class KjjEnvironmentPostProcessor implements EnvironmentPostProcessor {

    @VisibleForTesting
    public static final Map<String, String> defProperties;
    @VisibleForTesting
    public static final String DUIBA_BOOT_DEFAULT_CONFIG = "duibaBootDefaultConfig";

    static {
        Map<String, String> overrideProperties = new HashMap<>();
        //配置禁用netflix servo监控
        overrideProperties.put("spring.metrics.servo.enabled", "false");
        overrideProperties.put("spring.metrics.export.enabled", "false");
        //启动时初始化ribbon
        overrideProperties.put("ribbon.eager-load.enabled", "true");
        //禁用一些health检查，以避免类似【elasticsearch不可用导致整个服务不可用】的问题
        overrideProperties.put("management.health.elasticsearch.enabled", "false");
        overrideProperties.put("management.health.mail.enabled", "false");
        overrideProperties.put("management.health.cassandra.enabled", "false");
        overrideProperties.put("management.health.couchbase.enabled", "false");
        overrideProperties.put("management.health.mongo.enabled", "false");
        overrideProperties.put("management.health.rabbit.enabled", "false");
        overrideProperties.put("management.health.redis.enabled", "false");
        overrideProperties.put("management.health.solr.enabled", "false");
        overrideProperties.put("management.health.hazelcast.enabled", "false");

        //强制CharacterEncodingFilter对response也编码（默认只会对request编码）
        overrideProperties.put("spring.http.encoding.force", "true");
        //让zipkin客户端使用http方式发送消息（显示指定，以防止客户端引入kafka时自动使用kafka来发送）
        overrideProperties.put("spring.zipkin.sender.type", "web");
        //不自动注册，我们自己来注册到eureka上。
        overrideProperties.put("spring.cloud.service-registry.auto-registration.enabled", "false");
        //springmvc/webflux出错时返回的消息里也带上exception字段（这个字段表示异常类的类名，在spring-cloud接口中很有用，advancedFeignClient依赖这个字段做些特殊处理）
        overrideProperties.put("server.error.include-exception", "true");
        //暴露所有endpoint到web端口上
        overrideProperties.put("management.endpoints.web.exposure.include", "*");
        //sleuth/brave使用，表示X-B3-Flags这个字段可以跨span传输
        overrideProperties.put("spring.sleuth.baggage-keys", "X-B3-Flags");
        //spring-boot2修改了jackson的默认配置，会把date类型转换为字符串输出，（1.*默认行为是输出为long），加上如下配置来保持1.*时代的设置：
        overrideProperties.put("spring.jackson.serialization.write-dates-as-timestamps", "true");

        //自动识别X-Forwarded-For and X-Forwarded-Proto头.
        overrideProperties.put("server.use-forward-headers", "true");
        //取消自动把com.zaxxer.hikari.HikariDataSource bean转化为refresh scope的动作
        overrideProperties.put("spring.cloud.refresh.refreshable", "");

        //禁止刷新eurekaClient，以防止刷新时应用从eureka服务端短暂消失导致问题。
        overrideProperties.put("eureka.client.refresh.enable", "false");

        //不支持refresh的bean
        overrideProperties.put("spring.cloud.refresh.never-refreshable", "com.zaxxer.hikari.HikariDataSource,cn.com.duibaboot.ext.autoconfigure.perftest.PerfTestRoutingDataSourceForHikari,org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory");

        defProperties = Collections.unmodifiableMap(overrideProperties);
    }

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        final Class<?> mainClass = application.getMainApplicationClass();
        if(Objects.isNull(mainClass) || !mainClass.isAnnotationPresent(EnableKjjNettyServer.class)){//判断到是bootstrap的application则跳过
            return;
        }
        Map<String, Object> properties = new HashMap<>();
        properties.putAll(defProperties);

        PropertySource<?> propertySource = new MapPropertySource(DUIBA_BOOT_DEFAULT_CONFIG,
                properties);
        //最后的优先级最低
        environment.getPropertySources().addLast(propertySource);

        configRedisProperties(environment, properties);
        configEurekaProperties(environment, properties);
    }

    private void configRedisProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        //禁用Spring官方的 RedisAutoConfiguration。
        String property = "spring.autoconfigure.exclude";
        String excludeAutoConfigClasses = StringUtils.defaultString(environment.getProperty(property), null);
        List<String> excludeList = new ArrayList<>();
        if(StringUtils.isNotBlank(excludeAutoConfigClasses)) {
            excludeList.add(excludeAutoConfigClasses);
        }
        excludeList.add(RedisAutoConfiguration.class.getName());
        excludeList.add("org.springframework.boot.actuate.autoconfigure.hazelcast.HazelcastHealthContributorAutoConfiguration");
        //有多个redisConnectionFactory的情况下RedisReactiveAutoConfiguration会报错，先禁用
        excludeList.add("org.springframework.boot.autoconfigure.data.redis.RedisReactiveAutoConfiguration");
        excludeAutoConfigClasses = Joiner.on(",").skipNulls().join(excludeList);
        properties.put(property, excludeAutoConfigClasses);

        if (!DuibaRedisCondition.isRedisAutoConfigEnable(environment)) {
            //禁用 RedisRepositoriesAutoConfiguration
            properties.put("spring.data.redis.repositories.enabled", "false");
        }
    }

    /**
     * 如果设置了contextpath，则设置默认的eureka相关属性（会注册到eureka元数据中）
     * @param environment
     * @param properties
     */
    private void configEurekaProperties(ConfigurableEnvironment environment, Map<String, Object> properties){
        String contextPath = environment.getProperty("server.context-path");
        if(StringUtils.isBlank(contextPath)){
            contextPath = environment.getProperty("server.contextPath");
        }
        if(!StringUtils.isBlank(contextPath)) {
            properties.put("eureka.instance.home-page-url-path", contextPath);
//            properties.put("eureka.instance.status-page-url-path", contextPath + "/info");
//            properties.put("eureka.instance.health-check-url-path", contextPath + "/health");
        }
    }
}
