package cn.com.duiba.kjy.base.api.queue;

import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ExecutorService;

/**
 * 聚合http请求，批量处理，用线程池消费，子类不需要再引入线程池
 * @author lizhi
 * @date 2020/4/5 11:40 AM
 */
@Slf4j
public abstract class AbstractAsyncHttpQueueService<T extends HttpRequestBaseBean> extends AbstractHttpQueueService<T> {

    @Resource
    private ExecutorService executorService;

    /**
     * 构造一个支持压测、并发、批量消费的Queue
     *
     * @param capacity          BlockingQueue的最大允许容量, 如果<=0 视为无界队列
     * @param batchSize         一次批量消费的个数
     * @param maxWaitSize       一次批量消费的阈值，当已入队的数据个数达到此阈值时，触发批量消费（若消费回调阻塞，则无法消费）
     *                          如果此阈值 <= 0则不会触发个数阈值消费（这种情况下数据只会在时间到达maxWaitTimeMillis阈值后才会被消费，这意味着数据可能会在队列中存在比较长的时间）
     * @param maxWaitTimeMillis 数据最大延迟消费时间的阈值(ms)，当有数据在队列中存在时间超过此阈值时，触发批量消费（若消费回调阻塞，则无法消费）。
     */
    public AbstractAsyncHttpQueueService(int capacity, int batchSize, int maxWaitSize, int maxWaitTimeMillis) {
        super(capacity, batchSize, maxWaitSize, maxWaitTimeMillis);
    }

    @Override
    protected void batchConsume(List<T> list) {
        executorService.execute(()-> {
            try {
                CatUtils.executeInCatTransaction(() -> {
                    batchConsumeHttpMsg(list);
                    return null;
                }, "AsyncContext", this.getClass().getName());
            } catch (Throwable e) {
                log.error("batchConsumeHttpMsg, class={}, e:", this.getClass().getName(), e);
            }
        });
    }
}
