package cn.com.duiba.kjy.base.api.queue;

import cn.com.duiba.kjy.base.api.utils.NumberUtil;
import cn.com.duiba.kjy.base.exception.errorcode.BaseErrorCode;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 聚合http请求，直播维度批量处理
 * @author lizhi
 * @date 2020/4/5 11:40 AM
 */
@Slf4j
public abstract class AbstractHttpQueueDealWithLiveService<T extends HttpRequestBaseBean> extends AbstractHttpQueueService<T> {


    /**
     * 构造一个支持压测、并发、批量消费的Queue
     *
     * @param capacity          BlockingQueue的最大允许容量, 如果<=0 视为无界队列
     * @param batchSize         一次批量消费的个数
     * @param maxWaitSize       一次批量消费的阈值，当已入队的数据个数达到此阈值时，触发批量消费（若消费回调阻塞，则无法消费）
     *                          如果此阈值 <= 0则不会触发个数阈值消费（这种情况下数据只会在时间到达maxWaitTimeMillis阈值后才会被消费，这意味着数据可能会在队列中存在比较长的时间）
     * @param maxWaitTimeMillis 数据最大延迟消费时间的阈值(ms)，当有数据在队列中存在时间超过此阈值时，触发批量消费（若消费回调阻塞，则无法消费）。
     */
    public AbstractHttpQueueDealWithLiveService(int capacity, int batchSize, int maxWaitSize, int maxWaitTimeMillis) {
        super(capacity, batchSize, maxWaitSize, maxWaitTimeMillis);
    }

    @Override
    protected void batchConsumeMsg(List<T> list) {
        Map<Long, List<T>> liveIdBeanListMap = list.stream().filter(bean -> {
            if (NumberUtil.isNullOrLteZero(bean.getLiveId())) {
                log.error("batchConsumeHttpMsg, liveId is null or zero, bean={}", bean);
                overRequestFail(bean, BaseErrorCode.PARAM_ERROR);
                return false;
            }
            return true;
        }).collect(Collectors.groupingBy(HttpRequestBaseBean::getLiveId));
        liveIdBeanListMap.forEach((key,beanList)->{
            try {
                batchConsumeLiveMsg(key, beanList);
            }catch (Exception e){
                //如果发生任何异常，都尝试去结束一次请求。避免请求卡死在这儿。（等待请求超时需要5S）
                batchOverRequestFail(beanList, BaseErrorCode.SYSTEM_BUSY);
                log.error("batchConsumeHttpMsg, class={}, e:", this.getClass().getName(), e);
            }
        });
    }

    /**
     * 直播维度批量消费
     * @param liveId 直播id
     * @param list 消息集合
     */
    protected abstract void batchConsumeLiveMsg(Long liveId, List<T> list);
}
