package cn.com.duiba.kjy.base.api.enums;

import cn.com.duiba.kjy.base.api.bean.login.LoginConstant;
import cn.com.duiba.kjy.base.api.constant.DefaultConstant;
import cn.com.duiba.kjy.base.api.utils.IdMakeUtil;
import cn.com.duiba.kjy.base.api.utils.UrlUtils;
import cn.com.duiba.wolf.utils.UrlUtils2;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.net.URI;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 *<a href="http://cf.dui88.com/pages/viewpage.action?pageId=132499328">前端技术方案</a> <br/>
 *<a href="http://cf.dui88.com/pages/viewpage.action?pageId=132492647">后端技术方案</a>
 * @author dugq
 * @date 2021/12/21 3:29 下午
 */
@Getter
@Slf4j
public enum ProviderChannelEnum {
    //channel_id = tb_official_account表
    WECHAT_OA(1,"/oax","个微公众号"),
    WX_WORK(2,"/qywx","企微应用"),
    THIRD_PARTY_APP(3,"/app","第三方APP"),
    ;
    private final Integer code;
    private final String urlPrefix;
    private final String desc;

    ProviderChannelEnum(Integer code,String urlPrefix, String desc) {
        this.code = code;
        this.urlPrefix = urlPrefix;
        this.desc = desc;
    }

    private static final Map<Integer, ProviderChannelEnum> ENUM_MAP = Collections.unmodifiableMap(
            Arrays.stream(values()).collect(Collectors.toMap(ProviderChannelEnum::getCode, Function.identity(), (v1, v2) -> {
                log.error("ProviderChannelEnum, type distinct, type={}", v2.getCode());
                return v2;
            }))
    );

    /**
     * 根据状态获取枚举
     * @param code 状态
     * @return 枚举
     */
    public static ProviderChannelEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        return ENUM_MAP.get(code);
    }

    /**
     * 获取拼接链接.
     * 如果渠道为空，则返回原链接
     * @param channelId 渠道ID
     * @param subUrl 原始链接
     * @return 新的带有平台的链接
     */
    public String getUrl(String channelId,String subUrl){
        if (StringUtils.isBlank(channelId)){
            return subUrl;
        }
        return urlPrefix +"/"+ channelId + subUrl;
    }

    public String getUrl(Long channelId,String subUrl){
        if (Objects.isNull(channelId)){
            return subUrl;
        }
        return urlPrefix +"/"+IdMakeUtil.encodingId(channelId) + subUrl;
    }

    /**
     * <h1>在URL中自动拼接渠道信息</h1>
     * 分为两种
     * <li>前端链接:在URL的Path的最前面追加/type/channelMarking</li>
     * <li>后端链接:需要追加参数channelMarking</li>
     *
     * @param channelId 渠道ID 必填。为空时将会返回原链接
     * @param subUrl 短链接.必填，为空是将会自动变更为 / 根目录
     * @param serverDomain subURL没有域名时必填。例如：https://wx.kjjcrm.com
     * @return 带有渠道信息的URL
     */
    public String autoJoinChannel(Long channelId,String subUrl,String serverDomain){
        return doJoinUrl(serverDomain,subUrl,IdMakeUtil.encodingId(channelId));
    }

    /**
     * 功能同上
     */
    public String autoJoinChannel(String channelMarking,String subUrl,String serverDomain){
        return doJoinUrl(serverDomain,subUrl,channelMarking);
    }

    @NotNull
    private String doJoinUrl(String serverDomain, String url, String channelMarking) {
        //url有问题时，直接转至目标域名的根目录
        if (StringUtils.isBlank(url)){
            url = "/";
        }
        final URI uri = URI.create(url);
        final StringBuilder formatUri = new StringBuilder();
        //添加https://domain
        appendServer(serverDomain, uri, formatUri);
        Map<String,String> channelParam = appendPath(channelMarking, uri, formatUri);
        return appendQuery(uri, formatUri,channelParam);
    }

    private String appendQuery(URI uri, StringBuilder formatUri, Map<String,String> channelParam) {
        final String redirectUrl = formatUri.toString();
        //原URL没有参数列表，则直接追加渠道作为参数即可
        if (StringUtils.isBlank(uri.getRawQuery())){
            if (MapUtils.isNotEmpty(channelParam)){
                return UrlUtils2.appendParams(redirectUrl,channelParam);
            }else{
                return redirectUrl;
            }
        }
        //原URL附带了参数信息，为保证指定渠道和页面渠道相同，利用map的特性，用指定渠道填充或者覆盖原URL的渠道
        final Map<String, String> queryMap = UrlUtils.parseQueryString(uri.getRawQuery());
        queryMap.putAll(channelParam);
        return UrlUtils2.appendParams(redirectUrl,queryMap);
    }

    private Map<String,String> appendPath(String channelMarking, URI uri, StringBuilder formatUri) {
        Map<String,String> extParam = new HashMap<>();
        final String rawPath = getPath(uri.getRawPath());
        if (StringUtils.isBlank(channelMarking)){
            formatUri.append(rawPath);
        }else{
            //兼容二跳的链接 /kjj/jump?scene=X&urlPrefix=/oax/xxx
            if (StringUtils.startsWith(rawPath, DefaultConstant.JUMP_URL)){
                final String prefix = ProviderChannelEnum.getByCode(this.getCode()).getUrl(channelMarking, "");
                final Map<String, String> queryMap = UrlUtils.parseQueryString(uri.getRawQuery());
                if (!queryMap.containsKey(DefaultConstant.JUMP_CHANNEL_PARAM_NAME)){
                    extParam.put(DefaultConstant.JUMP_CHANNEL_PARAM_NAME,prefix);
                }
                formatUri.append(rawPath);
            }else if (StringUtils.startsWith(rawPath, "/pages") || StringUtils.startsWith(rawPath, "/basic")){ //前端域名统一添加前缀
                final String path = ProviderChannelEnum.getByCode(this.getCode()).getUrl(channelMarking, rawPath);
                formatUri.append(path);
            }else{ //其它情况返回false留给后续追加参数时，把渠道追加上去
                formatUri.append(rawPath);
                extParam.put(LoginConstant.USING_CHANNEL_PARAM_NAME,channelMarking);
            }
        }
        return extParam;
    }

    private void appendServer(String serverDomain, URI uri, StringBuilder formatUri) {
        //append domain include scheme and host
        if (StringUtils.isBlank(uri.getHost())){
            if (StringUtils.isNotBlank(serverDomain)){
                formatUri.append(serverDomain);
            }
        }else{
            if (StringUtils.isNotBlank(uri.getScheme())){
                formatUri.append(uri.getScheme());
                formatUri.append("://");
            }else{
                formatUri.append("https://");
            }
            formatUri.append(uri.getHost());
        }
    }

    //首页已经迁移到新的地址了。由后端转发
    private String getPath(String path) {
        if (StringUtils.equals("/",path)){
            return DefaultConstant.HOME_URL;
        }
        return path;
    }
}
