package cn.com.duiba.kjy.base.api.queue;

import cn.com.duiba.kjy.base.api.request.ErrorCodeInterface;
import cn.com.duiba.kjy.base.api.request.Result;
import cn.com.duiba.kjy.base.api.request.ResultBuilders;
import cn.com.duiba.kjy.base.api.response.ResponseUtils;
import cn.com.duiba.kjy.base.exception.errorcode.BaseErrorCode;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.AsyncContext;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 聚合http请求，批量处理
 * @author lizhi
 * @date 2020/4/5 11:40 AM
 */
@Slf4j
public abstract class AbstractHttpQueueService<T extends HttpRequestBaseBean> extends AbstractQueueService<T> {

    /**
     * 构造一个支持压测、并发、批量消费的Queue
     *
     * @param capacity          BlockingQueue的最大允许容量, 如果<=0 视为无界队列
     * @param batchSize         一次批量消费的个数
     * @param maxWaitSize       一次批量消费的阈值，当已入队的数据个数达到此阈值时，触发批量消费（若消费回调阻塞，则无法消费）
     *                          如果此阈值 <= 0则不会触发个数阈值消费（这种情况下数据只会在时间到达maxWaitTimeMillis阈值后才会被消费，这意味着数据可能会在队列中存在比较长的时间）
     * @param maxWaitTimeMillis 数据最大延迟消费时间的阈值(ms)，当有数据在队列中存在时间超过此阈值时，触发批量消费（若消费回调阻塞，则无法消费）。
     */
    public AbstractHttpQueueService(int capacity, int batchSize, int maxWaitSize, int maxWaitTimeMillis) {
        super(capacity, batchSize, maxWaitSize, maxWaitTimeMillis);
    }

    @Override
    public void addQueue(T t) {
        try {
            super.addQueue(t);
        } catch (IllegalStateException e){
            log.warn("addQueue, queue full or queue not start, class={}, e:", this.getClass().getName(), e);
            overRequestFail(t, BaseErrorCode.SYSTEM_BUSY);
        } catch (Exception e) {
            log.error("addQueue, error, class={}, e:", this.getClass().getName(), e);
            overRequestFail(t, BaseErrorCode.SYSTEM_BUSY);
        }
    }

    /**
     * 重写批量消费方法，确保所有请求都能结束
     * @param list 消息集合
     */
    @Override
    protected void batchConsume(List<T> list) {
        batchConsumeHttpMsg(list);
    }

    protected void batchConsumeHttpMsg(List<T> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        //获取所有AsyncContext
        List<AsyncContext> contextList = list.stream().map(T::getContext).collect(Collectors.toList());
        try {
            batchConsumeMsg(list);
        } catch (Exception e){
            //如果发生任何异常，都尝试去结束一次请求。避免请求卡死在这儿。（等待请求超时需要5S）
            batchOverRequestFail(list, BaseErrorCode.SYSTEM_BUSY);
            log.error("batchConsumeHttpMsg, class={}, e:", this.getClass().getName(), e);
        } finally {
            //检查是否有未关闭的请求，没有关闭的打错误日志，并关闭（等待请求超时需要5S）
            contextList.forEach(this::overRequest);
        }
    }

    /**
     * 结束请求，返回成功
     * @param httpRequestBaseBean 请求对象
     * @param resultVo 返回的结果
     * @param <R> 返回结果类型
     */
    protected <R> void overRequestSuccess(T httpRequestBaseBean, R resultVo) {
        overRequest(httpRequestBaseBean, ResultBuilders.success(resultVo));
    }

    /**
     * 结束请求，返回失败
     * @param httpRequestBaseBean 请求对象
     * @param errorCode 失败错误码
     */
    protected void overRequestFail(T httpRequestBaseBean, ErrorCodeInterface errorCode) {
        overRequest(httpRequestBaseBean, ResultBuilders.fail(errorCode));
    }

    /**
     * 批量结束请求，返回成功
     * @param subBeanList 请求对象集合
     * @param resultVo 返回的结果
     * @param <R> 返回结果类型
     */
    protected <R> void batchOverRequestSuccess(List<T> subBeanList, R resultVo) {
        batchOverRequest(subBeanList, ResultBuilders.success(resultVo));
    }

    /**
     * 批量结束请求，返回失败
     * @param subBeanList 请求对象集合
     * @param errorCode 失败错误码
     */
    protected void batchOverRequestFail(List<T> subBeanList, ErrorCodeInterface errorCode) {
        batchOverRequest(subBeanList, ResultBuilders.fail(errorCode));
    }

    /**
     * 批量结束请求
     * @param subBeanList 请求对象集合
     * @param result 返回的结果
     * @param <R> 返回结果类型
     */
    private <R> void batchOverRequest(List<T> subBeanList, Result<R> result) {
        if (CollectionUtils.isEmpty(subBeanList)) {
            return;
        }
        subBeanList.forEach(bean -> overRequest(bean, result));
    }

    /**
     * 结束请求
     * @param bean 请求对象
     * @param result 返回的结果
     * @param <R> 返回结果类型
     */
    private <R> void overRequest(T bean, Result<R> result) {
        try {
            log(bean, result);
            ResponseUtils.out((HttpServletResponse) bean.getContext().getResponse(), result);
            bean.getContext().complete();
        } catch (Exception e) {
            log.warn("结束请求错误, class={}", this.getClass().getName(), e);
        }
    }

    private <R> void log(T bean, Result<R> result) {
        try {
            HttpServletRequest request = (HttpServletRequest) bean.getContext().getRequest();
            String uri = request.getRequestURI();
            JSONObject jsonObject = new JSONObject();
            if (StringUtils.isNotBlank(uri)) {
                jsonObject.put("url_path", StringUtils.replace(uri, "//", "/"));
            }
            if(result != null){
                jsonObject.put("resp_body", getResponseBody(result));
            }
            jsonObject.put("live_user_id", bean.getLiveUserId());
            log.info("accessLog:{}",jsonObject.toJSONString());
        } catch (Exception e) {
            log.error("log, ", e);
        }
    }

    private <R> String getResponseBody(Result<R> result){
        String code = result.getCode();
        if (!Objects.equals(ErrorCodeInterface.DEFAULT_SUCCESS_CODE, code)) {
            return code;
        }
        R data = result.getData();
        return String.valueOf(data);
    }

    /**
     * 结束请求
     * @param context 异步上下文
     */
    protected void overRequest(AsyncContext context) {
        if (context == null) {
            return;
        }
        ServletResponse response;
        try {
            response = context.getResponse();
            if (response == null || response.isCommitted()) {
                return;
            }
        } catch (IllegalStateException e) {
            //获取response失败，不做任何处理
            return;
        } catch (Exception e) {
            log.warn("overRequest, get response fail, e:", e);
            return;
        }
        try {
            log.error("overRequest, not close, class={}", this.getClass().getName());
            ResponseUtils.out((HttpServletResponse) response, ResultBuilders.fail(BaseErrorCode.SYSTEM_ERROR));
            context.complete();
        } catch (Exception e) {
            log.warn("overRequest, error, class={}", this.getClass().getName(), e);
        }
    }
}
