package cn.com.duiba.kjy.base.api.listener;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.ApplicationObjectSupport;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.lang.annotation.Annotation;
import java.util.Map;
import java.util.Objects;

/**
 * @author dugq
 * @date 2021/11/9 11:38 上午
 */
@Configuration
@Slf4j
@ConditionalOnProperty(name="kjj.url.prefix")
public class UrlListener extends ApplicationObjectSupport implements InitializingBean, EnvironmentAware {
    private Environment environment;

    @Override
    public void afterPropertiesSet() throws Exception {
        final String urlPrefix = getUrlPrefix(this.environment);
        if (StringUtils.isBlank(urlPrefix)){
            return;
        }
        doValidator(urlPrefix, RestController.class);
        doValidator(urlPrefix, Controller.class);
    }

    private void doValidator(String urlPrefix, Class<? extends Annotation> annotationType) {
        final Map<String, Object> restCtrlBean = obtainApplicationContext().getBeansWithAnnotation(annotationType);
        for (Object bean : restCtrlBean.values()) {
             RequestMapping annotation = AnnotatedElementUtils.findMergedAnnotation(bean.getClass(), RequestMapping.class);
            if (Objects.isNull(annotation)){
                log.error("bean name = {} has no requestMapping it is not standard.please make url start with {}",bean.getClass().getName(), urlPrefix);
                continue;
            }
            final String[] pathList = annotation.path();
            if (ArrayUtils.isEmpty(pathList)){
                log.error("bean name = {} has error requestMapping it is not standard.please make url start with {}",bean.getClass().getName(), urlPrefix);
            }
            for (String path : pathList) {
                if (!path.startsWith(urlPrefix)){
                    log.error("bean name = {} error requestMapping it is not standard.please make url start with {}",bean.getClass().getName(), urlPrefix);
                    break;
                }
            }
        }
    }

    private String getUrlPrefix(Environment environment){
        return environment.getProperty("kjj.url.prefix");
    }


    @Override
    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }
}
