package cn.com.duiba.kjy.base.utils;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.kjy.base.api.utils.NumberUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

/**
 * 兑吧积分商城工具类
 * @author lizhi
 * @date 2022/11/30 8:03 下午
 */
@Slf4j
public class DuibaCmsUtil {

    private DuibaCmsUtil() {}

    /**
     * uid的最小长度
     */
    private static final int UID_MIN_LENGTH = 22;

    /**
     * 位数标识长度
     *
     * 位数标识为 01-99的字符串
     */
    private static final int DIGIT_LENGTH = 2;

    /**
     * 兑吧积分商城用户唯一性标识前缀
     */
    private static final String PRE_WORD = "KL";

    /**
     * 生成兑吧积分商城用户唯一性标识
     *
     * (公司ID位数+直播用户ID位数<=14位时，返回的uid长度为22位)
     * (若两个ID总长度超过14，则返回的uid长度为22+超出位数)
     * (前缀 + 环境标识 + "1" + 2位公司ID位数标识 + 2位用户ID位数标识 + 随机数 + 简单加密后的公司ID + 简单加密后的用户ID)
     *
     * @param companyId 公司ID
     * @param liveUserId 直播用户ID
     * @return 兑吧积分商城用户唯一性标识。为空情况：主键为空或小于0、id位数超过99位
     */
    public static String convert2Uid(Long companyId, Long liveUserId) {
        if (NumberUtil.isNullOrLteZero(companyId) || NumberUtil.isNullOrLteZero(liveUserId)) {
            return "";
        }
        // 环境标识
        String evn = getEvn();
        // 简单加密后的公司ID 和 简单加密后的用户ID
        String encodeCompanyIdStr = getEncodeIdStr(String.valueOf(companyId));
        String encodeLiveUserIdStr = getEncodeIdStr(String.valueOf(liveUserId));
        if (getIdLengthDigit(encodeCompanyIdStr) > DIGIT_LENGTH || getIdLengthDigit(encodeLiveUserIdStr) > DIGIT_LENGTH) {
            return "";
        }
        // 2位公司ID位数标识 + 2位用户ID位数标识
        String digitMark = getDigitMark(encodeCompanyIdStr) + getDigitMark(encodeLiveUserIdStr);
        // 简单加密后的公司ID + 简单加密后的用户ID
        String encodeIdStr = encodeCompanyIdStr + encodeLiveUserIdStr;
        // 除随机数外的长度=前缀 + 环境标识 + "1" + 2位公司ID位数标识 + 2位用户ID位数标识 + 简单加密后的公司ID + 简单加密后的用户ID
        int sureLength = PRE_WORD.length() + evn.length() + 1 + digitMark.length() + encodeIdStr.length();
        String replaceStr = getReplaceStr(encodeIdStr, UID_MIN_LENGTH - sureLength);
        return PRE_WORD + evn + "1" + digitMark + replaceStr + encodeIdStr;
    }

    /**
     * 解析兑吧积分商城用户唯一性标识
     *
     * (前缀 + 环境标识 + "1" + 2位公司ID位数标识 + 2位用户ID位数标识 + 随机数 + 简单加密后的公司ID + 简单加密后的用户ID)
     *
     * @param uid 兑吧积分商城用户唯一性标识
     * @return Pair对象 K: companyId V:liveUserId
     */
    public static Pair<Long, Long> parseFromUid(String uid) {
        if (StringUtils.isBlank(uid)) {
            return Pair.of(null, null);
        }
        String pre = PRE_WORD + getEvn();
        if (!uid.startsWith(pre)) {
            return Pair.of(null, null);
        }
        int digitStartIndex = pre.length() + 1;
        // 加密公司ID长度 加密用户ID长度
        int companyIdLength = parseIdLength(uid, digitStartIndex);
        int liveUserIdLength = parseIdLength(uid, digitStartIndex + DIGIT_LENGTH);
        Long companyId = decodeId(uid, uid.length() - liveUserIdLength - companyIdLength, uid.length() - liveUserIdLength);
        Long liveUserId = decodeId(uid, uid.length() - liveUserIdLength, uid.length());
        return Pair.of(companyId, liveUserId);
    }

    private static Long decodeId(String uid, int startIndex, int endIndex) {
        String encodeIdStr = uid.substring(startIndex, endIndex);
        String decodeIdStr = getDecodeIdStr(encodeIdStr);
        try {
            return Long.parseLong(decodeIdStr);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    private static int parseIdLength(String uid, int startIndex) {
        String idDigit = uid.substring(startIndex, startIndex + DIGIT_LENGTH);
        return Integer.parseInt(idDigit);
    }

    private static String getEvn() {
        if (SpringEnvironmentUtils.isProdEnv() || SpringEnvironmentUtils.isPreEnv()) {
            return "P";
        }
        if (SpringEnvironmentUtils.isTestEnv()) {
            return "T";
        }
        return "D";
    }

    /**
     * 获取加密ID的位数标记（01-99的字符串）
     * @param encodeIdStr 加密的ID
     * @return 加密ID的位数标记
     */
    private static String getDigitMark(String encodeIdStr) {
        // ID的长度
        int length = encodeIdStr.length();
        // ID的长度有多少位数，比如长度20，位数是2
        int digit = getLengthDigit(length);
        // 需要补 "0" 数量
        int zeroNum = DIGIT_LENGTH - digit;
        return getZero(zeroNum) + length;
    }

    /**
     * 获取加密ID的长度有多少位（比如长度20，位数是2）
     * @param encodeIdStr 加密的ID
     * @return 加密ID的长度有几位数
     */
    private static int getIdLengthDigit(String encodeIdStr) {
        int length = encodeIdStr.length();
        return getLengthDigit(length);
    }

    /**
     * 获取长度有几位数（比如长度20，位数是2）
     * @param length 长度
     * @return 长度有几位数
     */
    private static int getLengthDigit(int length) {
        // 长度有多少位数，比如长度20，位数是2
        String lengthStr = String.valueOf(length);
        return lengthStr.length();
    }

    /**
     * 加密ID
     * @param idStr id字符串
     * @return 加密后的ID字符串
     */
    private static String getEncodeIdStr(String idStr) {
        char[] chars = idStr.toCharArray();
        //数字转换
        convert(chars);
        //位置交换
        exchange(chars);
        return new String(chars);
    }

    /**
     * 解密ID
     * @param encodeIdStr 加密的ID字符串
     * @return 解密后的ID字符串
     */
    private static String getDecodeIdStr(String encodeIdStr) {
        if (encodeIdStr == null || encodeIdStr.trim().length() == 0) {
            return encodeIdStr;
        }
        char[] chars = encodeIdStr.toCharArray();
        //位置交换
        decodeExchange(chars);
        //数字转换
        decodeConvert(chars);
        return new String(chars);
    }


    private static void decodeExchange(char[] chars) {
        exchange(chars);
    }

    /**
     * 将数组中相邻两位位置交换
     * @param chars 需要位置交换的数组
     */
    private static void exchange(char[] chars) {
        for (int i = 0; i < chars.length/2; i++) {
            int index = i * 2;
            char temp = chars[index];
            chars[index] = chars[index+1];
            chars[index+1] = temp;
        }
    }

    private static void decodeConvert(char[] chars) {
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i]));
            if (num >= 7) {
                num = num - 7;
            } else {
                num = num + 3;
            }
            chars[i] = (char) (num + '0');
        }
    }

    /**
     * 将每一位数字+7，然后取余10
     * @param chars 需要转换数字的数组
     */
    private static void convert(char[] chars) {
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i]));
            num = (num + 7) % 10;
            chars[i] = (char) (num + '0');
        }
    }

    /**
     * 获取加密ID的补充数值
     * 规则：
     *      1.获取基础补充字符串（加密后的ID长度超过替换长度，截取加密ID，不足，前后补0）
     *      2.将补充字符串每一位 平方，然后取余 (位置+值)%10
     * @param encodeIdStr 加密的ID
     * @param replaceLength 需要补充的长度
     * @return 补充数值
     */
    private static String getReplaceStr(String encodeIdStr, int replaceLength) {
        if (replaceLength <= 0) {
            return "";
        }
        int num = replaceLength - encodeIdStr.length();
        String replaceStr;
        if (num < 0) {
            replaceStr = encodeIdStr.substring(0,replaceLength);
        } else if (num == 0){
            replaceStr = encodeIdStr;
        } else {
            replaceStr = getZero(num/2) + encodeIdStr + getZero(num - num/2);
        }
        return replace(replaceStr);
    }

    private static String replace(String replaceStr) {
        char[] chars = replaceStr.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            int num = Integer.parseInt(String.valueOf(chars[i])) + i + 1;
            int pow = num * num;
            int yu = (num + i) % 10;
            if (yu == 0) {
                yu = 10;
            }
            int result = pow % yu;
            chars[i] = (char) (result + '0');
        }
        return new String(chars);
    }

    /**
     * 获取0
     * @param num 需要的0的个数
     * @return 0字符串
     */
    private static String getZero(int num) {
        if (num <= 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < num; i++) {
            sb.append("0");
        }
        return sb.toString();
    }
}
