package cn.com.duiba.kjy.base.api.utils;

import cn.com.duiba.kjy.base.api.constant.DefaultConstant;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Created by zhangshun on 2019/2/28.
 */
public class UrlUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(UrlUtils.class);

    private UrlUtils() {
        //to do something
    }

    public static String encoder(String url) {
        try {
            return URLEncoder.encode(url, DefaultConstant.DEFAULT_ENCODING);
        } catch (Exception e) {
            LOGGER.warn("url encoder error url={}", url, e);
        }
        return url;
    }

    public static String decoder(String url) {
        try {
            return URLDecoder.decode(url, DefaultConstant.DEFAULT_ENCODING);
        } catch (Exception e) {
            LOGGER.warn("url decoder error url={}", url, e);
        }
        return url;
    }

    //很多时候，我们的uri可能会多次加密，所以这里也多次解密。
    public static String decodeV2(String url){
        if (StringUtils.isBlank(url)){
            return url;
        }
        int i=0;
        while (i++<4){
            if (url.startsWith("/") || url.startsWith("http:") || url.startsWith("https:") ){
                return url;
            }
            url = UrlUtils.decoder(url);
        }
        return url;
    }

    public static Map<String, String> parseQueryString(String queryString) {

        if (StringUtils.isBlank(queryString)) {
            return Collections.emptyMap();
        }

        int idx = queryString.indexOf('?');

        String paramStr = "";

        if (idx >= 0) {
            paramStr = new String(queryString.substring(idx + 1));
        } else {
            paramStr = queryString;
        }

        Map<String, String> map = Maps.newHashMap();

        List<NameValuePair> list = URLEncodedUtils.parse(paramStr, Charset.forName(DefaultConstant.DEFAULT_ENCODING));

        for (NameValuePair pair : list) {
            map.put(pair.getName(), pair.getValue());
        }

        return map;
    }

    public static String getHost(String url) {
        if (StringUtils.isBlank(url)) {
            return StringUtils.EMPTY;
        }
        try {
            URI u = new URI(url);
            return u.getHost();
        } catch (URISyntaxException e) {
            LOGGER.info("url转换错误 {}", url);
            return StringUtils.EMPTY;
        }
    }

    public static String getPath(String url) {
        if (StringUtils.isBlank(url)) {
            return StringUtils.EMPTY;
        }
        try {
            URI u = new URI(url);
            return u.getPath();
        } catch (URISyntaxException e) {
            LOGGER.info("url转换错误 {}", url);
            return StringUtils.EMPTY;
        }
    }

    public static String getUrl(String url) {
        if (StringUtils.startsWith(url, "//")) {
            return "https:" + url;
        }
        return url;
    }

    /**
     * 在url上拼接query param
     * 可以过滤空字符串
     *
     * @param url
     * @param infos
     * @return
     */
    public static String appendQueryParam(String url, String... infos) {
        if (StringUtils.isEmpty(url) || ArrayUtils.isEmpty(infos)) {
            return url;
        }

        StringBuilder sb = new StringBuilder(url);
        for (int i = 0; i < infos.length; i++) {
            String trimInfo = StringUtils.trim(infos[i]);
            if (StringUtils.isEmpty(trimInfo)) {
                continue;
            }
            if (sb.indexOf("?") != -1) {
                sb.append('&');
            } else {
                sb.append('?');
            }
            sb.append(trimInfo);
        }
        return sb.toString();
    }

    /**
     * 追加 HttpServletRequest 中的参数(原URL中已有的参数不会再追加)
     * @param url 需要追加参数的链接
     * @param queryString  HttpServletRequest 中的参数，通过 getQueryString() 方法获取
     * @return 追加参数后的链接
     */
    public static String addRequestParams(String url, String queryString) {
        return addRequestParams(url, queryString, "cid", "dpm");
    }

    /**
     * 追加 HttpServletRequest 中的参数(原URL中已有的参数不会再追加)
     * @param url 需要追加参数的链接
     * @param queryString  HttpServletRequest 中的参数，通过 getQueryString() 方法获取
     * @param removeParams  HttpServletRequest 中需要去除的参数
     * @return 追加参数后的链接
     */
    public static String addRequestParams(String url, String queryString, String... removeParams) {
        if (StringUtils.isBlank(url) || StringUtils.isBlank(queryString)) {
            return url;
        }
        Map<String, String> requestParams = parseQueryString(queryString);
        if (MapUtils.isEmpty(requestParams)) {
            return url;
        }
        //去掉特定参数
        removeKeys(requestParams, removeParams);
        if (MapUtils.isEmpty(requestParams)) {
            return url;
        }
        //去掉原来URL中已经包含的参数
        Map<String, String> params = parseQueryString(url);
        if (MapUtils.isNotEmpty(params)) {
            requestParams.keySet().removeIf(params::containsKey);
        }
        if (MapUtils.isEmpty(requestParams)) {
            return url;
        }
        return cn.com.duiba.wolf.utils.UrlUtils.appendParams(url, requestParams);
    }

    private static void removeKeys(Map<String, String> requestParams, String... removeParams) {
        if (MapUtils.isEmpty(requestParams)) {
            return;
        }
        if (removeParams == null || removeParams.length == 0) {
            return;
        }
        for (String param : removeParams) {
            requestParams.remove(param);
        }
    }

    /**
     * 给链接添加域名（链接本身以http开头，则返回链接本身，否则返回域名+链接）
     * @param domain 域名
     * @param url 链接
     * @return 添加域名后的链接
     */
    public static String addDomain(String domain, String url) {
        if (StringUtils.isBlank(url)) {
            return "";
        }
        if (url.toLowerCase().startsWith("http")) {
            return url;
        }
        return domain + url;
    }
}
