package cn.com.duiba.kjy.base.api.request;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import lombok.Data;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * 分页返回体
 *
 * @author xuzhigang
 * @date 2021/12/9 5:48 下午
 **/
@Data
public class PageResult<T> implements Serializable {
    private static final long serialVersionUID = 5033472050991731743L;
    /**
     * 页码
     */
    private Integer pageIndex;
    /**
     * 每页条数
     */
    private Integer pageSize;
    /**
     * 总数
     */
    private Long totalCount;
    /**
     * list数据
     */
    private List<T> list;
    /**
     * 是否有下一页
     */
    private Boolean hasNext;

    public Boolean getHasNext() {
        if(Objects.isNull(hasNext)){
            Long totalPage = getTotalPage();
            if(Objects.nonNull(totalPage)){
                return totalPage - pageIndex > 0;
            }
        }
        return hasNext;
    }

    public Long getTotalPage() {
        if (Objects.nonNull(pageIndex) && Objects.nonNull(totalCount) && Objects.nonNull(pageSize)) {
            return totalCount / pageSize + (totalCount % pageSize > 0 ? 1 : 0);
        }
        return null;
    }

    public static <T> PageResult<T> init(Integer pageIndex, Integer pageSize) {
        PageResult<T> pageResult = new PageResult<>();
        pageResult.setList(Lists.newArrayList());
        pageResult.setTotalCount(0L);
        pageResult.setPageIndex(pageIndex);
        pageResult.setPageSize(pageSize);
        return pageResult;
    }

    /**
     * 获取没有下一页的默认结果(瀑布流下拉统计)
     */
    public static <T> PageResult<T> getNoNextResult() {
        PageResult<T> pageResult = new PageResult<>();
        pageResult.setList(Collections.emptyList());
        pageResult.setHasNext(false);
        return pageResult;
    }

    public void setHasNext(Boolean hasNext) {
        this.hasNext = hasNext;
    }

    /**
     * 复制分页结果
     * @param sourcePage 源分页结果
     * @param destClz 目标类
     * @param <T> 目标类范型
     * @return 复制后的分页结果
     */
    public static  <T> PageResult<T> copyPage(PageResult<?> sourcePage, Class<T> destClz) {
        if (sourcePage == null) {
            return null;
        }
        PageResult<T> result = PageResult.init(sourcePage.getPageIndex(), sourcePage.getPageSize());
        if (destClz == null) {
            return result;
        }
        final List<T> destList = BeanUtils.copyList(sourcePage.getList(), destClz);
        result.setList(destList);
        result.setTotalCount(sourcePage.getTotalCount());
        return result;
    }

}
