package cn.com.duiba.kjy.api.util;

import cn.com.duiba.kjy.api.enums.push.PushEventEnum;
import cn.com.duiba.kjy.api.enums.push.PushMessageTypeEnum;
import cn.com.duiba.wolf.utils.UrlUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    private static final String SEPARATOR = "_";

    private static final String DPM_HEAD = "S";

    /**
     * 将推送参数转换成一个参数值
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStr(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : IdMakerUtil.encodingId(oaId)) +  SEPARATOR + (eventEnum == null ? "" : eventEnum.getEventType()) + SEPARATOR
                + ( messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数转换成一个参数值（oaId不加密）
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送参数值
     */
    public static String getPushParamStrShort(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return (oaId == null ? "" : (oaId + 3)) +  SEPARATOR + (eventEnum == null ? "" : eventEnum.getEventType()) + SEPARATOR
                + ( messageTypeEnum == null ? "" : messageTypeEnum.getType()) + SEPARATOR + (urlOrder == null ? "" : urlOrder)
                + SEPARATOR + (dpm == null ? "" : dpm);
    }

    /**
     * 将推送参数值，转换成推送参数map
     * @param pushParamStr 推送参数值
     * @return 推送参数map
     */
    public static Map<String, String> getPushParamMapByStr(String pushParamStr) {
        if (StringUtils.isBlank(pushParamStr)) {
            return Collections.emptyMap();
        }
        Long oaId = null;
        String pushId = null;
        Integer messageType = null;
        Integer urlOrder = null;
        String dpm = null;
        String[] split = pushParamStr.split(SEPARATOR);
        for (int i = 0; i < split.length; i++) {
            String str = split[i];
            if (StringUtils.isBlank(str)) {
                continue;
            }
            if (i == 0) {
                if (NumberUtils.isNumber(str)) {
                    oaId = Long.parseLong(str) - 3;
                } else {
                    oaId = IdMakerUtil.decodingId(str);
                }
            }
            if (i == 1) {
                pushId = str;
            }
            if (i == 2 && NumberUtils.isNumber(str)) {
                messageType = Integer.parseInt(str);
            }
            if (i == 3 && NumberUtils.isNumber(str)) {
                urlOrder = Integer.parseInt(str);
            }
            if (i == 4) {
                dpm = str;
            }
        }
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getPushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> pushParamMap = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取map中的推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数（不包含dpm）
     */
    public static String getPushParam(Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.buildURLParams(pushParamMap);
    }

    /**
     * 获取推送埋点参数
     * @param oaId 公众号ID
     * @param eventEnum 推送事件类型
     * @param messageTypeEnum 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数
     */
    public static String getEncodePushParam(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        return encode(getPushParam(oaId,eventEnum, messageTypeEnum, urlOrder, dpm));
    }

    /**
     * URL增加推送埋点参数
     * @param url 链接
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 带参数的URL
     */
    public static String buildPushParam(String url, Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        Map<String, String> params = getPushParamMap(oaId, eventEnum, messageTypeEnum, urlOrder, dpm);
        int index = url.indexOf("?");
        if (index < 0) {
            return UrlUtils.appendParams(url, params);
        }
        Map<String, String> map = UrlUtils.uRLRequest(url);
        if (MapUtils.isEmpty(map)) {
            return UrlUtils.appendParams(url, params);
        }
        String uriUrl = map.get("uri");
        if (StringUtils.isBlank(uriUrl)) {
            return UrlUtils.appendParams(url, params);
        }
        String decode = decode(uriUrl);
        String decodePushUrl = UrlUtils.appendParams(decode, params);
        map.put("uri", encode(decodePushUrl));
        return UrlUtils.appendParams(UrlUtils.urlPage(url), map);
    }

    /**
     * URL追加map中的推送埋点参数（不包含dpm）
     * @param url 链接
     * @param parameterMap request中的所有参数
     * @return 带推送参数的URL
     */
    public static String buildPushParam(String url, Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils.appendParams(url, pushParamMap);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param eventEnum 推送事件
     * @param messageTypeEnum 消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, PushEventEnum eventEnum, PushMessageTypeEnum messageTypeEnum, Integer urlOrder, String dpm) {
        String pushId = Objects.isNull(eventEnum) ? null : eventEnum.getEventType();
        Integer messageType = Objects.isNull(messageTypeEnum) ? null : messageTypeEnum.getType();
        return getPushParamMap(oaId,pushId, messageType, urlOrder, dpm);
    }

    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param pushId 推送事件Id
     * @param messageType 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String pushId, Integer messageType, Integer urlOrder, String dpm) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakerUtil.encodingId(oaId));
        }
        if (Objects.nonNull(pushId)) {
            params.put("push_id", pushId);
        }
        if (Objects.nonNull(messageType)) {
            params.put("message_type", String.valueOf(messageType));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        params.put("access_source", getAccessSourceByPushId(pushId));
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        return params;
    }

    /**
     * 根据pushId获取accessSource
     * @param pushId
     * @return
     */
    private static String getAccessSourceByPushId(String pushId) {
        if (StringUtils.isBlank(pushId) || pushId.length() < 2) {
            return "1";
        }
        String accessSource = "10";
        String word = pushId.substring(0, 1);
        String num = pushId.substring(1);
        while (num.length() < 3) {
            num = "0" + num;
        }
        switch (word) {
            case "S":
                accessSource += "1";
                break;
            case "D":
                accessSource += "2";
                break;
            case "H":
                accessSource += "3";
                break;
            case "V":
                accessSource += "4";
                break;
            default:
                accessSource += "5";
                break;
        }
        return accessSource + num;
    }

    /**
     * 获取请求中推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数map
     */
    public static Map<String, String> getPushParamMap(Map<String, String[]> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return Collections.emptyMap();
        }
        Map<String, String> params = new HashMap<>(16);
        Set<String> pushKey = getPushParamMap(0L, "", 0, 0, null).keySet();
        for (String key : pushKey) {
            String[] param = parameterMap.get(key);
            if (isNotEmpty(param)) {
                params.put(key, param[0]);
            }
        }
        return params;
    }

    /**
     * 根据推送事件，获取推送点击埋点
     * @param eventEnum 推送事件
     * @return 点击埋点
     */
    public static String getDpmByPushEventEnum(PushEventEnum eventEnum) {
        if (eventEnum == null) {
            return "";
        }
        if (eventEnum == PushEventEnum.STATISTICS_DAY) {
            return "11.9.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_WEEK) {
            return "11.10.0.0";
        }
        if (eventEnum == PushEventEnum.SUBSCRIBE_WELCOME) {
            return "11.12.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVATION) {
            return "11.14.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_FIRST) {
            return "11.15.0.0";
        }
        if (eventEnum == PushEventEnum.VISIT_NON_FIRST) {
            return "11.16.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT) {
            return "11.17.0.0";
        }
        if (eventEnum ==  PushEventEnum.PREFERENTIAL_INFORMATION) {
            return "11.18.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_ACCESS) {
            return "11.19.0.0";
        }
        if (eventEnum == PushEventEnum.PRIVATE_CHAT_SEND) {
            return "11.20.0.0";
        }
        if (eventEnum == PushEventEnum.DISTRIBUTION) {
            return "11.22.0.0";
        }
        if (eventEnum == PushEventEnum.REWARDED) {
            return "11.23.0.0";
        }
        if (eventEnum == PushEventEnum.STATISTICS_VIP_INCOME) {
            return "11.24.0.0";
        }
        if (eventEnum == PushEventEnum.INTERACTIVE_PUSH) {
            return "11.25.0.0";
        }
        if (eventEnum == PushEventEnum.SELLER_CARD_REAL_TIME_PUSH) {
            return "11.26.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_CONTENT_PUSH) {
            return "11.27.0.0";
        }
        if (eventEnum == PushEventEnum.FORWARD_SELLER_CARD_PUSH) {
            return "11.28.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_NON_REAL_TIME) {
            return "11.30.0.0";
        }
        if (eventEnum == PushEventEnum.ACTIVITY_FORM_SUBMIT_LOTTERY) {
            return "11.31.0.0";
        }
        if (eventEnum == PushEventEnum.WEEKLY_THUMBS_UP) {
            return "11.32.0.0";
        }
        if (eventEnum == PushEventEnum.SCAN_PROM) {
            return "11.34.0.0";
        }
        if (eventEnum == PushEventEnum.NATIONAL_DAY_ACTIVITY) {
            return "11.37.0.0";
        }
        if (eventEnum == PushEventEnum.CLICK_CONTACT_PHONE) {
            return "11.0.1.28";
        }
        if (eventEnum == PushEventEnum.CLICK_CONTACT_WX) {
            return "11.0.1.29";
        }
        if (eventEnum == PushEventEnum.READING) {
            return "11.0.1.30";
        }
        if (eventEnum == PushEventEnum.CLICK_CONTACT) {
            return "11.0.1.32";
        }
        if (eventEnum == PushEventEnum.LIKE_CARD_INTERACTION_FROM_OTHER) {
            return "11.0.1.33";
        }
        if (eventEnum == PushEventEnum.INVESTMENT_INVESTMENT_TIMING_CONTENT_PUSH1) {
            return "11.0.1.37";
        }
        if (eventEnum == PushEventEnum.INVESTMENT_INVESTMENT_TIMING_CONTENT_PUSH2) {
            return "11.0.1.38";
        }
        if (eventEnum == PushEventEnum.INVESTMENT_INVESTMENT_TIMING_CONTENT_PUSH3) {
            return "11.0.1.43";
        }
        if (eventEnum == PushEventEnum.INCREASE_USER_CLUE_ACTIVITY) {
            return "11.0.1.39";
        }
        if (eventEnum == PushEventEnum.INVESTMENT_CLUE_ACTIVITY) {
            return "11.0.1.40";
        }
        if (eventEnum == PushEventEnum.INCREASE_USER_CLUE_ARTICLE) {
            return "11.0.1.41";
        }
        if (eventEnum == PushEventEnum.INVESTMENT_CLUE_ARTICLE) {
            return "11.0.1.42";
        }
        if (eventEnum == PushEventEnum.SINGLE_FESTIVAL_ACTIVITY) {
            return "11.0.1.45";
        }
        if (eventEnum == PushEventEnum.SINGLE_FESTIVAL_WELCOME) {
            return "11.0.1.44";
        }
        if (eventEnum == PushEventEnum.SINGLE_FESTIVAL_DAILY_PUSH) {
            return "11.0.1.46";
        }

        return calculateDpm(eventEnum);
    }

    /**
     * 根据eventEnum计算dpm
     * @param eventEnum
     * @return
     */
    private static String calculateDpm(PushEventEnum eventEnum) {
        String head=eventEnum.getEventType().substring(0,1);
        String tail=eventEnum.getEventType().substring(1,3);
        //dpm由4位组成，用"."隔开
        if (StringUtils.equals(head,DPM_HEAD)){
            return "11.0.1."+tail;
        }
        return "";
    }

    /**
     * URL编码
     * @param url 链接
     * @return 编码后的链接
     */
    public static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    /**
     * URL解码
     * @param url 链接
     * @return 解码后的链接
     */
    public static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static boolean isNotEmpty(String[] param) {
        return Objects.nonNull(param) && param.length > 0;
    }
}
