package cn.com.duiba.kjy.api.api.enums.mgm;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author zhuenhui
 * @date 2022/05/19 14:47
 * @description MGM - m2状态枚举
 */
@Getter
@AllArgsConstructor
public enum MgmM2StatusEnum {

    /**
     * 不符合办卡条件: 是否新户：否
     * 审批拒绝: 审批状态：拒绝
     * 初审通过: 初审日期：YYYYMMDD
     * 已激活: 激活日期: YYYYMMDD
     * 已首刷: 首刷日期:YYYYMMDD
     * 申请: 申卡日期：YYYYMMDD
     * 无效：申卡次月底前未完成首刷
     */
    NOT_MATCH(1, "不符合办卡条件", ""),
    REJECT(2, "审批拒绝", "已拒绝"),
    FIRST_AUDIT_PASS(3, "初审通过", "待面签"),
    ACTIVATE(4, "已激活", "已批核"),
    CONSUME(5, "完成首刷", "已首刷"),
    APPLY(6, "申请", "审核中"),
    INVALID(7, "无效", ""),
    UNKNOWN(8, "未知", ""),
    ;

    /**
     * 已邀请集合
     * 办理中信银行信用卡的新用户，并未完成首刷（完成申卡、通过初审、激活卡片均算已邀请）
     */
    public static final Set<Integer> INVITE_LIST = Collections.unmodifiableSet(
            Stream.of(
                    APPLY.getStatus(),
                    FIRST_AUDIT_PASS.getStatus(),
                    ACTIVATE.getStatus()
            ).collect(Collectors.toSet())
    );

    /**
     * 中信接口返回没有新户状态字段的枚举集合（newCustFlag）
     */
    public static final Set<Integer> NO_NEW_CUST_FLAG_LIST = Collections.unmodifiableSet(
            Stream.of(
                    APPLY.getStatus(),
                    FIRST_AUDIT_PASS.getStatus(),
                    REJECT.getStatus()
            ).collect(Collectors.toSet())
    );

    /**
     * 中信审批状态返回字段枚举Map
     */
    private static final Map<String, MgmM2StatusEnum> CARD_DESC_MAP = Collections.unmodifiableMap(
            Arrays.stream(values())
                    .filter(m2Status -> StringUtils.isNotBlank(m2Status.getCardDesc()))
                    .collect(Collectors.toMap(MgmM2StatusEnum::getCardDesc, Function.identity(), (v1, v2) -> v2))
    );

    /**
     * 根据中信审批状态返回字段选择枚举
     *
     * @param cardDesc 中信审批状态返回字段
     * @return 状态枚举
     */
    public static MgmM2StatusEnum getByCardDesc(String cardDesc) {
        if (StringUtils.isBlank(cardDesc)) {
            return null;
        }
        return CARD_DESC_MAP.get(cardDesc);
    }

    /**
     * 状态
     */
    private final Integer status;

    /**
     * 描述
     */
    private final String desc;

    /**
     * 中信审批状态返回字段描述
     */
    private final String cardDesc;

}
