package cn.com.duiba.local.ext.exception;

/**
 * 正常的业务逻辑异常,注意此异常不会触发Hystrix熔断或降级，比如考虑以下这种场景： 用户使用手机号注册，web端调用center层的一个注册接口，
 * 如果库里已经有该手机号，则接口可以抛出异常 new BizException("用户不存在").withCode("user.already.exists"); 调用端需要明确捕
 * 获该异常， 来判断发生了这种情况，并返回给客户对应的错误信息。 切记此异常是需要调用端明确处理的业务异常，如果是服务器内部异常，比如数据
 * 库连不上等客户端无法处理的异常，则不应该使用BizException。  这个异常有些特殊之处，通常@HystrixDisabled注解的异常被客户端拦截到的
 * 时候会是HystrixBadRequestException, 当前异常会特殊处理，客户端catch到的异常会是BizException
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
public class BizException extends Exception {
    private static final int MAX_DEPTH = 25;

    /**
     * 异常错误码, 比如用户已存在可以表示为 user.already.exists, 图形验证码错误可以表示为 invalid.verify.code，命名请尽量有意义些
     * <br>
     * 这个字段可以用于表示国际化的key。
     */
    private String code;

    /**
     * 构造一个业务异常，通常而言这个异常的message是应该直接回显给用户的，比如"用户已存在".
     *
     * @param message 默认异常消息，表示具体的错误信息，比如『用户不存在』(后期国际化以后，这个可以用于表示默认异常消息，服务端优先根据
     *                key去找到对应语言的错误消息并替换message【服务端从RpcContext中获得当前语言环境】，找不到则不替换)
     */
    public BizException(String message) {
        super(message);
    }

    /**
     * 设置异常错误码
     *
     * @param code
     * @return
     */
    public BizException withCode(String code) {
        this.code = code;
        return this;
    }

    /**
     * 异常错误码, 比如用户已存在可以表示为 user.already.exists, 图形验证码错误可以表示为 invalid.verify.code，命名请尽量有意义些
     * <br>
     * 后期国际化的时候，这个字段可以用于表示国际化的key。
     */
    public String getCode() {
        return code;
    }

    /**
     * 判断异常的根cause是不是BizException，如果是的话，返回这个BizException，如果不是，原样抛出异常
     *
     * @param e 异常
     * @return BizException
     */
    public static BizException getOrPropagate(Throwable e) {
        Throwable finalE = getFinalCause(e);
        if (finalE instanceof BizException) {
            return (BizException) finalE;
        }

        throw propagate(e);
    }

    private static Throwable getFinalCause(Throwable e) {
        int i = 0;
        Throwable e1 = e;
        while (e1.getCause() != null) {
            if (i++ >= MAX_DEPTH) {
                // stack too deep to get final cause
                return new RuntimeException("Stack too deep to get final cause");
            }
            e1 = e1.getCause();
        }
        return e1;
    }

    private static RuntimeException propagate(Throwable t) {
        if (t instanceof RuntimeException) {
            throw (RuntimeException) t;
        } else if (t instanceof Error) {
            throw (Error) t;
        } else {
            throw new RuntimeException(t);
        }
    }
}