package cn.com.duiba.local.autoconfigure.web.servlet;

import cn.com.duiba.local.ext.exception.BizException;
import cn.com.duiba.local.wolf.entity.JsonResult;
import cn.com.duiba.local.wolf.message.BaseError;
import com.alibaba.csp.sentinel.slots.block.BlockException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.InvocationTargetException;

/**
 * WebMvc全局异常处理
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Slf4j
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@ConditionalOnClass({Servlet.class, DispatcherServlet.class, WebMvcConfigurer.class})
@RestControllerAdvice
public class WebMvcExceptionHandler {
    @ExceptionHandler(BizException.class)
    public JsonResult<Void> bizExceptionHandler(BizException e) {
        log.warn("BizException [{}]", e.getMessage(), e);
        return JsonResult.fail(e.getCode(), e.getMessage());
    }

    @ExceptionHandler({MethodArgumentNotValidException.class, BindException.class})
    public JsonResult<Void> methodArgumentNotValidExceptionHandler(Exception e) {
        BindingResult bindResult = null;
        if (e instanceof BindException) {
            bindResult = ((BindException) e).getBindingResult();
        } else if (e instanceof MethodArgumentNotValidException) {
            bindResult = ((MethodArgumentNotValidException) e).getBindingResult();
        }
        String message = "参数错误";
        if (bindResult != null && bindResult.hasErrors()) {
            ObjectError objectError = bindResult.getAllErrors().get(0);
            message = objectError.getDefaultMessage();
        }
        return JsonResult.fail(BaseError.PARAM_ERROR.getCode(), message);
    }

    @ExceptionHandler(IllegalArgumentException.class)
    public JsonResult<Void> illegalArgumentExceptionHandler(IllegalArgumentException e) {
        log.warn("IllegalArgumentException", e);
        return JsonResult.fail(e.getMessage());
    }

    @ExceptionHandler(BlockException.class)
    public JsonResult<Void> sentinelBlockHandler(BlockException e) {
        log.warn("Blocked by Sentinel [{}]", e.getRule());
        return JsonResult.fail(BaseError.API_LIMIT.getMsg(), BaseError.API_LIMIT.getCode());
    }

    @ExceptionHandler(Exception.class)
    public JsonResult<Void> exceptionHandler(Exception e, HttpServletRequest request) {
        String message = e.getMessage();
        if (e instanceof InvocationTargetException && message == null) {
            message = ((InvocationTargetException) e).getTargetException().getMessage();
        }
        log.error("Exception [{} -> {}]:", request.getRequestURI(), message, e);
        return JsonResult.fail(BaseError.SYSTEM_ERROR.getMsg());
    }
}