package cn.com.duiba.local.autoconfigure.mq.rocket;

import cn.com.duiba.local.ext.api.event.MainContextRefreshedEvent;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;

/**
 * Rocket MQ 自动装配
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Slf4j
@Configuration
@ConditionalOnClass({DefaultMQProducer.class, DefaultMQPushConsumer.class})
@EnableConfigurationProperties(RocketMqProperties.class)
public class RocketMqAutoConfiguration {
    protected abstract static class RocketMqClientConfiguration {
        @Resource
        protected RocketMqProperties rocketMqProperties;
    }

    @Configuration
    @ConditionalOnClass(DefaultMQProducer.class)
    @ConditionalOnMissingBean(name = "rocketMqProducer")
    @ConditionalOnProperty(prefix = RocketMqProperties.ROCKET_PREFIX, name = "producer.enable", havingValue = "true")
    protected static class RocketMqProducerConfiguration extends RocketMqClientConfiguration {
        @Bean(name = "rocketMqProducer", destroyMethod = "shutdown")
        public DefaultMQProducer rocketMqProducer() throws MQClientException {
            DefaultMQProducer producer = new DefaultMQProducer(rocketMqProperties.getProducer().getGroup());
            producer.setNamesrvAddr(rocketMqProperties.getNameSrvAddress());
            producer.setSendMsgTimeout(rocketMqProperties.getProducer().getSendMsgTimeoutMillis());
            producer.start();
            return producer;
        }
    }

    @Configuration
    @ConditionalOnClass(DefaultMQPushConsumer.class)
    @ConditionalOnMissingBean(name = "rocketMqConsumer")
    @ConditionalOnProperty(prefix = RocketMqProperties.ROCKET_PREFIX, name = "consumer.enable", havingValue = "true")
    protected static class RocketMqConsumerConfiguration extends RocketMqClientConfiguration {
        @Bean(name = "rocketMqConsumer", destroyMethod = "shutdown")
        public DefaultMQPushConsumer rocketMqConsumer() throws MQClientException {
            RocketMqProperties.Consumer consumerProperties = rocketMqProperties.getConsumer();
            DefaultMQPushConsumer consumer = new DefaultMQPushConsumer(consumerProperties.getGroup());
            consumer.setNamesrvAddr(rocketMqProperties.getNameSrvAddress());
            consumer.setMaxReconsumeTimes(consumerProperties.getMaxReConsumeTimes());
            consumer.setConsumeThreadMin(consumerProperties.getConsumeMinThreadNumber());
            consumer.setConsumeThreadMax(Math.max(consumerProperties.getMaxReConsumeTimes(), 64));
            consumer.setMessageModel(consumerProperties.getMessageModel());
            consumer.setConsumeMessageBatchMaxSize(consumerProperties.getConsumeMessageBatchMaxSize());
            String topics = consumerProperties.getTopics();
            String[] topicArr = topics.split(",");
            for (String topic : topicArr) {
                consumer.subscribe(topic, "*");
            }
            return consumer;
        }

        @EventListener(MainContextRefreshedEvent.class)
        public void initListener() {
            new RocketMqConsumerListener().startRun();
        }
    }
}