package cn.com.duiba.local.autoconfigure.web.servlet;

import cn.com.duiba.local.ext.api.exception.BizException;
import cn.com.duiba.local.ext.sentinel.SentinelExceptionEnum;
import cn.com.duiba.local.wolf.entity.JsonResult;
import cn.com.duiba.local.wolf.message.BaseError;
import cn.hutool.core.io.IoUtil;
import cn.hutool.http.ContentType;
import com.alibaba.csp.sentinel.slots.block.BlockException;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeException;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.web.error.ErrorAttributeOptions;
import org.springframework.boot.web.servlet.error.ErrorAttributes;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.lang.NonNull;
import org.springframework.web.context.request.ServletWebRequest;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.UndeclaredThrowableException;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * WebMvc全局异常处理
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Slf4j
@Order(Ordered.LOWEST_PRECEDENCE - 10)
public class WebMvcExceptionHandler implements HandlerExceptionResolver {
    @Resource
    private ErrorAttributes errorAttributes;

    @Override
    public ModelAndView resolveException(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response,
                                         Object handler, @NonNull Exception ex) {
        response.setContentType(ContentType.build(ContentType.JSON, StandardCharsets.UTF_8));
        Throwable t = ex;
        try {
            if (t instanceof UndeclaredThrowableException) {
                t = ((UndeclaredThrowableException) t).getUndeclaredThrowable();
                return undeclaredExceptionHandler(t, response);
            }
            if (t instanceof BizException) {
                return bizExceptionHandler(t, response);
            }
            if (BlockException.isBlockException(t)) {
                return sentinelExceptionHandler(t, response);
            }
            return globalExceptionHandler(t, request, response);
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    private ModelAndView undeclaredExceptionHandler(Throwable t, HttpServletResponse response) throws IOException {
        if (t instanceof DegradeException) {
            ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(BaseError.API_DEGRADE.getCode(),
                    BaseError.API_DEGRADE.getMsg())));
        } else if (t instanceof BlockException) {
            ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(BaseError.API_FLOW.getCode(),
                    BaseError.API_FLOW.getMsg())));
        } else {
            JsonResult.fail(BaseError.SYSTEM_ERROR.getCode(), BaseError.SYSTEM_ERROR.getMsg());
        }
        return new ModelAndView();
    }

    private ModelAndView sentinelExceptionHandler(Throwable t, HttpServletResponse response) throws IOException {
        String msg = JSON.toJSONString(JsonResult.fail(BaseError.API_FLOW.getCode(), BaseError.API_FLOW.getMsg()));
        String errorMsg = t.getMessage();
        if (null != errorMsg) {
            if (t.getMessage().contains(SentinelExceptionEnum.DEGRADE.getType())) {
                msg = JSON.toJSONString(JsonResult.fail(BaseError.API_DEGRADE.getCode(), BaseError.API_DEGRADE.getMsg()));
            }
        }
        ioWrite(response.getOutputStream(), msg);
        return new ModelAndView();
    }

    private ModelAndView globalExceptionHandler(Throwable t, HttpServletRequest request,
                                                HttpServletResponse response) throws IOException {
        log.error("Exception [{} -> {}]:", request.getRequestURI(), t.getMessage(), t);
        ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(BaseError.SYSTEM_ERROR.getCode(),
                BaseError.SYSTEM_ERROR.getMsg())));
        return new ModelAndView();
    }

    private ModelAndView bizExceptionHandler(Throwable t, HttpServletResponse response) throws IOException {
        String code = null != ((BizException) t).getCode() ? ((BizException) t).getCode() :
                BaseError.SYSTEM_ERROR.getCode();
        response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        ioWrite(response.getOutputStream(), JSON.toJSONString(JsonResult.fail(code, t.getMessage())));
        return new ModelAndView();
    }

    private void ioWrite(ServletOutputStream outputStream, String msg) {
        IoUtil.writeUtf8(outputStream, true, msg);
    }


    private Map<String, Object> getErrorAttributes(HttpServletRequest request) {
        WebRequest webRequest = new ServletWebRequest(request);
        return this.errorAttributes.getErrorAttributes(webRequest, ErrorAttributeOptions.defaults());
    }
}