package cn.com.duiba.local.wolf.entity;

import lombok.Data;

/**
 * Controller响应json统一格式
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @since 0.0.1
 **/
@Data
public class JsonResult<T> {

    /**
     * 默认错误码
     */
    private static final String DEFAULT_FAIL_CODE = "-1";

    /**
     * 默认成功码
     */
    private static final String DEFAULT_SUCCESS_CODE = "0";

    /**
     * 操作是否成功
     */
    private boolean success;

    /**
     * 描述
     */
    private String desc;

    /**
     * 数据包体
     */
    private T data;

    /**
     * 返回code
     */
    private String code;

    /**
     * 当前服务器时间戳
     */
    private long timestamp;

    private JsonResult() {
    }

    /**
     * 带自定义响应码的描述信息数据对象Result
     *
     * @param code 错误码
     * @param desc 失败描述
     * @param <T>  数据包泛型
     * @return Result
     */
    public static <T> JsonResult<T> fail(String code, String desc) {
        JsonResult<T> result = new JsonResult<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(false);
        result.setCode(code);
        result.setDesc(desc);
        return result;
    }

    /**
     * 使用默认错误码,描述错误信息(统一使用默认错误码)
     *
     * @param desc 失败描述
     * @param <T>  数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> fail(String desc) {
        return fail(DEFAULT_FAIL_CODE, desc);
    }

    /**
     * 带有数据的失败返回
     *
     * @param code 失败码
     * @param desc 失败信息
     * @param data 数据
     * @param <T>  数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> fail(String code, String desc, T data) {
        JsonResult<T> result = new JsonResult<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(false);
        result.setCode(code);
        result.setDesc(desc);
        result.setData(data);
        return result;
    }

    /**
     * 带有数据的失败返回,统一使用默认错误码
     *
     * @param desc 失败信息
     * @param data 数据
     * @param <T>  数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> fail(String desc, T data) {
        return fail(DEFAULT_FAIL_CODE, desc, data);
    }

    /**
     * 使用自定义成功码的成功数据对象构建
     *
     * @param code 成功码
     * @param data 数据
     * @param <T>  数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> success(String code, T data) {
        JsonResult<T> result = new JsonResult<>();
        result.setTimestamp(System.currentTimeMillis());
        result.setSuccess(true);
        result.setCode(code);
        result.setData(data);
        result.setDesc("OK");
        return result;
    }

    /**
     * 统一使用默认成功码的成功数据对象构建
     *
     * @param data 数据
     * @param <T>  数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> success(T data) {
        return success(DEFAULT_SUCCESS_CODE, data);
    }

    /**
     * 成功,不返回任何数据,统一使用默认成功码
     *
     * @param <T> 数据包泛型
     * @return JsonResult
     */
    public static <T> JsonResult<T> success() {
        return success(DEFAULT_SUCCESS_CODE, null);
    }
}