package cn.com.duiba.nezha.alg.alg.adx;

import com.alibaba.fastjson.JSON;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.alg.vo.AdxDo;

public class AdxBidding {

    /**
     * ADX出价
     *
     * @param adxDoInfo 模型CTR, 统计CTR，统计点击价值(分/单次点击)，最低minRoi(小数，如1.50)
     * @param adxBiddingRoiFactor  最低ROI调节因子
     * @return (adx标准出价:分/千次曝光)
     */

    public static Long getAdxParPrice(AdxDo adxDoInfo, Double adxBiddingRoiFactor) {


        //设置默认值
        Double defaultPreCtr  = 0.07; //默认ctr
        Double defaultStatClickValue = 0.10*100; //默认点击价值
        Double defaultMinRoi = 1.00; //默认最低ROI
        Double defaultRoiFactor = 1.00; //默认ROI调节因子
        Double roiFactorLowerLimit = 0.80; //ROI调节因子下限
        Double roiFactorUpperLimit = 1.30; //ROI调节因子上限

        if(adxBiddingRoiFactor == null || adxBiddingRoiFactor < roiFactorLowerLimit || adxBiddingRoiFactor > roiFactorUpperLimit){
            adxBiddingRoiFactor = defaultRoiFactor;
        }

        Long parPrice = Math.round(Math.floor(defaultPreCtr * defaultStatClickValue * 1000/(defaultMinRoi * adxBiddingRoiFactor)));


        if (AssertUtil.isNotEmpty(adxDoInfo)){

            Double currentPreValue = adxDoInfo.getCurrentPreValue(); //模型CTR
            Double statCtr = adxDoInfo.getStatCtr(); //统计CTR
            Double statClickValue = adxDoInfo.getStatClickValue(); //统计点击价值(分/单次点击)
            Double minRoi = adxDoInfo.getMinRoi(); //最低minRoi(小数，如1.50)

            //预估preCtr默认值
            Double preCtr = getPreCTR(currentPreValue, statCtr);
            if (preCtr == null || preCtr <= 0.00 || preCtr >= 1.00) {
                preCtr = defaultPreCtr;
            }
            //System.out.println("testPreCtr:" + preCtr);

            // 统计点击价值默认值
            if (statClickValue == null || statClickValue <= 0.00) {
                statClickValue = defaultStatClickValue;
            }

            // 最低ROI默认值
            if (minRoi == null || minRoi < 1.00) {
                minRoi = defaultMinRoi;
            }

            //adx标准出价 = 预估的收益回报/（最低ROI*最低ROI调节因子）*1000
            parPrice = Math.round(Math.floor(preCtr * statClickValue * 1000/(minRoi * adxBiddingRoiFactor)));

        }
        return parPrice;
    }





    /**
     * 预估CTR
     *
     * @param modelPreValue 模型CTR
     * @param statValue 统计CTR
     * @return 预估CTR
     */

    public static Double getPreCTR(Double modelPreValue, Double statValue) {

        //设置ctr默认值
        Double ret = 0.07;

        Double preCtrFactor = 0.0000; // 预估CTR调节因子
        Double vUpperLimit = 0.9999; // 点击率上限
        Double vLowerLimit = 0.0000; // 点击率下限
        Double fUpperLimit = 3.00; // 比例上限
        Double fLowerLimit = 0.20; // 比例下限

        // 统计CTR默认值
        if (statValue == null || statValue < vLowerLimit || statValue > vUpperLimit) {
            statValue = ret;
        }

        // 模型CTR默认值
        if (modelPreValue == null || modelPreValue < vLowerLimit || modelPreValue > vUpperLimit) {
            ret = statValue;

        } else{

            // 融合CTR
            if (modelPreValue/statValue > fLowerLimit && modelPreValue/statValue < fUpperLimit){
                ret = preCtrFactor * modelPreValue + (1 - preCtrFactor) * statValue;
            } else {
                ret = statValue;
            }

        }

        return ret;
    }







    //单元测试
    public static void main(String[] args) {

        try {

            AdxDo adxDoInfo = new AdxDo();
            adxDoInfo.setCurrentPreValue(null);
            adxDoInfo.setStatCtr(0.0);
            adxDoInfo.setStatClickValue(0.00);
            adxDoInfo.setMinRoi(0.0);
            Double roiFactor=0.00;

            Long testParPrice = AdxBidding.getAdxParPrice(adxDoInfo, roiFactor);

            System.out.println("testGetAdxParPrice:" + JSON.toJSONString(testParPrice));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
