package cn.com.duiba.nezha.alg.alg.adx.rcmd2;

import cn.com.duiba.nezha.alg.alg.adx.rtbbid2.AdxRoiFactor;
import cn.com.duiba.nezha.alg.alg.vo.adx.flowfilter.AdxIndexStatDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.rcmd2.AdIdeaDo;
import cn.com.duiba.nezha.alg.alg.vo.adx.rcmd2.IdeaUnitDo;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;

public class AdxRcmdBase {

    private static final Logger logger = LoggerFactory.getLogger(AdxRcmdBase.class);


    /**
     * 获取交集（返回列表2）
     *
     * @param list1 列表1 --原始集
     * @param list2 列表2 --召回集
     */
    public static List<AdIdeaDo> getIntersection(List<AdIdeaDo> list1, List<AdIdeaDo> list2) {

        if (AssertUtil.isAnyEmpty(list1, list2)) {
            return null;
        }

        return list1.stream().filter(a -> list2.stream().anyMatch(b -> {

            if (!Objects.equals(a.getIdeaId(), b.getIdeaId())) {
                return false;
            }

            List<IdeaUnitDo> collect = a.getIdeaUnitDos().stream().filter(c -> b.getIdeaUnitDos().stream().anyMatch(d -> {


                if (!Objects.equals(c.getIdeaUnitId(), d.getIdeaUnitId())) {
                    return false;
                }
                c.setStatAdEcpm(d.getStatAdEcpm());
                c.setIsNew(d.getIsNew());
                c.setLast1DayAdCost(d.getLast1DayAdCost());
                c.setLast1DayExpCnt(d.getLast1DayExpCnt());
                c.setRatioType(d.getRatioType());

                return true;
            })).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(collect)) {
                return false;
            }

//            a.setIsColdStart(b.getIsColdStart());
            a.setIdeaUnitDos(collect);
            return true;

        })).collect(Collectors.toList());


    }


    /**
     * 随机选取
     *
     * @param ideaUnitList 展平列表
     * @param limitSize    选取个数(小于展平列表size)
     */
    public static List<AdIdeaDo> getRandomList(List<IdeaUnitDo> ideaUnitList, Integer limitSize, Map<Long, Boolean> coldStartMap) {

        List<IdeaUnitDo> candidateList;
        if (ideaUnitList.size() <= limitSize) {
            candidateList = ideaUnitList;
        } else {
            //随机选取
            Collections.shuffle(ideaUnitList);
            candidateList = ideaUnitList.subList(0, limitSize);
        }

        //candidateList转化为candidates
        List<AdIdeaDo> candidates = getCovertList(candidateList);
        candidates.forEach(s -> s.setIsColdStart(coldStartMap.get(s.getIdeaId())));

        return candidates;
    }


    /**
     * 列表转化
     *
     * @param ideaUnitList 展平列表
     * @return adIdeaDos 原始列表
     */
    public static List<AdIdeaDo> getCovertList(List<IdeaUnitDo> ideaUnitList) {

        List<AdIdeaDo> adIdeaDoList = new ArrayList<>();
        ideaUnitList.stream().forEach(ido -> {
            AdIdeaDo tmpAdIdeaDo = new AdIdeaDo();
            tmpAdIdeaDo.setIdeaId(ido.getAdIdeaId());

            List<IdeaUnitDo> tmpIdeaUnitDos = new ArrayList<>();
            tmpIdeaUnitDos.add(ido);
            tmpAdIdeaDo.setIdeaUnitDos(tmpIdeaUnitDos);

            adIdeaDoList.add(tmpAdIdeaDo);
        });

        //根据计划id聚合
        Map<Long, AdIdeaDo> adIdeaMap = new HashMap<>();
        adIdeaDoList.stream().forEach(ado -> {
            AdIdeaDo tmp = adIdeaMap.get(ado.getIdeaId());
            if (tmp == null) {
                adIdeaMap.put(ado.getIdeaId(), ado);
            } else {
                tmp.getIdeaUnitDos().addAll(ado.getIdeaUnitDos());
            }
        });

        List<AdIdeaDo> ret = new ArrayList<AdIdeaDo>(adIdeaMap.values());
        return ret;
    }


    /**
     * 按比例区间聚合
     *
     * @param ideaUnitDos 展平列表
     * @return 创意按比例区间聚合
     */
    public static Map<Integer, List<IdeaUnitDo>> aggregateByRatioType(List<IdeaUnitDo> ideaUnitDos) {

        Map<Integer, List<IdeaUnitDo>> ideaUnitMap = new HashMap<>();
        ideaUnitDos.stream().forEach(ado -> {

            Integer ratioType = ado.getRatioType();
            if (ratioType == null) {
                logger.warn("ratioType is null, adIdeaId is {}, ideaUnitId is {}", ado.getAdIdeaId(), ado.getIdeaUnitId());
                return;
            }

            List<IdeaUnitDo> subList = ideaUnitMap.get(ratioType);
            if (subList == null) {
                List<IdeaUnitDo> tmp = new ArrayList<>();
                tmp.add(ado);
                ideaUnitMap.put(ratioType, tmp);
            } else {
                subList.add(ado);
            }

        });

        return ideaUnitMap;
    }


    /**
     * 计算冷启动池和优选池大小
     */
    public static long getRecallSize(List<IdeaUnitDo> newList, List<IdeaUnitDo> oldList,
                                     int limitSize, double coldRate, String type) {

        long coldSize = DataUtil.double2Long(limitSize * coldRate);
        long bestSize = limitSize - coldSize;

        if (newList.size() <= coldSize && oldList.size() <= bestSize) {
            coldSize = newList.size();
            bestSize = oldList.size();

        } else if (newList.size() <= coldSize) {
            coldSize = newList.size();
            bestSize = limitSize - coldSize;

        } else if (oldList.size() <= bestSize) {
            bestSize = oldList.size();
            coldSize = limitSize - bestSize;
        }

        long ret = 0L;
        if ("COLD".equals(type)) {
            return coldSize;
        } else if ("BEST".equals(type)) {
            return bestSize;
        }
        return ret;
    }


    /**
     * 计算统计指标
     */
    public static AdxIndexStatDo getStatusVo(AdxIndexStatDo adxStatBaseDo, Integer bidMode, Double target) {

        AdxIndexStatDo ret = Optional.ofNullable(adxStatBaseDo).orElse(new AdxIndexStatDo());

        Boolean isConfident = isConfident(adxStatBaseDo);
        Long bidCnt = AdxIndexStatDo.getBidCnt(adxStatBaseDo);
        Long sucCnt = AdxIndexStatDo.getBidSucCnt(adxStatBaseDo);
        Long expCnt = AdxIndexStatDo.getExpCnt(adxStatBaseDo);
        Long clickCnt = AdxIndexStatDo.getClickCnt(adxStatBaseDo);
        Long launchCnt = AdxIndexStatDo.getLaunchCnt(adxStatBaseDo);
        Long adxCost = AdxIndexStatDo.getAdxCost(adxStatBaseDo);
        Long adCost = AdxIndexStatDo.getAdCost(adxStatBaseDo);

        ret.setIsConfident(isConfident);
        ret.setBid(bidCnt);
        ret.setBidSuc(sucCnt);
        ret.setExp(expCnt);
        ret.setClick(clickCnt);
        ret.setAdvertLaunch(launchCnt);
        ret.setAdxCost(adxCost);
        ret.setAdCost(adCost);

        double avgFactor = getAvgFactor(adxStatBaseDo);
        double actRoi = DataUtil.division((adCost + 100), (adxCost + 100), 4); //实际roi
        double actCpc = DataUtil.division((adxCost + 100), (clickCnt + 1), 4);  //实际cpc
        double actValue = bidMode == 2 ? DataUtil.division((adxCost + 10 * target), (clickCnt + 10), 4)
                : DataUtil.division((adxCost + 1000 * target), (adCost + 1000), 4); //目标实际值
        double bias = getBias(ret, target, bidMode, actValue, 0.0);
        double sucRate = DataUtil.division(sucCnt, (bidCnt + 100), 4); //实际suc

        ret.setAvgfactor(avgFactor);
        ret.setBias(bias);
        ret.setActRoi(actRoi);
        ret.setActCpc(actCpc);
        ret.setActValue(actValue);
        ret.setSucRate(sucRate);

        return ret;
    }


    /**
     * 计算偏差
     * 分成偏差 = ((adx消耗/广告消耗) - 目标分成）/ 目标分成
     * cpc偏差 = ((adx消耗/adx点击) - 目标cpc）/ 目标cpc
     */
    public static double getBias(AdxIndexStatDo statDo, Double target, Integer bidMode,
                                 double actValue, Double defaultValue) {

        double ret = defaultValue;
        if (AssertUtil.isAllNotEmpty(statDo, target, bidMode, actValue)) {
            ret = DataUtil.division((actValue - target), target, 4);

            if (statDo.getIsConfident()) {
                // 置信
                ret = MathUtil.stdwithBoundary(ret, -0.5, 0.5);

            } else {

                if (isCostConfident(statDo)) {
                    // 消耗置信
                    ret = MathUtil.stdwithBoundary(ret, -0.2, 0.2);
                } else {
                    // 曝光置信
                    ret = MathUtil.stdwithBoundary(ret, -0.1, 0.1);
                }
            }

        }

        return ret;
    }


    /**
     * 判断消耗是否置信
     *
     * @param adxIndexStatDo
     * @return
     */
    public static boolean isCostConfident(AdxIndexStatDo adxIndexStatDo) {
        boolean ret = false;
        if (adxIndexStatDo != null && AdxIndexStatDo.getAdxCost(adxIndexStatDo) > 2000) {
            ret = true;
        }
        return ret;
    }

    /**
     * 判断曝光是否置信
     *
     * @param adxIndexStatDo
     * @return
     */
    public static boolean isExpConfident(AdxIndexStatDo adxIndexStatDo) {
        boolean ret = false;
        if (adxIndexStatDo != null && AdxIndexStatDo.getExpCnt(adxIndexStatDo) > 500) {
            ret = true;
        }
        return ret;
    }

    /**
     * 判断是否置信
     *
     * @param adxIndexStatDo
     * @return
     */
    public static boolean isConfident(AdxIndexStatDo adxIndexStatDo) {
        boolean ret = false;
        if (isCostConfident(adxIndexStatDo) && isExpConfident(adxIndexStatDo)) {
            ret = true;
        }
        return ret;
    }


    public static Double getStatCtr(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if (adxIndexStatDo != null && AdxIndexStatDo.getExpCnt(adxIndexStatDo) > 200) {
            ret = DataUtil.division(AdxIndexStatDo.getClickCnt(adxIndexStatDo),
                    AdxIndexStatDo.getExpCnt(adxIndexStatDo), 5);
        }
        return ret;
    }

    public static Double getStatCtr(AdxIndexStatDo adxIndexStatDo, Double baseValue) {
        Double ret = getStatCtr(adxIndexStatDo);
        ret = MathUtil.mean(ret, baseValue, 0.5);
        return ret;
    }

    public static Double getStatCtr(Double base, AdxIndexStatDo minStat,
                                    AdxIndexStatDo hourStat, AdxIndexStatDo dayStat) {

        double r1 = 0.5, r2 = 0.3, r3 = 0.2;
        double mergeExpCnt = r1 * AdxIndexStatDo.getExpCnt(minStat)
                + r2 * AdxIndexStatDo.getExpCnt(hourStat)
                + r3 * AdxIndexStatDo.getExpCnt(dayStat);
        double mergeClickCnt = r1 * AdxIndexStatDo.getClickCnt(minStat)
                + r2 * AdxIndexStatDo.getClickCnt(hourStat)
                + r3 * AdxIndexStatDo.getClickCnt(dayStat);

        Double stat = DataUtil.division(mergeClickCnt, mergeExpCnt + 1L, 5);
        Double ret = MathUtil.mean(base, stat, 0.7);
        return DataUtil.formatDouble(ret, 5);
    }

    public static Double getStatPvLaunch(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if (adxIndexStatDo != null && AdxIndexStatDo.getClickCnt(adxIndexStatDo) > 100) {
            ret = DataUtil.division(AdxIndexStatDo.getLaunchCnt(adxIndexStatDo),
                    AdxIndexStatDo.getClickCnt(adxIndexStatDo), 5);
        }
        return ret;
    }

    public static Double getStatPvLaunch(AdxIndexStatDo adxIndexStatDo, Double baseValue) {
        Double ret = getStatPvLaunch(adxIndexStatDo);
        ret = MathUtil.mean(ret, baseValue, 0.5);
        return ret;
    }

    public static Double getStatPvLaunch(Double base, AdxIndexStatDo minStat,
                                         AdxIndexStatDo hourStat, AdxIndexStatDo dayStat) {

        double r1 = 0.5, r2 = 0.3, r3 = 0.2;
        double mergeClickCnt = r1 * AdxIndexStatDo.getClickCnt(minStat)
                + r2 * AdxIndexStatDo.getClickCnt(hourStat)
                + r3 * AdxIndexStatDo.getClickCnt(dayStat);
        double mergeLaunchCnt = r1 * AdxIndexStatDo.getLaunchCnt(minStat)
                + r2 * AdxIndexStatDo.getLaunchCnt(hourStat)
                + r3 * AdxIndexStatDo.getLaunchCnt(dayStat);

        Double stat = DataUtil.division(mergeLaunchCnt, mergeClickCnt + 1L, 5);
        Double ret = MathUtil.mean(base, stat, 0.7);
        return DataUtil.formatDouble(ret, 5);
    }


    public static Double getStatArpu(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if (adxIndexStatDo != null && AdxIndexStatDo.getLaunchCnt(adxIndexStatDo) > 50) {
            ret = DataUtil.division(AdxIndexStatDo.getAdCost(adxIndexStatDo),
                    AdxIndexStatDo.getLaunchCnt(adxIndexStatDo), 5);
        }
        return ret;
    }

    public static Double getStatArpu(AdxIndexStatDo adxIndexStatDo, Double baseValue) {
        Double ret = getStatArpu(adxIndexStatDo);
        ret = MathUtil.mean(ret, baseValue, 0.5);
        return ret;
    }

    public static Double getStatArpu(Double base, AdxIndexStatDo minStat,
                                     AdxIndexStatDo hourStat, AdxIndexStatDo dayStat) {

        double r1 = 0.5, r2 = 0.3, r3 = 0.2;
        double mergeLaunchCnt = r1 * AdxIndexStatDo.getLaunchCnt(minStat)
                + r2 * AdxIndexStatDo.getLaunchCnt(hourStat)
                + r3 * AdxIndexStatDo.getLaunchCnt(dayStat);
        double mergeAdCost = r1 * AdxIndexStatDo.getAdCost(minStat)
                + r2 * AdxIndexStatDo.getAdCost(hourStat)
                + r3 * AdxIndexStatDo.getAdCost(dayStat);

        Double stat = DataUtil.division(mergeAdCost, mergeLaunchCnt + 1L, 5);
        Double ret = MathUtil.mean(base, stat, 0.7);
        return DataUtil.formatDouble(ret, 5);
    }

    public static Double getStatEcpm(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if (adxIndexStatDo != null && AdxIndexStatDo.getExpCnt(adxIndexStatDo) > 500) {
            ret = DataUtil.division(AdxIndexStatDo.getAdCost(adxIndexStatDo) * 1000,
                    AdxIndexStatDo.getExpCnt(adxIndexStatDo), 5);
        }
        return ret;
    }

    public static Double getStatEcpm(AdxIndexStatDo adxIndexStatDo, Double baseValue) {
        Double ret = getStatEcpm(adxIndexStatDo);
        ret = MathUtil.mean(ret, baseValue, 0.5);
        return ret;
    }

    public static Double getStatEcpm(Double base, AdxIndexStatDo minStat,
                                     AdxIndexStatDo hourStat, AdxIndexStatDo dayStat) {

        double r1 = 0.5, r2 = 0.3, r3 = 0.2;
        double mergeExpCnt = r1 * AdxIndexStatDo.getExpCnt(minStat)
                + r2 * AdxIndexStatDo.getExpCnt(hourStat)
                + r3 * AdxIndexStatDo.getExpCnt(dayStat);
        double mergeAdCost = r1 * AdxIndexStatDo.getAdCost(minStat)
                + r2 * AdxIndexStatDo.getAdCost(hourStat)
                + r3 * AdxIndexStatDo.getAdCost(dayStat);

        Double stat = DataUtil.division(mergeAdCost * 1000, mergeExpCnt + 1L, 5);
        Double ret = MathUtil.mean(base, stat, 0.7);
        return DataUtil.formatDouble(ret, 5);
    }

    public static Double getStatCpm(AdxIndexStatDo adxIndexStatDo) {
        Double ret = null;
        if (adxIndexStatDo != null && AdxIndexStatDo.getExpCnt(adxIndexStatDo) > 500) {
            ret = DataUtil.division(AdxIndexStatDo.getAdxCost(adxIndexStatDo) * 1000,
                    AdxIndexStatDo.getExpCnt(adxIndexStatDo), 5);
        }
        return ret;
    }

    public static Double getStatCpm(AdxIndexStatDo adxIndexStatDo, Double baseValue) {
        Double ret = getStatCpm(adxIndexStatDo);
        if (ret == null) {
            ret = baseValue;
        }
        return ret;
    }

    public static Double getStatCpm(Double base, AdxIndexStatDo minStat,
                                    AdxIndexStatDo hourStat, AdxIndexStatDo dayStat) {

        double r1 = 0.5, r2 = 0.3, r3 = 0.2;
        double mergeExpCnt = r1 * AdxIndexStatDo.getExpCnt(minStat)
                + r2 * AdxIndexStatDo.getExpCnt(hourStat)
                + r3 * AdxIndexStatDo.getExpCnt(dayStat);
        double mergeAdxCost = r1 * AdxIndexStatDo.getAdxCost(minStat)
                + r2 * AdxIndexStatDo.getAdxCost(hourStat)
                + r3 * AdxIndexStatDo.getAdxCost(dayStat);

        Double stat = DataUtil.division(mergeAdxCost * 1000, mergeExpCnt + 1L, 5);
        Double ret = MathUtil.mean(base, stat, 0.7);
        return DataUtil.formatDouble(ret, 5);
    }

    /**
     * 计算平均维稳因子
     *
     * @return
     */
    public static double getAvgFactor(AdxIndexStatDo adxIndexStatDo) {
        Double ret = 1.0;
        if (adxIndexStatDo != null && adxIndexStatDo.getBid() != null && adxIndexStatDo.getFactor() != null) {
            ret = DataUtil.division(adxIndexStatDo.getFactor() + 10 * 1.0, AdxIndexStatDo.getBidCnt(adxIndexStatDo) + 10L, 5);
        }
        return ret;
    }
}
