package cn.com.duiba.nezha.alg.alg.adx;

import cn.com.duiba.nezha.alg.alg.enums.AdxIndex;
import cn.com.duiba.nezha.alg.alg.enums.AdxLevel;
import cn.com.duiba.nezha.alg.alg.enums.AdxStrategy;
import cn.com.duiba.nezha.alg.alg.vo.*;
import cn.com.duiba.nezha.alg.common.util.AssertUtil;
import cn.com.duiba.nezha.alg.common.util.DataUtil;
import cn.com.duiba.nezha.alg.common.util.MathUtil;
import com.alibaba.fastjson.JSON;

import java.util.*;
import java.util.zip.DeflaterOutputStream;


public class StrategyBid {


    /**
     * 策略一：实时调节ROI因子，预估出价
     *
     * @param adxDoInfo
     * @param adxRoiFactorDoInfo
     * @param advertConsumeInfo
     * @return level, price
     */
    public static Map<String, String> getStrategyFirBid(AdxDo adxDoInfo,
                                                        AdxRoiFactorDo adxRoiFactorDoInfo,
                                                        Map<String, Long> advertConsumeInfo) {

        // ROI调节因子默认值
        Double defaultRoiFactor = 1.00;

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_ZER.getCode();
        Long price = AdxBidding.getAdxParPrice(adxDoInfo, defaultRoiFactor, advertConsumeInfo);

        if (AssertUtil.isNotEmpty(adxRoiFactorDoInfo)) {

            Double roiFactor = adxRoiFactorDoInfo.getAdxRoiFactor();
            price = AdxBidding.getAdxParPrice(adxDoInfo, roiFactor, advertConsumeInfo);
        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略二：价格试探，寻找最优出价
     *
     * @param priceExplorationDoInfo
     * @param defaultPrice
     * @param minPrice
     * @return level, price
     */
    public static Map<String, String> getStrategySecBid(AdxPriceExplorationDo priceExplorationDoInfo,
                                                        Long defaultPrice, Long minPrice) {


        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_TWO.getCode();
        Long price = defaultPrice;

        if (AssertUtil.isNotEmpty(priceExplorationDoInfo)) {

            Map<String, Double> priceMap = priceExplorationDoInfo.getPriceExploreMap();
            Map<String, Double> FlowRateMap = priceExplorationDoInfo.getPriceFlowRateMap();
            Map<String, String> result = flowSplit(FlowRateMap, priceMap, defaultPrice);

            level = result.get("level");
            price = DataUtil.double2Long(DataUtil.string2Double(result.get("price")));

        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略三：因子试探，寻找最优，预估出价
     *
     * @param adxDoInfo
     * @param factorExplorationDoInfo
     * @param advertConsumeInfo
     * @param minPrice
     * @return level, price
     */
    public static Map<String, String> getStrategyThiBid(AdxDo adxDoInfo,
                                                        AdxFactorExplorationDo factorExplorationDoInfo,
                                                        Map<String, Long> advertConsumeInfo,
                                                        Long minPrice) {


        // ROI调节因子默认值
        Double defaultRoiFactor = 1.00;

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_TWO.getCode();
        Long price = AdxBidding.getAdxParPrice(adxDoInfo, defaultRoiFactor, advertConsumeInfo);

        if (AssertUtil.isNotEmpty(factorExplorationDoInfo)) {

            Map<String, Double> factorMap = factorExplorationDoInfo.getFactorExploreMap();
            Map<String, Double> FlowRateMap = factorExplorationDoInfo.getFactorFlowRateMap();
            Map<String, String> result = flowSplit(FlowRateMap, factorMap, defaultRoiFactor);

            level = result.get("level");
            Double roiFactor = DataUtil.string2Double(result.get("factor"));
            price = AdxBidding.getAdxParPrice(adxDoInfo, roiFactor, advertConsumeInfo);

        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.Long2String(price));

        return retMap;
    }


    /**
     * 策略二：流量分配，返回level和试探出价
     *
     * @param flowRateMap
     * @param priceMap
     * @param defaultValue
     * @return
     */
    public static Map<String, String> flowSplit(Map<String, Double> flowRateMap, Map<String, Double> priceMap, Long defaultValue) {

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_TWO.getCode();
        Double price = DataUtil.toDouble(defaultValue);

        Double sumRate = 0.00;
        Double[] sumRateList = new Double[AdxLevel.values().length - 1];

        if (AssertUtil.isAllNotEmpty(flowRateMap, priceMap)) {

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    sumRate += nullToDefault(flowRateMap.get(key), 0.0);
                    sumRateList[i - 1] = sumRate;
                }
            }

            double random = Math.random();

            for (int i = 0; i < sumRateList.length; i++) {
                double bound = sumRateList[i];

                if (random <= bound) {
                    level = DataUtil.Integer2String(i + 1);
                    price = priceMap.get(level);
                    if (price == null) {
                        price = DataUtil.toDouble(defaultValue);
                    }
                    break;
                }
            }
        }

        retMap.put("level", level);
        retMap.put("price", DataUtil.double2String(price));
        return retMap;
    }


    /**
     * 策略三：流量分配，返回level和试探因子
     *
     * @param flowRateMap
     * @param factorMap
     * @param defaultValue
     * @return
     */
    public static Map<String, String> flowSplit(Map<String, Double> flowRateMap, Map<String, Double> factorMap, Double defaultValue) {

        Map<String, String> retMap = new HashMap<>();
        String level = AdxLevel.ADX_LEVEL_TWO.getCode();
        Double factor = defaultValue;

        Double sumRate = 0.00;
        Double[] sumRateList = new Double[AdxLevel.values().length - 1];

        if (AssertUtil.isAllNotEmpty(flowRateMap, factorMap)) {

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    sumRate += nullToDefault(flowRateMap.get(key), 0.0);
                    sumRateList[i - 1] = sumRate;
                }
            }

            double random = Math.random();

            for (int i = 0; i < sumRateList.length; i++) {
                double bound = sumRateList[i];

                if (random <= bound) {
                    level = DataUtil.Integer2String(i + 1);
                    factor = factorMap.get(level);
                    if (factor == null) {
                        factor = defaultValue;
                    }
                    break;
                }
            }
        }

        retMap.put("level", level);
        retMap.put("factor", DataUtil.double2String(factor));
        return retMap;
    }


    /**
     * 获取指定策略数据
     *
     * @param arrayListInfo
     * @param strategyLabel
     * @return 获取指定策略数据
     */
    public static List<AdxLevelDo> getStrategyInfo(ArrayList<AdxStrategyDo> arrayListInfo, String strategyLabel) {

        List<AdxLevelDo> ret = new ArrayList<>();
        if (AssertUtil.isNotEmpty(arrayListInfo)) {
            for (int i = 0; i < AdxStrategy.values().length; i++) {
                AdxStrategyDo ind = arrayListInfo.get(i);
                if (strategyLabel.equals(ind.getStrategy())) {
                    ret = ind.getLevelDoList();
                }
            }
        }
        return ret;
    }


    /**
     * 合计所有level中相应指标数据
     *
     * @param levelListInfo
     * @param indexLabel
     * @param defaultValue
     * @return 默认值
     */
    public static Long getSumLevelIndex(List<AdxLevelDo> levelListInfo, String indexLabel, Long defaultValue) {

        Long ret = 0L;
        if (AssertUtil.isNotEmpty(levelListInfo)) {
            for (AdxLevelDo levelDo : levelListInfo) {
                if (AssertUtil.isNotEmpty(levelDo)) {
                    Map<String, Long> valueMap = levelDo.getValueMap();

                    if (AssertUtil.isNotEmpty(valueMap)
                            && valueMap.get(indexLabel) != null
                            && valueMap.get(indexLabel) >= 0L) {

                        ret += valueMap.get(indexLabel);
                    }

                } else {
                    ret += defaultValue;
                }
            }
        }
        return ret;
    }


    /**
     * 获取不同level的相应指标数据
     *
     * @param levelListInfo
     * @param indexLabel
     * @param defaultValue
     * @return 默认值
     */
    public static Map<String, Long> getLevelIndex(List<AdxLevelDo> levelListInfo, String indexLabel, Long defaultValue) {

        Map<String, Long> retMap = new HashMap<>();
        Long val = defaultValue;
        String key;
        for (AdxLevel adxLevel : AdxLevel.values()) {
            key = adxLevel.getCode();

            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                if (AssertUtil.isNotEmpty(levelListInfo)) {

                    int i = DataUtil.toInt(DataUtil.string2Long(key));
                    AdxLevelDo ind = levelListInfo.get(i);

                    if (AssertUtil.isNotEmpty(ind)) {
                        Map<String, Long> indMap = ind.getValueMap();

                        if (AssertUtil.isNotEmpty(indMap)
                                && indMap.get(indexLabel) != null
                                && indMap.get(indexLabel) >= 0L) {

                            val = indMap.get(indexLabel);
                        }
                    }
                }
                retMap.put(key, val);
            }
        }
        return retMap;
    }


    /**
     * 计算指定策略的竞价成功率
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Double getSucRate(List<AdxLevelDo> levelListInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultCnt = 50L;

        if (AssertUtil.isNotEmpty(levelListInfo)) {
            //竞价量
            Long bidCnt = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.BID.getCode(), defaultValue);
            //竞价成功量
            Long sucCnt = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.BID_SUC.getCode(), defaultValue);

            if (bidCnt >= defaultCnt) {
                ret = StrategyBid.getNormalValue(DataUtil.division(sucCnt, bidCnt, 6), 0.0, 0.0, 1.0);
            }
        }
        return ret;
    }


    /**
     * 计算创意/资源位整体竞价成功率
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getSucRate(Map<String, Long> mapInfo) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultCnt = 50L;

        if (AssertUtil.isNotEmpty(mapInfo)) {
            //竞价量
            Long bidCnt = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.BID.getCode()), defaultValue);
            //竞价成功量
            Long sucCnt = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.BID_SUC.getCode()), defaultValue);

            if (bidCnt >= defaultCnt) {
                ret = StrategyBid.getNormalValue(DataUtil.division(sucCnt, bidCnt, 6), 0.0, 0.0, 1.0);
            }
        }
        return ret;
    }


    /**
     * 计算不同level的竞价成功率
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Map<String, Double> getLevelSucRate(List<AdxLevelDo> levelListInfo) {

        Map<String, Double> retMap = new HashMap<>();
        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultCnt = 10L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isNotEmpty(levelListInfo)) {
                    //竞价量
                    Map<String, Long> bidCnt = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.BID.getCode(), defaultValue);
                    //竞价成功量
                    Map<String, Long> sucCnt = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.BID_SUC.getCode(), defaultValue);

                    if (bidCnt.get(key) >= defaultCnt) {
                        ret = StrategyBid.getNormalValue(DataUtil.division(sucCnt.get(key), bidCnt.get(key), 6), 0.0, 0.0, 1.0);
                        retMap.put(key, ret);

                    } else {
                        retMap.put(key, 0.0);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 计算指定策略的roi
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Double getRoi(List<AdxLevelDo> levelListInfo, Double targetRoi) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultAdxConsume = 50 * 1000 * 10000L;

        if (AssertUtil.isNotEmpty(levelListInfo)) {
            //adx消耗(分*10000/cpm)
            Long adxConsume = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.ADX_CONSUME.getCode(), defaultValue);
            //广告消耗(分)
            Long advertConsume = StrategyBid.getSumLevelIndex(levelListInfo, AdxIndex.ADVERT_CONSUME.getCode(), defaultValue);

            if (adxConsume >= defaultAdxConsume) {
                ret = DataUtil.formatDouble((DataUtil.division(advertConsume, adxConsume, 13) * 10000 * 1000), 6);

            } else {
                ret = DataUtil.formatDouble((DataUtil.division((advertConsume + defaultAdxConsume * targetRoi / 1000 / 10000),
                        (adxConsume + defaultAdxConsume), 13) * 10000 * 1000), 6);
            }
        }
        return ret;
    }


    /**
     * 计算创意/资源位整体的roi
     *
     * @param mapInfo
     * @return 默认值
     */
    public static Double getRoi(Map<String, Long> mapInfo, Double targetRoi) {

        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultAdxConsume = 50 * 1000 * 10000L;

        if (AssertUtil.isNotEmpty(mapInfo)) {
            //adx消耗(分*10000/cpm)
            Long adxConsume = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.ADX_CONSUME.getCode()), defaultValue);
            //广告消耗(分)
            Long advertConsume = StrategyBid.nullToDefault(mapInfo.get(AdxIndex.ADVERT_CONSUME.getCode()), defaultValue);

            if (adxConsume >= defaultAdxConsume) {
                ret = DataUtil.formatDouble((DataUtil.division(advertConsume, adxConsume, 13) * 10000 * 1000), 6);

            } else {
                ret = DataUtil.formatDouble((DataUtil.division((advertConsume + defaultAdxConsume * targetRoi / 1000 / 10000),
                        (adxConsume + defaultAdxConsume), 13) * 10000 * 1000), 6);
            }
        }

        return ret;
    }


    /**
     * 计算不同level的roi
     *
     * @param levelListInfo
     * @return 默认值
     */
    public static Map<String, Double> getLevelRoi(List<AdxLevelDo> levelListInfo, Double targetRoi) {

        Map<String, Double> retMap = new HashMap<>();
        Double ret = 0.0;
        Long defaultValue = 0L;
        Long defaultAdxConsume = 25 * 1000 * 10000L;

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                if (AssertUtil.isNotEmpty(levelListInfo)) {

                    //adx消耗(分*10000/cpm)
                    Map<String, Long> adxConsume = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.ADX_CONSUME.getCode(), defaultValue);
                    //广告消耗(分)
                    Map<String, Long> advertConsume = StrategyBid.getLevelIndex(levelListInfo, AdxIndex.ADVERT_CONSUME.getCode(), defaultValue);

                    if (adxConsume.get(key) >= defaultAdxConsume) {
                        ret = DataUtil.formatDouble((DataUtil.division(advertConsume.get(key), adxConsume.get(key), 13) * 10000 * 1000), 6);
                        retMap.put(key, ret);

                    } else {
                        ret = DataUtil.formatDouble((DataUtil.division((advertConsume.get(key) + defaultAdxConsume * targetRoi / 1000 / 10000),
                                (adxConsume.get(key)+ defaultAdxConsume), 13) * 10000 * 1000), 6);
                        retMap.put(key, ret);
                    }

                } else {
                    retMap.put(key, 0.0);
                }
            }
        }
        return retMap;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToDefault(Long value, Long defaultValue) {

        Long ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double nullToDefault(Double value, Double defaultValue) {

        Double ret = defaultValue;
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 缺失情况，重置为defaultValue
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Long nullToDefault(Long value, Double defaultValue) {

        Long ret = DataUtil.double2Long(defaultValue);
        if (value != null && value > defaultValue) {
            ret = value;
        }
        return ret;
    }


    /**
     * 获取正常范围内的值
     *
     * @param value
     * @param defaultValue
     * @return 默认值
     */
    public static Double getNormalValue(Double value, Double defaultValue, Double lower, Double upper) {

        Double ret = defaultValue;
        if (value != null) {
            ret = value < lower ? lower : (value > upper ? upper : value);
        }
        return ret;
    }


    /**
     * 分桶函数1，获取右边界（左开右闭区间）
     *
     * @param value
     * @param bucketList 不为空，且不含有空值（未判断）
     * @return 分桶值
     */
    public static Double bucket(Double value, Double[] bucketList) {
        Double ret = 0.00;

        if (value != null && bucketList != null && bucketList.length > 0) {
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];
                if (value <= bound) {
                    ret = bound;
                    break;
                }
            }

            if (value > bucketList[bucketList.length - 1]) {
                ret = bucketList[bucketList.length - 1];
            }
        }
        return ret;
    }


    /**
     * 分桶函数2，获取区间序列（左闭右开区间）
     *
     * @param value
     * @param targetValue
     * @param bucketList  不为空，且不含有空值（未判断）
     * @return
     */
    public static Long bucket(Double value, Double targetValue, Double[] bucketList) {
        long ret = 0;
        if (value != null && targetValue != null && bucketList != null && bucketList.length > 0) {
            ret = bucketList.length - 1;
            for (int i = 0; i < bucketList.length; i++) {
                double bound = bucketList[i];

                if (value < targetValue * bound) {
                    ret = i - 1;
                    break;
                }
            }
        }
        return ret;
    }


    /**
     * 平滑分桶函数（左闭右开区间）
     *
     * @param value
     * @param bucketList 不为空，且不含有空值（未判断）
     * @return
     */
    public static Double getConfidenceWeight(double value, double[] bucketList, double[] weightList) {
        double ret = 0;

        if (bucketList != null && bucketList.length > 0 && weightList != null && weightList.length == bucketList.length) {
            double lastWeight = weightList[0];
            double lastBound = bucketList[0];

            for (int i = 0; i < bucketList.length; i++) {

                double bound = bucketList[i];
                if (value < bound) {
                    double curWeight = weightList[i];
                    double curBound = bucketList[i];
                    if (i > 0) {
                        ret = lastWeight + (curWeight - lastWeight) * (value - lastBound) / (curBound - lastBound);
                    } else {
                        ret = curWeight;
                    }

                    break;
                }

                lastWeight = weightList[i];
                lastBound = bucketList[i];
            }

            if (value >= bucketList[bucketList.length - 1]) {
                ret = weightList[weightList.length - 1];
            }
        }

        return DataUtil.formatDouble(ret, 3);
    }


    /**
     * 选取最优level（左闭右开区间）
     *
     * @param targetRoi
     * @param realRoiLevel
     * @param realSucRateLevel
     * @return level
     */
    public static String selectBestLevel(Double targetRoi,
                                         Map<String, Double> realRoiLevel,
                                         Map<String, Double> realSucRateLevel) {

        Double roiLimit = 0.95;
        if (targetRoi != null && targetRoi > 1.0) {
            roiLimit = Math.min(DataUtil.division(1.30,targetRoi),0.95);
        }

        String ret = AdxLevel.ADX_LEVEL_TWO.getCode();

        if (AssertUtil.isAllNotEmpty(realRoiLevel, realSucRateLevel)) {
            Double tmpScore = -1.0;

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();

                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                    Double roi = realRoiLevel.get(key);
                    Double sucRate = realSucRateLevel.get(key);
                    Double roiDiff = nullToDefault(DataUtil.division(roi, targetRoi), 0.0);
                    Double score = (roiDiff - roiLimit) * sucRate;

                    if (sucRate > 0.01 && score > tmpScore) {
                        ret = key;
                        tmpScore = score;
                    }
                }
            }
        }
        return ret;
    }


    /**
     * 获取最优竞价约束上限
     *
     * @param priceMap
     * @param sucRateMap
     * @param lastUpLimitPrice
     * @return 竞价约束上限
     */
    public static Double selectUpLimitFee(Map<String, Double> priceMap,
                                          Map<String, Double> sucRateMap,
                                          Double lastUpLimitPrice) {

        Double ret = lastUpLimitPrice;
        Double sucLimit = 0.95;

        if (AssertUtil.isAllNotEmpty(priceMap, sucRateMap)) {

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();

                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                    Double sucRate = sucRateMap.get(key);
                    Double price = priceMap.get(key);

                    // 1 非法数据，跳过
                    if (AssertUtil.isAnyEmpty(sucRate, price)) {
                        continue;
                    }

                    // 2 失效判断
                    // 成功率过低 且 当前价格大于历史出价
                    if (sucRate < sucLimit && ret != null && price > ret) {
                        ret = null;
                    }

                    // 3 更新判断
                    if (sucRate >= sucLimit) {
                        // 存在历史，且当前出价更低
                        if (ret != null && price < ret) {
                            ret = price;
                        }
                        // 不存在历史
                        if (ret == null) {
                            ret = price;
                        }
                    }


                }
            }
        }
        return ret;
    }


    /**
     * ROI置信处理（策略3-根据分level调节因子）
     *
     * @param roiMap
     * @param adxConsumeLevelMs
     * @param adxRoiControlDoInfo
     * @return 分level-ROI
     */
    public static Map<String, Double> getRoiFactorRectify(Map<String, Double> roiMap,
                                                          Map<String, Long> adxConsumeLevelMs,
                                                          AdxRoiControlDo adxRoiControlDoInfo) {

        Map<String, Double> retMap = new HashMap<>();

        Long adxConsumeLimit = 300 * 1000 * 10000L;
        Long adxConsumeBaseLimit = 25 * 1000 * 10000L;
        Double baseRoi = StrategyBid.nullToDefault(roiMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 0.0);

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                retMap.put(key, roiMap.get(key));
            }
        }

        AdxFactorExplorationDo lastFactorDoInfo = adxRoiControlDoInfo.getLastFactorExplorationDo();
        if (AssertUtil.isNotEmpty(lastFactorDoInfo)) {

            Map<String, Double> lastFactorMap = lastFactorDoInfo.getFactorExploreMap();
            Double baseFactor = 1.0;
            if (AssertUtil.isNotEmpty(lastFactorMap)) {
                baseFactor = StrategyBid.nullToDefault(lastFactorMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 1.0);
            }

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                    if (AssertUtil.isNotEmpty(adxConsumeLevelMs)) {

                        if (adxConsumeLevelMs.get(key) < adxConsumeLimit
                                && adxConsumeLevelMs.get(AdxLevel.ADX_LEVEL_TWO.getCode()) >= adxConsumeBaseLimit) {

                            Double diff = DataUtil.division(lastFactorMap.get(key), baseFactor);
                            Double roi = baseRoi * diff;

                            retMap.put(key, DataUtil.formatDouble(roi,6));
                        }
                    }
                }
            }
        }

        return retMap;
    }



    /**
     * ROI置信处理（策略2-根据分level出价）
     *
     * @param roiMap
     * @param adxConsumeLevelMs(分*10000/cpm)
     * @param adxRoiControlDoInfo
     * @return 分level-ROI
     */
    public static Map<String, Double> getRoiPriceRectify(Map<String, Double> roiMap,
                                                         Map<String, Long> adxConsumeLevelMs,
                                                         AdxRoiControlDo adxRoiControlDoInfo) {

        Map<String, Double> retMap = new HashMap<>();

        Long adxConsumeLimit = 300 * 1000 * 10000L;
        Long adxConsumeBaseLimit = 25 * 1000 * 10000L;
        Double baseRoi = StrategyBid.nullToDefault(roiMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 0.0);

        for (AdxLevel adxLevel : AdxLevel.values()) {
            String key = adxLevel.getCode();
            if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {
                retMap.put(key, roiMap.get(key));
            }
        }

        AdxPriceExplorationDo lastPriceDoInfo = adxRoiControlDoInfo.getLastPriceExplorationDo();
        if (AssertUtil.isNotEmpty(lastPriceDoInfo)) {

            Map<String, Double> lastPriceMap = lastPriceDoInfo.getPriceExploreMap();
            Double basePrice = 10.0;// 默认出价（分/cpm）
            if (AssertUtil.isNotEmpty(lastPriceMap)) {
                basePrice = StrategyBid.nullToDefault(lastPriceMap.get(AdxLevel.ADX_LEVEL_TWO.getCode()), 10.0);
            }

            for (AdxLevel adxLevel : AdxLevel.values()) {
                String key = adxLevel.getCode();
                if (!key.equals(AdxLevel.ADX_LEVEL_ZER.getCode())) {

                    if (AssertUtil.isNotEmpty(adxConsumeLevelMs)) {

                        if (adxConsumeLevelMs.get(key) < adxConsumeLimit
                                && adxConsumeLevelMs.get(AdxLevel.ADX_LEVEL_TWO.getCode()) >= adxConsumeBaseLimit) {

                            Double diff = DataUtil.division(basePrice, lastPriceMap.get(key));
                            Double roi = baseRoi * diff;

                            retMap.put(key, DataUtil.formatDouble(roi,6));
                        }
                    }
                }
            }
        }

        return retMap;
    }



    /**
     * 基准level数据是否置信
     *
     * @param resoSucRate
     * @param sucRateLevel
     * @param adxConsumeLevel(分*10000/cpm)
     * @return
     */
    public static Long getBaseLevelConfident(Double resoSucRate,
                                             Map<String, Double> sucRateLevel,
                                             Map<String, Long> adxConsumeLevel) {

        Long ret = 0L;

        if (AssertUtil.isAllNotEmpty(sucRateLevel,resoSucRate,adxConsumeLevel)) {

            if (sucRateLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode()) > 0.05
                    && sucRateLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode()) > resoSucRate * 0.20
                    && adxConsumeLevel.get(AdxLevel.ADX_LEVEL_TWO.getCode()) > 200 * 1000 * 10000L){

                ret = 1L;
            }
        }

        return ret;
    }



    /**
     *针对墨迹获取相对ROI分界
     *
     * @param roi
     * @param targetRoi
     * @param sucRate
     * @param resoSucRate
     * @param adxConsume(分)
     * @return
     */
    public static Double getMojiRoiLimit(Double roi, Double targetRoi,
                                         Double sucRate, Double resoSucRate,
                                         Double adxConsume) {

        Double ret = 0.98;

        if (AssertUtil.isAllNotEmpty(roi,targetRoi, sucRate, resoSucRate, adxConsume)) {

            if (sucRate > 0.05 && sucRate > resoSucRate * 0.20
                    && adxConsume > 200.0){

                Double diff = DataUtil.division(roi, targetRoi);
                ret = diff > 1.05 ? 0.95 : (diff > 1.00 ? 0.97 : (diff > 0.95 ? 0.98 : 1.00));
            }
        }

        return ret;
    }




    /**
     *针对墨迹获取相对ROI分界
     *
     * @param roi
     * @param targetRoi
     * @param sucRate
     * @param resoSucRate
     * @param adxConsume(分)
     * @return
     */
    public static Double getRemainStableLimit(Double roi, Double targetRoi,
                                              Double sucRate, Double resoSucRate,
                                              Double adxConsume) {

        Double ret = 0.20;

        if (AssertUtil.isAllNotEmpty(roi, targetRoi, sucRate, resoSucRate, adxConsume)) {

            if (sucRate > 0.05 && sucRate > resoSucRate * 0.20
                    && adxConsume > 200.0){

                Double diff = DataUtil.division(roi, targetRoi);
                ret = diff > 1.10 ? 0.30 : (diff > 0.90 ? 0.20 : 0.30);
                //ret = diff > 1.10 ? 0.30 : (diff > 1.00 ? 0.20 : (diff > 0.95 ? 0.10 : (diff > 0.85 ? 0.15 : (diff > 0.75 ? 0.2 : 0.3))));
            }
        }

        return ret;
    }



    /**
     *针对墨迹获取维稳目标ROI
     *
     * @param roiDay
     * @param roiMs
     * @param targetRoi
     * @param sMin
     * @return
     */
    public static Double getRemainStableRoi(Double roiDay, Double roiMs,
                                            Double targetRoi, Double sMin) {

        Double ret = roiMs;

        if (AssertUtil.isAllNotEmpty(roiDay, roiMs, targetRoi, sMin)) {

            if (roiMs > targetRoi * 0.90 && roiDay > targetRoi * 0.98 && roiDay < targetRoi * 1.0) {

                ret = Math.max(roiMs, targetRoi * sMin);

            } else if (roiMs > targetRoi * 0.85 && roiDay > targetRoi * 1.00 && roiDay < targetRoi * 1.03) {

                ret = Math.max(roiMs, targetRoi * sMin);

            } else if (roiMs > targetRoi * 0.80 && roiDay > targetRoi * 1.03) {

                ret = Math.max(roiMs, targetRoi * sMin);
            }

        }

        return ret;
    }



    /**
     *冷启动高价组-试探流量占比
     *
     * @param roiMsDiff
     * @param adxConsumeTryMs(分*10000/cpm)
     * @return
     */
    public static Double getTryFlowRate(Double roiMsDiff, Long adxConsumeTryMs, Double limit) {

        Double ret = 0.20;

        if (AssertUtil.isAllNotEmpty(roiMsDiff, adxConsumeTryMs, limit)) {

            Double adxConsumeMs = DataUtil.division(adxConsumeTryMs, 1000L*10000L);

            if (roiMsDiff > 0.98) {

                ret += Math.min(DataUtil.division(roiMsDiff, 0.98) * 0.40, (limit - 0.2));

            } else {

                if (adxConsumeMs < 200) {

                    ret += (1.0 - DataUtil.division(adxConsumeMs, 200.0)) * 0.60;

                } else if (adxConsumeMs > 1000) {

                    ret = 0.1 - Math.min(DataUtil.division(adxConsumeMs, 1000.0) - 1.0, 1.0) * 0.1;

                } else {

                    ret = 0.2 - Math.min(DataUtil.division((adxConsumeMs - 200.0), (1000.0 - 200.0)), 1.0) * 0.1;

                }

            }

        }

        return StrategyBid.getNormalValue(ret,0.20, 0.02, limit);
    }



    /**
     *墨迹-冷启动高价组-试探流量占比
     *
     * @param roiDay
     * @param roiMs
     * @param targetRoi
     * @param adxConsumeTryMs(分*10000/cpm)
     * @return
     */
    public static Double getMojiTryFlowRate(Double roiDay, Double roiMs, Double targetRoi, Long adxConsumeTryMs) {

        Double ret = 0.20;

        if (AssertUtil.isAllNotEmpty(roiDay, roiMs, targetRoi, adxConsumeTryMs)) {

            Double adxConsumeMs = DataUtil.division(adxConsumeTryMs, 1000L*10000L);

            if (adxConsumeMs < 200) {

                ret = 0.20 + (1.0 - DataUtil.division(adxConsumeMs, 200.0)) * 0.60;

            } else {

                Double tmpDiff1 = DataUtil.division(roiDay, targetRoi, 3);
                Double tmpFlowRate1 = tmpDiff1 < 0.95 ? 0.2 : (tmpDiff1 < 0.98 ? 0.25 : (tmpDiff1 < 1.1 ? 0.30 : 0.35));
                Double tmpDiff2 = DataUtil.division(roiMs, targetRoi, 3);
                Double tmpFlowRate2 = tmpDiff2 < 0.95 ? 0.2 : (tmpDiff2 < 0.98 ? 0.25 : (tmpDiff2 < 1.1 ? 0.30 : 0.35));

                ret = DataUtil.formatDouble((0.3 * tmpFlowRate1 + 0.7 * tmpFlowRate2),3);
            }

        }

        return StrategyBid.getNormalValue(ret,0.20, 0.20,0.80);
    }



    /**
     *墨迹-正常启动高价组-流量占比
     *
     * @param roiDay
     * @param roiMs
     * @param targetRoi
     * @return
     */
    public static Double getMojiNormalFlowRate(Double roiDay, Double roiMs, Double targetRoi, Long adxConsumeTryMs) {

        Double ret = 0.10;

        if (AssertUtil.isAllNotEmpty(roiDay, roiMs, targetRoi, adxConsumeTryMs)) {

            Double adxConsumeMs = DataUtil.division(adxConsumeTryMs, 1000L*10000L);

            if (adxConsumeMs < 200) {

                ret = 0.10 + (1.0 - DataUtil.division(adxConsumeMs, 200.0)) * 0.20;

            } else {

                Double tmpDiff1 = DataUtil.division(roiDay, targetRoi, 3);
                Double tmpFlowRate1 = tmpDiff1 < 0.95 ? 0.1 : (tmpDiff1 < 0.98 ? 0.15 : (tmpDiff1 < 1.1 ? 0.20 : 0.25));
                Double tmpDiff2 = DataUtil.division(roiMs, targetRoi, 3);
                Double tmpFlowRate2 = tmpDiff2 < 0.95 ? 0.1 : (tmpDiff2 < 0.98 ? 0.15 : (tmpDiff2 < 1.1 ? 0.20 : 0.25));

                ret = DataUtil.formatDouble((0.3 * tmpFlowRate1 + 0.7 * tmpFlowRate2),3);

            }

        }

        return StrategyBid.getNormalValue(ret,0.10, 0.10,0.30);
    }



    /**
     * 消耗置信保障
     *
     * @param roi
     * @return 消耗置信保障(分)
     */
    public static Double getConfidentAdxConsume(Double roi) {

        Double ret = 100.0;

        if (AssertUtil.isNotEmpty(roi)){

            ret = 100.0 + DataUtil.division(roi-1, 1.15-1) * 100.0 ;
        }

        return getNormalValue(ret,100.0,100.0,200.0);

    }



    /**
     * 冷启动-试探涨价幅度
     *
     * @param adxConsume(分*10000/cpm)
     * @param roi
     * @param limit
     * @return
     */
    public static Double getTryUppStepCon(Long adxConsume, Double roi, Double limit) {

        Double ret = 0.0;

        if (AssertUtil.isAllNotEmpty(adxConsume, roi, limit)) {

            Double consume = DataUtil.division(adxConsume, 1000L*10000L);
            Double confidentConsume = getConfidentAdxConsume(roi);

            if (consume <= confidentConsume) {
                ret = (1.0 - DataUtil.division(consume, confidentConsume)) * limit;

            } else {
                ret = Math.max(1.0 - DataUtil.division(consume, confidentConsume), -1) * limit;

            }

        }

        return DataUtil.formatDouble(getNormalValue(ret,0.0, -limit, limit),3);

    }



    /**
     *策略3-冷启动高价组-试探流量占比
     *
     * @param roiMsDiff
     * @param adxConsumeTryMs(分*10000/cpm)
     * @param adxRoiControlDoInfo
     * @return
     */
    public static Double getTryFlowRate(Double roiMsDiff, Long adxConsumeTryMs,
                                        AdxRoiControlDo adxRoiControlDoInfo) {

        Double ret = 0.20;
        Double lastRate = 0.20;

        AdxFactorExplorationDo lastFactorDoInfo = adxRoiControlDoInfo.getLastFactorExplorationDo();
        if (AssertUtil.isNotEmpty(lastFactorDoInfo)){
            Map<String, Double> lastFlowRateMap = lastFactorDoInfo.getFactorFlowRateMap();
            if (AssertUtil.isNotEmpty(lastFlowRateMap)){
                lastRate = getNormalValue(lastFlowRateMap.get(AdxLevel.ADX_LEVEL_ONE.getCode()), lastRate, 0.02, 1.0);
            }
        }


        if (AssertUtil.isAllNotEmpty(roiMsDiff, adxConsumeTryMs)) {

            Double adxConsumeMs = nullToDefault(DataUtil.division(adxConsumeTryMs, 1000L*10000L),0.0);
            Double baseRate = getNormalValue((DataUtil.division(200.0, adxConsumeMs) * lastRate), lastRate, 0.02,0.8);

            if (roiMsDiff > 0.98) {
                ret = baseRate + Math.min(DataUtil.division(roiMsDiff, 0.98) - 1.0, 1.0) * (1.0 - baseRate);

            } else {

                if (adxConsumeMs < 200) {
                    ret = baseRate;

                } else if (adxConsumeMs > 1000) {
                    ret = baseRate * 1.5;

                } else {
                    ret = baseRate * 1.2;

                }

            }

        }

        return StrategyBid.getNormalValue(ret,0.20, 0.02, 1.0);
    }



    /**
     * 策略1--冷启动试探
     *
     * @param realConsumeMs    实际消耗(分)
     * @param confidentConsume 消耗置信保障(分)
     * @return 冷启动试探步伐
     */
    public static Double getFirTryStep(Double realConsumeMs, Double confidentConsume) {

        Double ret = 0.03;

        if (AssertUtil.isAllNotEmpty(realConsumeMs, confidentConsume)) {

            ret = 0.01 + Math.min(1.0 - realConsumeMs/confidentConsume, 1.0) * 0.02;

        }

        return StrategyBid.getNormalValue(ret,0.03, 0.01, 0.03);
    }



    /**
     * 衰减器
     *
     * @param lastDecay  上一次衰减结果
     * @param times      衰减期
     * @return
     */
    public static ArrayList<Double> getDecay(ArrayList<Double> lastDecay,
                                             Double statMsData,
                                             Integer times) {

        ArrayList<Double> ret = new ArrayList<>();

        if (AssertUtil.isNotEmpty(lastDecay)) {
            ret = lastDecay;
        }

        if (AssertUtil.isNotEmpty(statMsData)) {
            ret.add(statMsData);
        }

        if (AssertUtil.isNotEmpty(times) && ret.size() > times) {
            do { ret.remove(0);
            } while (ret.size() > times);
        }

        return ret;
    }



    /**
     * 多期衰减指标合成计算
     *
     * @param lastDecayData  衰减结果
     * @param adxConsumeMs   实时数据
     * @return
     */
    public static Double getSumDecayIndex(Double lastDecayData, Double adxConsumeMs) {


        adxConsumeMs = nullToDefault(adxConsumeMs, 0.0);
        Double ret = adxConsumeMs;

        Double decayRate = adxConsumeMs > 1000.0 ? 0.5 : (adxConsumeMs > 200.0 ? 0.7 : 0.9);

        if (AssertUtil.isAllNotEmpty(lastDecayData)) {
            ret += MathUtil.dot(lastDecayData, decayRate, 6);
        }

        return DataUtil.formatDouble(ret, 6);
    }



    //单元测试
    public static void main(String[] args) {

        try {

            ArrayList<Double> al = new ArrayList();
            al.add(12.0);
            al.add(13.0);

            System.out.println("getDecay:" + JSON.toJSONString(getDecay(al, null, 4)));
            System.out.println("getSumDecayIndex:" + JSON.toJSONString(getSumDecayIndex(100.0, 202.0)));

            Map<String, Double> roiLevel1 = new HashMap<>();
            roiLevel1.put("1", 1.99);
            roiLevel1.put("2", 1.05);
            roiLevel1.put("3", 1.10);

            Map<String, Double> priceMap = new HashMap<>();
            priceMap.put("1", 1320.0);
            priceMap.put("2", 1370.0);
            priceMap.put("3", 1400.0);

            Map<String, Double> sucRateLevel1 = new HashMap<>();
            sucRateLevel1.put("1", 0.26);
            sucRateLevel1.put("2", 0.50);
            sucRateLevel1.put("3", 0.50);

            Map<String, Long> adxConsumeLevelMs = new HashMap<>();
            adxConsumeLevelMs.put("1", 1320 * 1000 * 10000L);
            adxConsumeLevelMs.put("2", 1370 * 1000 * 10000L);
            adxConsumeLevelMs.put("3", 1400 * 1000 * 10000L);


            Double targetRoi1 = 1.0;
            System.out.println("selectBestLevel:" + JSON.toJSONString(selectBestLevel(targetRoi1, roiLevel1, sucRateLevel1)));
            System.out.println("selectUpLimitFee:" + JSON.toJSONString(selectUpLimitFee(priceMap, sucRateLevel1, 1500.0)));
            //System.out.println("getRoiRectify:" + JSON.toJSONString(getRoiRectify(roiMap, adxConsumeLevelMs, adxRoiControlDoInfo)));


            Map<String, Long> map1 = new HashMap<>();
            map1.put(AdxIndex.ADX_CONSUME.getCode(), 1200 * 1000 * 10000L);
            map1.put(AdxIndex.ADVERT_CONSUME.getCode(), 1400L);

            System.out.println("selectBestLevel:" + JSON.toJSONString(selectBestLevel(1.10, null, null)));

            Map<String, Double> flowRateMap1 = new HashMap<>();
            flowRateMap1.put("1", 0.20);
            flowRateMap1.put("2", 0.50);
            flowRateMap1.put("3", 0.30);

            Map<String, Double> priceMap1 = new HashMap<>();
            priceMap1.put("1", 1320.0);
            priceMap1.put("2", 1370.0);
            priceMap1.put("3", 1400.0);

            System.out.println("PRICE:" + JSON.toJSONString(DataUtil.double2Long(DataUtil.string2Double("1400.0"))));
            System.out.println("flowSplit:" + JSON.toJSONString(flowSplit(flowRateMap1, priceMap1, 1350L)));

            System.out.println("getSumLevelIndex:" + JSON.toJSONString(getSumLevelIndex(null, null, 1L)));

            System.out.println("getLevelIndex:" + JSON.toJSONString(getLevelIndex(null, null, 0L)));

            System.out.println("getLevelSucRate:" + JSON.toJSONString(getLevelSucRate(null)));

            System.out.println("getLevelRoi:" + JSON.toJSONString(getLevelRoi(null,1.20)));

            System.out.println("getRoi:" + JSON.toJSONString(getRoi(map1,1.20)));

            Double roiFactor = getNormalValue(0.6, 1.00, 0.7, 1.4);
            System.out.println("getNormalValue:" + JSON.toJSONString(roiFactor));

            Double roiLimit1 = -0.1;                     // ROI比例阈值
            double[] roiIntervel1 = {0.00, 0.90, 1.10};  // ROI分区间
            double[] roiWeight1 = {1.0, 1.5, 4.0};  // ROI打分权重
            System.out.println("getConfidenceWeight:" + JSON.toJSONString(getConfidenceWeight(roiLimit1, roiIntervel1, roiWeight1)));

            System.out.println("getTryFlowRate:" + JSON.toJSONString(getTryFlowRate(0.1,30*10000000L, 0.8)));
            System.out.println("getConfidentAdxConsume:" + JSON.toJSONString(getConfidentAdxConsume(1.1)));
            System.out.println("getTryUppStepCon:" + JSON.toJSONString(getTryUppStepCon(30*10000000L, 1.15, 0.08)));
            System.out.println("getFirTryStep:" + JSON.toJSONString(getFirTryStep(100.0,200.0)));
            System.out.println("AdxStatData.getPreInterval:" + JSON.toJSONString(AdxStatData.getPreInterval(null)));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}

